;################################################################################
;#										#
;# avr-chipbasic2 - single chip basic computer with ATmega644			#
;# chip8 interpreter	 							#
;# copyright (c) 2006-2010 Joerg Wolfram (joerg@jcwolfram.de)			#
;#										#
;#										#
;# This program is free software; you can redistribute it and/or		#
;# modify it under the terms of the GNU General Public License			#
;# as published by the Free Software Foundation; either version 3		#
;# of the License, or (at your option) any later version.			#
;#										#
;# This program is distributed in the hope that it will be useful,		#
;# but WITHOUT ANY WARRANTY; without even the implied warranty of		#
;# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the GNU		#
;# General Public License for more details.					#
;#										#
;# You should have received a copy of the GNU General Public			#
;# License along with this library; if not, write to the			#
;# Free Software Foundation, Inc., 59 Temple Place - Suite 330,			#
;# Boston, MA 02111-1307, USA.							#
;#										#
;################################################################################

.CSEG
.include	"/usr/local/include/avr/m644Pdef.inc"
.include	"../../system/includes/api_macros.inc"
.include	"../../system/includes/libdef.asm"
.include	"../../system/includes/definitions.asm"

.equ		ch8_base	= 0x0188		;bas_inbuf

.equ		ch8_regs	= ch8_base+0
.equ		ch8_rcode	= ch8_base+24		;error code, PC,I
.equ		ch8_stack	= ch8_base+40

.equ		ch8_rand	= libmio_ram + 24
.equ		ch8_kflags	= libmio_ram + 37
.equ		ch8_kcode	= libmio_ram + 14
.equ		ch8_frame	= libmio_ram + 8
.equ		ch8_env		= libmio_ram + 21

.def		ch8_ireg_lo	= r4
.def		ch8_ireg_hi	= r5
.def		ch8_vmode	= r6
.def		ch8_sp		= r7
.def		PC_L		= r24
.def		PC_H		= r25
.def		cmd_hi		= r18		;because of MSB first
.def		cmd_lo		= r19

.org	0x4e00
;                   "......      N"  
start:		.db "Chip-8 V0.4 N",0xec,0xff,0xff
;icon def
		.db 0x07,0x03,0x03,0x0b
		.db "SCH8"
		.db 0x0d,0x0c,0x0c,0x0e

;dummy data
		.db 0x00,0x00,0x00,0x00

.org	0x4e10
;------------------------------------------------------------------------------
;check if we are at program place 2
;------------------------------------------------------------------------------
chip8_entry:	sts	ch8_rcode,const_0	;clear error code
		api_getprog
		cpi	ZL,1
		breq	chip8_entry_1

chip8_noprg2:	api_alertthis_nb
		.db	0x2f,"Works only at P2",0
		ret				;exit program because not program 2

chip8_entry_1:	lds	ZL,prg_xmem
		cpi	ZL,0xff
		brne	chip8_entry_2

chip8_noxmem:	api_alertthis_nb
		.db	0x2f,"No XMEM driver",0
		ret				;exit program because of XRAM is not available

chip8_entry_2:	

;------------------------------------------------------------------------------
;main screen mask
;------------------------------------------------------------------------------
chip8_main:	rcall	chip8_rclear		;clear registers
chip8_main_0:	ldi	XL,0x17			;color scheme (white on black)
		api_setcolor
		ldi	YL,0			;maximal sized box
		ldi	YH,0
		ldi	ZL,29
		ldi	ZH,21
		api_whbox
		ldi	XL,0x70			;color scheme (black on white)
		api_setcolor
chip8_main_1:	api_thistext
		.db 1,2,"Chip8/Schip Interpreter",0

;show loaded program
chip8_show:	ldi	XL,0x16			;color scheme (yellow on blue)
		api_setcolor
		api_thistext
		.db	4,2,"Program: ",0
		clr	XL			;start of XRAM
		clr	XH
		ldi	r16,12
chip8_show_1:	rcall	drv_xmem_rbyte
		mov	tempreg1,r18
		api_outchar
		dec	r16
		brne	chip8_show_1

		rcall	chip8_vstat

;------------------------------------------------------------------------------
;the key loop
;------------------------------------------------------------------------------
chip8_kloop:	ldi	ZL,LOW(chip8_mbar*2)
		ldi	ZH,HIGH(chip8_mbar*2)
		api_menu

		cpi	tempreg1,0x01		;ESC
		brne	chip8_kloop_1
		api_askthis_nb
		.db	0x2f,"Exit Chip-8 ?",0,0
		cpi	tempreg1,0		;no
		brne	chip8_exit
		rjmp	chip8_main_0
chip8_exit:	ret				;exit program

chip8_kloop_1:

chip8_kloop_2:	cpi	tempreg1,0x02		;F1 = receive
		brne	chip8_kloop_3
		rcall	chip8_recv
chip8_kloop2:	rjmp	chip8_main


chip8_kloop_3:	cpi	tempreg1,0x03		;F2 = save
		brne	chip8_kloop_4
		rcall	chip8_save
		rjmp	chip8_main_0

chip8_kloop_4:	cpi	tempreg1,0x04		;F3 = load
		brne	chip8_kloop_5
		rcall	chip8_load
		rjmp	chip8_main

chip8_kloop_5:	cpi	tempreg1,0x07		;CTRL+F1 = DLY-
		brne	chip8_kloop_6
		ldi	XL,15
		ldi	XH,0
		rcall	drv_xmem_rbyte
		dec	r18
		andi	r18,0x1f
		ldi	XL,15
		ldi	XH,0
		rcall	drv_xmem_wbyte
		rjmp	chip8_main_0

chip8_kloop_6:	cpi	tempreg1,0x08		;CTRL+F2 = DLY+
		brne	chip8_kloop_7
		ldi	XL,15
		ldi	XH,0
		rcall	drv_xmem_rbyte
		inc	r18
		andi	r18,0x1f
		ldi	XL,15
		ldi	XH,0
		rcall	drv_xmem_wbyte
		rjmp	chip8_main_0

chip8_kloop_7:	cpi	tempreg1,0x05		;F4 = exec
		brne	chip8_kloop
		ldi	ZL,LOW(0x7207*2)
		ldi	ZH,HIGH(0x7207*2)
		lpm	XH,Z
		ldi	XL,0x05			;mode 5 (128 x 64)
		cpi	XH,0xa8
		brne	setmode
		ldi	XL,0x07			;we use mode 7
setmode:	api_setvmode			;set video mode
		rcall	chip8_cls		;clear screen
		ldi	XL,0xa0			;select tone
		api_note			;init sound
		rcall	ch8_entry		;call interpreter
		ldi	YL,LOW(ch8_rcode)
		ldi	YH,HIGH(ch8_rcode)
		st	Y+,tempreg1		;store return code
		st	Y+,ch8_sp
		st	Y+,PC_H
		st	Y+,PC_L
		st	Y+,cmd_hi		;code hi
		st	Y+,cmd_lo		;code lo
		st	Y+,ch8_ireg_hi		;store I register
		st	Y+,ch8_ireg_lo
		st	Y+,r14			;store DELAY counter
		st	Y+,r15			;store SOUND counter

		ldi	XL,0x02			;border to red
		api_setborder

		api_waitkey
		ldi	XL,0x00
		api_setborder
		ldi	XL,0x00
		api_setvmode			;set video mode
chip8_exx1:	rjmp	chip8_main_0		;restart after execution

;------------------------------------------------------------------------------
;view status
;------------------------------------------------------------------------------
chip8_vstat:	ldi	XL,0x15			;color scheme (cyan on blue)
		api_setcolor
		ldi	YL,LOW(ch8_rcode)
		ldi	YH,HIGH(ch8_rcode)

		api_thistext
		.db	6,2,"Stat: ",0
		ld	XL,Y
		lsl	XL
		lsl	XL
		ldi	ZL,LOW(chip8_etext*2)
		ldi	ZH,HIGH(chip8_etext*2)
		add	ZL,XL
		adc	ZH,const_0
		api_romtext

		ldi	ctrl,0x01		;16 bit

		api_thistext
		.db	9,2,"PC: ",0
		ldd	XH,Y+2
		ldd	XL,Y+3
		sbiw	XL,2			;correct
		api_outhex

		api_thistext
		.db	8,14,"I-Reg: ",0
		ldd	XH,Y+6
		ldd	XL,Y+7
		api_outhex

		api_thistext
		.db	9,14,"CMD  : ",0
		ldd	XH,Y+4
		ldd	XL,Y+5
		api_outhex


		ldi	ctrl,0x00		;8 bit

chip8_vstat_0:

		api_thistext
		.db	8,2,"SP: ",0
		ldd	XL,Y+1
		api_outhex

		api_thistext
		.db	20,2,"DLY: ",0
		ldd	XL,Y+8
		api_outhex

		api_thistext
		.db	20,14,"SND: ",0
		ldd	XL,Y+9
		api_outhex

		ldi	r17,0			;first reg

chip8_vstat_1:	ldi	XL,2
		sbrc	r17,3
		ldi	XL,14
		mov	XH,r17
		andi	XH,0x07
		subi	XH,0xf5			;+11
		api_gotoxy
		ldi	tempreg1,'V'
		api_outchar
		mov	tempreg1,r17		;regnum
		subi	tempreg1,0xd0		;+0x30
		cpi	tempreg1,0x3a
		brcs	chip8_vstat_2
		subi	tempreg1,0xf9
chip8_vstat_2:	api_outchar			;output reg name
		api_thistext
		.db	255," = ",0,0
		ldi	YL,LOW(ch8_regs)	;register set
		ldi	YH,HIGH(ch8_regs)
		add	YL,r17
		adc	YH,const_0
		ld	XL,Y
		api_outhex
		inc	r17
		cpi	r17,0x10
		brne	chip8_vstat_1

		ldi	XL,0x16
		api_setcolor

		api_thistext
		.db	6,13,"VM-",0
		ldi	ZL,LOW(0x7207*2)
		ldi	ZH,HIGH(0x7207*2)
		lpm	XH,Z
		ldi	tempreg1,'5'
		cpi	XH,0xa8
		brne	chip8_vstat_3
		ldi	tempreg1,'7'
chip8_vstat_3:	api_outchar

		ldi	XL,0x17
		api_setcolor

		api_thistext
		.db	6,19,"Delay: ",0
		ldi	XL,15
		ldi	XH,0
		rcall	drv_xmem_rbyte
		mov	XL,r18
		api_outhex

		ret


chip8_rclear:	ldi	YL,LOW(ch8_regs)	;register set
		ldi	YH,HIGH(ch8_regs)
		ldi	r17,14
chip8_rclear_1:	st	Y+,const_0
		dec	r17
		brne	chip8_rclear_1
		ret

;------------------------------------------------------------------------------
;save code
;------------------------------------------------------------------------------
chip8_save_err:	api_alertthis_nb
		.db	0x2f,"Cannot overwrite",0
		ret				;cannot overwrite other files

chip8_save:	api_fs_check
		api_fs_fsel_nb
		.db	"Save Chip8 Code",0
		cpi	tempreg1,0xed		;ESC
		breq	chip8_save_end
		cpi	tempreg1,0x16		;CH8
		breq	chip8_save_1		;overwrite
		cpi	tempreg1,0xf0		;empty
		brcs	chip8_save_err		;cannot create
		mov	tempreg1,tempreg4	;file number
		ldi	tempreg2,16		;blocks
		ldi	tempreg3,0x16		;CH8 file type
		api_fs_create

chip8_save_1:	ldi	tempreg2,0		;first sector
		clr	XL			;reset XMEM pointer
		clr	XH

;copy one page
chip8_save_2:	ldi	r16,0			;bytes to do
		api_getarray			;start of array
		subi	YH,0xfe			;+512 (we use buffer 2)
chip8_save_3:	rcall	drv_xmem_rbyte		;read one byte to r18
		st	Y+,r18
		dec	r16
		brne	chip8_save_3

;write page to flash
		ldi	tempreg3,2		;we use buffer 2
		mov	tempreg1,tempreg4	;copy file number
		api_fs_write			;write block

		inc	tempreg2		;next block
		cpi	tempreg2,16		;block counter
		brne	chip8_save_2		;block loop
chip8_save_end:	ret

;------------------------------------------------------------------------------
;write 256 bytes from buffer 1 to XRAM
;------------------------------------------------------------------------------
chip8_wxram:	ldi	r16,0x00		;words to do
		api_getarray			;start of array
		subi	YH,0xfe			;+512 (we use buffer 2)
chip8_wxram_1:	ld	r18,Y+			;get byte from buffer
		rcall	drv_xmem_wbyte		;write
		dec	r16
		brne	chip8_wxram_1
		ret

;------------------------------------------------------------------------------
;load code
;------------------------------------------------------------------------------
chip8_load_err:	api_alertthis_nb
		.db	0x2f,"No CH8 file",0,0
		ret				;cannot load this

chip8_load:	api_fs_check
		api_fs_fsel_nb			;call fileselector
		.db	"Load Chip8 Code",0
		cpi	tempreg1,0xed		;ESC
		breq	chip8_load_end
		cpi	tempreg1,0x16		;CH8
		brne	chip8_load_err		;cannot load this
		rcall	chip8_clear		;clear memory
		mov	tempreg1,tempreg4	;file number
chip8_load_1:	ldi	tempreg2,0		;first sector
chip8_load_2:	ldi	tempreg3,2		;we use buffer 1
		api_fs_read			;read block to array
		rcall	chip8_wxram		;get 256 bytes
		inc	tempreg2		;next block
		cpi	tempreg2,16		;block counter
		brne	chip8_load_2		;block loop
chip8_load_end:	ret

;------------------------------------------------------------------------------
;receive code
;------------------------------------------------------------------------------
chip8_recv:	api_askthis_nb
		.db	0x60,"Get via XMODEM?",0,0
		sbrs	tempreg1,0
		ret				;return to main menu

		rcall	chip8_clear
		api_saytransfer			;

chip8_recv_1:	ldi	tempreg1,0x15		;send NAK to start transfer
		api_putser

		ldi	tempreg4,16

chip8_recv_2:	api_getarray
		subi	YH,0xfe			;we use buffer 2
		rcall	chip8_rblock
		rcall	chip8_rblock
		rcall	chip8_wxram		;write 256 bytes to XRAM
		dec	tempreg4
		brne	chip8_recv_2
chip8_recv_3:	api_getser
		cpi	tempreg1,0x04
		brne	chip8_recv_3
		rjmp	chip8_rblock1

chip8_rblock:	api_getxm
chip8_rblock1:	ldi	tempreg1,0x06		;send ACK
		api_putser
		ret

;------------------------------------------------------------------------------
;clear memory
;------------------------------------------------------------------------------
chip8_clear:	clr	XL			;first address
		clr	XH
		ldi	r18,'-'			;data=0
chip8_clear_1:	rcall	drv_xmem_wbyte
		cpi	XL,0x0c
		brne	chip8_clear_1
		ldi	r18,0x00
chip8_clear_2:	rcall	drv_xmem_wbyte
		sbrs	XH,4
		rjmp	chip8_clear_2
		clr	XL
		clr	XH
		ret

chip8_cls:	api_getvram
		ldi	XL,0x00
		ldi	XH,0x04
chip8_cls_1:	st	Y+,const_0
		sbiw	XL,1
		brne	chip8_cls_1
		ret

.include	"interpreter.asm"

;------------------------------------------------------------------------------
;memory routines
;------------------------------------------------------------------------------
drv_xmem_rbyte:	push	ZH
		push	ZL
		ldi	ZL,xmem_bread
		lds	ZH,prg_xmem
		icall
		pop	ZL
		pop	ZH
		ret

drv_xmem_wbyte:	push	ZH
		push	ZL
		ldi	ZL,xmem_bwrite
		lds	ZH,prg_xmem
		icall
		pop	ZL
		pop	ZH
		ret

drv_xmem_rword:	push	ZH
		push	ZL
		ldi	ZL,xmem_wread
		lds	ZH,prg_xmem
		icall
		pop	ZL
		pop	ZH
		ret


chip8_ctables:

.include	"ctable_04_05.inc"
.include	"ctable_08_10.inc"

;------------------------------------------------------------------------------
;the program menubar
;------------------------------------------------------------------------------
		;   "111111222222333333444444555555"
chip8_mbar:	.db " Exit  Recv  Save  Load   RUN ",1,0
		.db "       DLY-  DLY+             "

chip8_etext:	.db "OK ",0		;normal end
		.db "BRK",0		;CTRL+C
		.db "SP<",0		;stack underflow
		.db "SP>",0		;stack overflow
		.db "???",0		;unknown opcode
