;################################################################################
;#										#
;# libmio - multi i/o for ATMega644						#
;# main library									#
;# copyright (c) 2005-2015 Joerg Wolfram (joerg@jcwolfram.de)			#
;#										#
;# This library is free software; you can redistribute it and/or		#
;# modify it under the terms of the GNU Lesser General Public			#
;# License as published by the Free Software Foundation; either			#
;# version 3 of the License, or (at your option) any later version.		#
;#										#
;# This library is distributed in the hope that it will be useful,		#
;# but WITHOUT ANY WARRANTY; without even the implied warranty of		#
;# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU		#
;# Lesser General Public License for more details.				#
;#										#
;# You should have received a copy of the GNU Lesser General Public		#
;# License along with this library; if not, write to the			#
;# Free Software Foundation, Inc., 59 Temple Place - Suite 330,			#
;# Boston, MA 02111-1307, USA.							#
;#										#
;################################################################################

;-------------------------------------------------------------------------------
;port konfiguration for port B ... D
;-------------------------------------------------------------------------------
libmio_init:	ldi	XL,0xba			;port direction (+SPI)
		out	DDRB,XL
		ldi	XL,0xff
		out	PORTB,XL

		ldi	XL,0xf0			;port direction
		out	DDRC,XL
		ldi	XL,0x00
		out	PORTC,XL

		ldi	XL,0x00			;port direction
		out	DDRA,XL
		ldi	XL,0xFF
		out	PORTA,XL


		ldi	XL,0xf0			;set pin 4-7 as output
		out	DDRD,XL
		ldi	XL,0xff
		out	PORTD,XL
		sbic	GPIOR0,2
		sbi	DDRD,3
		sbic	GPIOR0,2
		cbi	PORTD,3

		in	XL,PINC			;get portc (bit 2,3 = config)
		andi	XL,0x0c			;clear editor mode bit
		out	GPIOR2,XL

;-------------------------------------------------------------------------------
;timer configuration
;-------------------------------------------------------------------------------
		ldi	XH,0xc2			;FAST-PWM-Mode 14 (PWM channel B)
		sts	TCCR1A,XH
		ldi	XH,0x19			;FAST-PWM-Mode (no prescaler)
		sts	TCCR1B,XH
		ldi	ZH,HIGH(libmio_counter_p-1)
		sbrs	XL,2			;1 pal/ntsc bit
		ldi	ZH,HIGH(libmio_counter_n-1)
		sts	ICR1H,ZH
		ldi	ZH,LOW(libmio_counter_p-1)
		sbrs	XL,2			;1 pal/ntsc bit
		ldi	ZH,LOW(libmio_counter_n-1)
		sts	ICR1L,ZH

		sts	OCR1AH,const_0		;we use only 8 bits
		ldi	ZH,libmio_hslen		;1 hsync length pal
		sts	OCR1AL,ZH

		ldi	ZH,HIGH(libmio_bend)	;border end Interrupt
		sts	OCR1BH,ZH
		ldi	ZH,LOW(libmio_bend)	;1 hor position pal
		sts	OCR1BL,ZH

		ldi	XL,0x06
		sts	TIMSK1,XL		;enable compare OC1A/OC1B int

		ldi	XL,0x02			;ctc but disable output
		out	TCCR0A,XL
		ldi	XL,0x01
		out	TCCR0B,XL
		ldi	XL,49			;ca. 200KHz
		out	OCR0A,XL

		ldi	XL,0x83			;timer 2 config
		sts	TCCR2A,XL
		ldi	XL,0x01			;timer 2 config
		sts	TCCR2B,XL
		ldi	XL,0x80			;zero signal
		sts	OCR2A,XL
		sts	ASSR,const_0

;-------------------------------------------------------------------------------
;usart configuration (synchronous for keyboard)
;-------------------------------------------------------------------------------
		sts	UCSR0A,const_0
		ldi	XL,0x10
		sts	UCSR0B,XL
		ldi	XL,0x66
		sts	UCSR0C,XL

;-------------------------------------------------------------------------------
;usart 1 configuration (serial1, this makes nothinhg at m644)
;-------------------------------------------------------------------------------
;		rjmp	libmio_u1e
		ldi	XL,0x02			;set U2X
		sts	UCSR1A,XL
		sts	UCSR1B,const_0		;disable RX,TX
		call	mem_getconf
		sbic	GPIOR0,5
		rjmp	libmio_u1e
		ldi	XL,6			;38400/8/N/1
		rcall	libmio_baud2		;set baudrate
		ldi	XL,0x18
		sts	UCSR1B,XL		;enable RX,TX
		ldi	XL,0x06
		sts	UCSR1C,XL		;8 bits, 1 stop, no parity
		sbi	DDRD,3
libmio_u1e:

;-------------------------------------------------------------------------------
;spi configuration (mode 3 master 5MHz/156kHz)
;-------------------------------------------------------------------------------
		call	libmio_setcf1

;-------------------------------------------------------------------------------
;clear video-RAM and set pointers
;-------------------------------------------------------------------------------
		ldi	XL,0x15			;color = white
		rcall	libmio_setcol		;set color
		ldi	YL,LOW(libmio_ram)
		ldi	YH,HIGH(libmio_ram)
		ldi	XL,LOW(libmio_vram)	;2
		st	Y+,XL			;2 start vram
		ldi	XL,HIGH(libmio_vram)	;2
		st	Y+,XL			;2 start vram
		ldi	XL,0x1e
libmio_init_c:	st	Y+,const_0
		dec	XL
		brne	libmio_init_c
		ldi	XL,0xaa
		sts	libmio_rand1,XL
		sts	libmio_rand3,XL
		sts	libmio_channel,const_0	;set output to screen
		sts	libmio_font,const_0	;set output to screen
		ldi	XL,0x3e
		sts	libmio_env,XL		;sound off

;------------------------------------------------------------------------------
; set display to normal mode
;------------------------------------------------------------------------------
libmio_nmode:	push	r16
		sts	libmio_vidmode,const_0
		ldi	r16,235
		rjmp	libmio_emodes

;------------------------------------------------------------------------------
; set display to editor mode
;------------------------------------------------------------------------------
libmio_emode:	push	r16
		ldi	r16,0x08
		sts	libmio_vidmode,r16
		ldi	r16,233
libmio_emodes:	sts	libmio_vislines,r16
		pop	r16
		ret

;------------------------------------------------------------------------------
; set video mode
;------------------------------------------------------------------------------
libmio_svmode:	lds	r19,libmio_vidmode	;old vidmode
		andi	r19,7
		cpi	r19,7
		brne	libmio_svm1
		clt				;exit
		lds	ZH,prg_vdrv
		ldi	ZL,vdrv_init		;FKT
		sbrs	ZH,7
		icall				;call init routine with exit flag(0)
libmio_svm1:	mov	r19,XL
		cpi	XL,0x00
		breq	libmio_vmode0
		cpi	XL,0x04
		brne	libmio_vmode1
		call	libmio_cptab
libmio_vmode0:	ldi	YL,libmio_v0cols
		ldi	YH,libmio_v0rows
		ldi	XL,235
		rjmp	libmio_vmodes

libmio_vmode1:	cpi	XL,0x01
		brne	libmio_vmode2
		sts	libmio_palette,const_0	;palette 0
		ldi	XL,0xe0			;white
		sts	libmio_palette+1,XL	;palette 1
		ldi	YL,libmio_v1cols
		ldi	YH,libmio_v1rows
		ldi	XL,237
		rjmp	libmio_vmodes

libmio_vmode2:	cpi	XL,0x02
		brne	libmio_vmode3
		ldi	YL,0x00
		ldi	YH,0x01
		st	Y,const_0		;palette 0
		ldi	XL,0x40			;red
		std	Y+1,XL			;palette 1
		ldi	XL,0xa0			;cyan
		std	Y+2,XL			;palette 2
		ldi	XL,0xe0			;white
		std	Y+3,XL			;palette 3
		ldi	YL,libmio_v2cols
		ldi	YH,libmio_v2rows
		ldi	XL,235
		rjmp	libmio_vmodes

libmio_vmode3:	cpi	XL,0x03
		brne	libmio_vmode5
		clr	XL
		clr	XH
libmio_vmode3a:	rcall	libmio_setpal
		inc	XL
		inc	XH
		cpi	XH,0x10
		brne	libmio_vmode3a
libmio_vmode13:	ldi	XL,237
		ldi	YL,libmio_v3cols
		ldi	YH,libmio_v3rows
		rjmp	libmio_vmodes

libmio_vmode5:	cpi	XL,0x05
		brne	libmio_vmode6
		ldi	YL,libmio_v5cols
		ldi	YH,libmio_v5rows
		ldi	XL,0x20			;blue
		sts	libmio_palette,XL	;palette 0
		ldi	XL,0xe0			;white
		sts	libmio_palette+1,XL	;palette 1
		ldi	XL,184

libmio_vmodes:	sts	libmio_clipx2,YL
		sts	libmio_clipy2,YH
		sts	libmio_vislines,XL
		sts	libmio_vidmode,r19	;set videomode
		ret

libmio_vmode6:	cpi	XL,0x06
		brne	libmio_vmode7
		call	libmio_copytab
		ldi	YL,libmio_v0cols
		ldi	YH,libmio_v0rows
		ldi	XL,235
		rjmp	libmio_vmodes

libmio_vmode7:	movw	r0,ZL			;save Z register
		lds	ZH,prg_vdrv
		sbrc	ZH,7
		rjmp	libmio_vmode7e
		set
		ldi	ZL,vdrv_init		;FKT
		icall
		rjmp	libmio_vmodes

libmio_vmode7e:	ldi	ereg,5			;constant too big
		ret

;------------------------------------------------------------------------------
; set color to XL
;------------------------------------------------------------------------------
libmio_setcol:	push	XH
		push	XL
		mov	XH,XL
		andi	XH,0x88
		lsr	XH
		lsr	XH
		lsr	XH
		lsl	XL
		andi	XL,0xee
libmio_setcol1:	or	XL,XH
		sts	libmio_color,XL
		pop	XL
		pop	XH
		ret

;------------------------------------------------------------------------------
; set channel to XL
;------------------------------------------------------------------------------
libmio_setchan:	sts	libmio_channel,XL
		ret

;------------------------------------------------------------------------------
; set FONT to XL
;------------------------------------------------------------------------------
libmio_setfont:	sts	libmio_font,XL
		ret

;------------------------------------------------------------------------------
; set palette index XH to XL
;------------------------------------------------------------------------------
libmio_setpal:	push	YH
		push	YL
		push	XL
		andi	XL,0x0f			;mask bits
		swap	XL
		lsl	XL
		brcc	libmio_setpl1
		ori	XL,0x10
libmio_setpl1:	ldi	YH,HIGH(libmio_palette)
		mov	YL,XH
		andi	YL,0x0f
		st	Y,XL
		pop	XL
		pop	YL
		pop	YH
		ret

;------------------------------------------------------------------------------
; set foreground color to XL
;------------------------------------------------------------------------------
libmio_setfgc:	push	XH
		push	XL
		andi	XL,0x0f			;limit
		swap	XL			;set to fg
		lsl	XL
		brcc	libmio_setfg1
		ori	XL,0x10
libmio_setfg1:	swap	XL			;swap back
		lds	XH,libmio_color
		andi	XH,0xf0			;remove FG color
		rjmp	libmio_setcol1

;------------------------------------------------------------------------------
; set border to XL
;------------------------------------------------------------------------------
libmio_setbrd:	push	XL
		swap	XL
		bst	XL,7
		lsl	XL
		bld	XL,4
		andi	XL,0xf0
		sts	libmio_border,XL
		pop	XL
		ret
;------------------------------------------------------------------------------
; write char tempreg1 to Y
;------------------------------------------------------------------------------
libmio_wchar:	st	Y,tempreg1
		lds	tempreg1,libmio_vidmode
		sbrc	tempreg1,3
		ret				;editor mode has no attr
		andi	tempreg1,0x07
		cpi	tempreg1,6		;mode 6 has no attr
		breq	libmio_wchar5
		push	YL
		push	YH
		push	tempreg2
		ldi	tempreg2,60
		add	YL,tempreg2
		adc	YH,const_0
		ldi	tempreg1,LOW(libmio_cols*libmio_rows-60)
		add	YL,tempreg1
		ldi	tempreg1,HIGH(libmio_cols*libmio_rows-60)
		adc	YH,tempreg1
libmio_wchar1:	lds	tempreg1,libmio_color
		st	Y,tempreg1
		pop	tempreg2
		pop	YH
		pop	YL
libmio_wchar5:	ret

;-------------------------------------------------------------------------------
; Bin16 -> ASCII (dez)
; XL,XH 	= number
; ctrl		= format
; 	Bit 0,1	 = format	00 = compact
;				01 = leading spaces
;				10 = leading zeroes 0...65536
;				11 = leading zeroes with +- sign  
;			
;	Bit 2,3	 = number of visible digits (2,3,4,5)
;	Bit 4,5	 = comma position (no  xxxx,x   xxx,xx  xx,xxx) 
;	Bit 6	1= big size (only in monochrome mode) 
;-------------------------------------------------------------------------------
.ifdef use_libmio_outd
libmio_outd:	push	tempreg1		;save registers
		push	tempreg2
		push	tempreg3
		push	XL
		push	XH
		ldi	tempreg1,0x20		;space
		mov	tempreg2,ctrl
		andi	tempreg2,0x03
		cpi	tempreg2,0x02		;leading zeroes without sign
		breq	libmio_outd_03
		cpi	tempreg2,0x03		;leading zeroes with +- sign
		brne	libmio_outd_01
		ldi	tempreg1,'+'		;set positive sign
libmio_outd_01:	sbrs	XH,7
		rjmp	libmio_outd_02
		com	XL			;X=-X
		com	XH			;
		add	XL,const_1		;
		adc	XH,const_0		;
		ldi	tempreg1,'-'		;sign is negative
libmio_outd_02:	call	libmio_char		;out sign


libmio_outd_03:	mov	tempreg2,ctrl		;format byte
		andi	tempreg2,0x0c		;mask number of digits
		mov	tempreg3,ctrl		;format byte
		andi	tempreg3,0x30		;mask comma position
		clt				;clear nonzero-flag
		ldi	tempreg1,47		;"0"-1 	
libmio_outd_04:	inc	tempreg1			;
		subi	XL,16			;-10000
		sbci	XH,39
		brcc	libmio_outd_04		;no overflow
		subi	XL,240			;+10000
		sbci	XH,216			;
		cpi	tempreg2,0x0c		;5 digits?
		brne	libmio_outd_05		;no		
		rcall	libmio_nc		;out digit 10^4

libmio_outd_05:	ldi	tempreg1,47		;"0"-1 	
libmio_outd_06:	inc	tempreg1			;
		subi	XL,232			;-1000
		sbci	XH,3			;
		brcc	libmio_outd_06		;no overflow
		subi	XL,24			;+1000
		sbci	XH,252			;
		sbrc	tempreg2,3		;skip if < 4 digits
		rcall	libmio_nc		;out digit 10^3
		cpi	tempreg3,0x30		;comma?
		brne	libmio_outd_07		;no
		rcall	libmio_comma		;

libmio_outd_07:	ldi	tempreg1,47		;"0"-1 	
libmio_outd_08:	inc	tempreg1		;+1
		subi	XL,100			;-100
		sbc	XH,const_0		;
		brcc	libmio_outd_08		;no overflow
		subi	XL,156			;+100
		sbci	XH,255			;
		cpi	tempreg2,0x00		;3 digits
		breq	libmio_outd_09
		rcall	libmio_nc		;out digit 10^2
libmio_outd_09:	cpi	tempreg3,0x20		;comma?
		brne	libmio_outd_10		;no
		rcall	libmio_comma		;

libmio_outd_10:	ldi	tempreg1,47		;"0"-1 	
libmio_outd_11:	inc	tempreg1		;+1
		subi	XL,10			;-10
		brcc	libmio_outd_11		;no overflow
		rcall	libmio_nc		;out digit 10^1
		subi	XL,246			;+10
		cpi	tempreg3,0x10		;comma?
		brne	libmio_outd_12		;no
		rcall	libmio_comma		;
		
libmio_outd_12:	ldi	tempreg1,48		;"0"
		add	tempreg1,XL		;LSB digit
		rcall	libmio_char
		pop	XH
		pop	XL
		pop	tempreg3
		pop	tempreg2
		pop	tempreg1
		ret

;-------------------------------------------------------------------------------
; out numeric char (tempreg1)
;-------------------------------------------------------------------------------
libmio_nc:	cpi	tempreg1,48		;is zero
		breq	libmio_nc_1		;yes
		set				;set flag
		rjmp	libmio_char		;send char
		
libmio_nc_1:	sbrc	ctrl,1			;skip if no leading zeroes
libmio_nc_2:	rjmp	libmio_char		;send zero
		brts	libmio_nc_2		;flag is set
		
		sbrs	ctrl,0			;skip if no leading spaces
		ret 
		rjmp	libmio_space		;send space
		ret
.endif

;-------------------------------------------------------------------------------
; Bin -> ASCII (hex) 
; (XL,XH) = number
; ctrl 		= format
; 	Bit 0	1= 16Bit (4 chars)
;	Bit 6	1= Big size
;-------------------------------------------------------------------------------
.ifdef use_libmio_outh
libmio_outh:	push	tempreg1		;save temp register
		sbrs	ctrl,0
		rjmp	libmio_outh1		;only 8
		mov	tempreg1,XH		;
		swap	tempreg1		;high-nibble
		rcall	libmio_oh
		mov	tempreg1,XH		;
		rcall	libmio_oh
libmio_outh1:	mov	tempreg1,XL		;
		swap	tempreg1		;high-nibble
		rcall	libmio_oh
		mov	tempreg1,XL		;
		rcall	libmio_oh
		pop	tempreg1		;restore temp register
		ret

;-------------------------------------------------------------------------------
; out hexadecimal char (tempreg1)
;-------------------------------------------------------------------------------
libmio_oh:	andi	tempreg1,0x0f		;mask nibble
		subi	tempreg1,0xd0		;+30
		cpi	tempreg1,0x3a		;>9
		brcs	libmio_oh_1		;jump, if not
		subi	tempreg1,0xf9		;+6
libmio_oh_1:	rjmp	libmio_char
.endif

;-------------------------------------------------------------------------------
; clear video-RAM
;-------------------------------------------------------------------------------
libmio_clr:	push_ptr			;save registers
		lds	tempreg1,libmio_vidmode	;video mode
		cpi	tempreg1,0x00		;standard?
		breq	libmio_fill		;yes
		cpi	tempreg1,0x04		;user 1?
		breq	libmio_fill		;yes
		ldi	YH,HIGH(libmio_vram)	;home position
		ldi	YL,LOW(libmio_vram)	;home position
		ldi	XL,libmio_rows
		cpi	tempreg1,0x06		;mode 6 is special
		breq	libmio_cls_g01
		cpi	tempreg1,0x07		;mode 7 is special
		breq	libmio_cls_07
		ldi	XL,libmio_rows*4
libmio_cls_g01:	ldi	XH,libmio_cols
libmio_cls_g02:	st	Y+,const_0		;write to RAM
		dec	XH			;decrement counter
		brne	libmio_cls_g02		;loop
		dec	XL			;decrement counter
		brne	libmio_cls_g01		;loop
		cpi	tempreg1,1		;graphics 1?
		brne	libmio_cls_g06
		lds	XL,libmio_color
		ldi	YH,HIGH(libmio_vram+2436)	;home position
		ldi	YL,LOW(libmio_vram+2436)	;home position
		ldi	XH,160
libmio_cls_g03:	st	Y+,XL
		st	Y+,XL
		dec	XH
		brne	libmio_cls_g03
libmio_cls_g04:	clr	libmio_cur_x		;set X-position
		clr	libmio_cur_y		;set Y-position
		sts	libmio_lpos,const_0
libmio_cls_g06:	pop_ptr

libmio_fill:	ldi	tempreg1,0x20		;space
		ldi	YH,HIGH(libmio_vram)	;home position
		ldi	YL,LOW(libmio_vram)	;home position
		ldi	XL,libmio_rows
libmio_cls_01:	ldi	XH,libmio_cols
libmio_cls_02:	st	Y+,tempreg1		;write to RAM
		dec	XH			;decrement counter
		brne	libmio_cls_02		;loop
		dec	XL			;decrement counter
		brne	libmio_cls_01		;loop
		lds	tempreg1,libmio_color
		ldi	XL,libmio_rows
libmio_cls_05:	ldi	XH,libmio_cols
libmio_cls_06:	st	Y+,tempreg1		;write to RAM
		dec	XH			;decrement counter
		brne	libmio_cls_06		;loop
		dec	XL			;decrement counter
		brne	libmio_cls_05		;loop
		rjmp	libmio_cls_g04

libmio_cls_07:	lds	ZH,prg_vdrv
		ldi	ZL,vdrv_cls
		icall				;routine 2 (CLS)
		rjmp	libmio_cls_g04

libmio_cls_04:	sts	libmio_lpos,const_0
		pop	XH
		pop	XL
		ret

;-------------------------------------------------------------------------------
; newline and calculate new position
;-------------------------------------------------------------------------------
.ifdef use_libmio_newline
libmio_nl1:	cpi	XL,7
		brne	libmio_cls_04		;NL works not here
		pop	XH			;restore registers
		pop	XL
		push	ZH
		push	ZL
		lds	ZH,prg_vdrv
		ldi	ZL,vdrv_newline
		icall
		pop	ZL
		pop	ZH
		ret

libmio_newline:	push	XL
		push	XH
		sts	libmio_lpos,const_0
		lds	XL,libmio_channel
		cpi	XL,0			;screen
		brne	libmio_newl1
		lds	XL,libmio_vidmode
		cpi	XL,0			;vm0
		brne	libmio_nl1
libmio_nlap:	ldi	XL,libmio_rows
		clr	libmio_cur_x		;begin of line
		inc	libmio_cur_y		;next line
		cp	XL,libmio_cur_y		;beyond screen
		brne	libmio_cls_04
		in	XL,GPIOR2
		sbrc	XL,7			;skip if scroll
libmio_nowr:	rjmp	libmio_wrap
		clr	XL
		ldi	XH,(libmio_rows-1)	;last line
		mov	libmio_cur_y,XH
		push	YL
		push	YH
		rjmp	libmio_char_04		;scroll screen

libmio_wrap:	mov	libmio_cur_x,const_0
		mov	libmio_cur_y,const_0
		rjmp	libmio_cls_04

libmio_newl1:	cpi	XL,1			;serial
		brne	libmio_newl2
		pop	XH
		pop	XL
		jmp	libmio_sysnl		;write serial

libmio_newl2:	cpi	XL,2			;parallel
		brne	libmio_newl3
		ldi	tempreg1,0x0d		;CR
		sbic	GPIOR0,6		;skip if no CR
		rcall	libmio_ppar
		ldi	tempreg1,0x0a		;newline
		sbic	GPIOR0,7		;skip if no LF
		rcall	libmio_ppar		;write parallel
		pop	XH
		pop	XL
		ret
		
libmio_newl3:	cpi	XL,3			;array
		brne	libmio_newl4
		pop	XH
		pop	XL
		ret

libmio_newl4:	cpi	XL,4			;serial2
		brne	libmio_newl5
		pop	XH
		pop	XL
		jmp	libmio_sernl2		;write serial

libmio_newl5:	pop	XH			;i2c
		pop	XL
		ldi	tempreg2,0x0a
		libi2c_wbyte
		ret
.endif

;-------------------------------------------------------------------------------
; output character and calculate new position (tempreg1=char)
;-------------------------------------------------------------------------------
libmio_comma:	ldi	tempreg1,0x2c		;set char to comma
		rjmp	libmio_char		;write
libmio_space:	ldi	tempreg1,32		;set char to space
libmio_char:	push	XL
		push	XH
		lds	XL,libmio_lpos
		inc	XL
		sts	libmio_lpos,XL
		lds	XL,libmio_channel
		cpi	XL,5
		brcs	libmio_chr_1
		pop	XH
		pop	XL
		mov	tempreg2,tempreg1	;copy char
		libi2c_wbyte			;output to i2c
		ret

libmio_chr_1:	cpi	XL,0			;screen
		breq	libmio_char2
		cpi	XL,3			;array
		brne	libmio_char0
		mov	XL,tempreg1
		clr	XH
		push	tempreg2
		push	tempreg3
		call	arr_write		;write to array
		pop	tempreg3
		pop	tempreg2
libmio_chr_x:	pop	XH
		pop	XL
		ret

libmio_char0:	pop	XH
		lds	XL,libmio_channel
		cpi	XL,1			;serial
		brne	libmio_char0a
		pop	XL
		jmp	libmio_sysput		;write serial (ch1)
libmio_char0a:	cpi	XL,2			;serial
		pop	XL
		brne	libmio_char0b
		rjmp	libmio_ppar		;write parallel (ch2)
libmio_char0b:	jmp	libmio_pser2		;serial 2 (ch4)


libmio_char2:	push	ZL			;save
		lds	ZL,libmio_vidmode	;get video mode
		andi	ZL,0x07
		cpi	ZL,0x00
		breq	libmio_char_0
		cpi	ZL,0x04
		breq	libmio_char_0
		cpi	ZL,0x06
		breq	libmio_char_0
		cpi	ZL,0x07			;custom mode
		brne	libmio_charnc
		push	ZH
		lds	ZH,prg_vdrv
		ldi	ZL,vdrv_char
		icall
		pop	ZH
		pop	ZL
		rjmp	libmio_chr_x

libmio_charnc:	rjmp	libmio_gchar		;graphics mode

libmio_char_0:	pop	ZL
libmio_char_ap:	push	YL
		push	YH
		rcall	libmio_calc		;calculate address
		sbrc	ctrl,6			;big chars
		rjmp	libmio_bchar

		rcall	libmio_wchar		;write char
		inc	XL			;next position
		cpi	XL,libmio_cols		;beyond the end of line
		brne	libmio_char_01
		clr	XL			;begin of line
		inc	XH			;of next line
		cpi	XH,libmio_rows		;beyond screen
		brne	libmio_char_01
		ldi	XH,libmio_rows-1	;last line
		rjmp	libmio_char_03		;we must scroll
libmio_char_01:	movw	libmio_cur_x,XL		;X
libmio_char_02:	pop	YH
		pop	YL
		pop	XH
		pop	XL
		ret

; scroll screen
libmio_char_03:	movw	libmio_cur_x,XL		;save X-position
libmio_char_04:	ldi	YH,HIGH(libmio_vram)	;home position
		ldi	YL,LOW(libmio_vram)	;home position
		push	tempreg1
		ldi	XL,libmio_rows-1	;lines -1
libmio_char_05:	ldi	XH,libmio_cols		
libmio_char_06:	ldd	tempreg1,Y+libmio_cols
		st	Y+,tempreg1
		dec	XH
		brne	libmio_char_06
		dec	XL
		brne	libmio_char_05
		ldi	XL,libmio_cols
		ldi	tempreg1,32		;space
libmio_char_07:	st	Y+,tempreg1
		dec	XL
		brne	libmio_char_07

libmio_char_14:	ldi	YH,HIGH(libmio_vram+libmio_cols*libmio_rows)	;attr start
		ldi	YL,LOW(libmio_vram+libmio_cols*libmio_rows)	;attr start

		ldi	XL,libmio_rows-1	;lines -1
libmio_char_15:	ldi	XH,libmio_cols
libmio_char_16:	ldd	tempreg1,Y+libmio_cols
		st	Y+,tempreg1
		dec	XH
		brne	libmio_char_16
		dec	XL
		brne	libmio_char_15
		ldi	XL,libmio_cols
		lds	tempreg1,libmio_color	;space
libmio_char_17:	st	Y+,tempreg1
		dec	XL
		brne	libmio_char_17
		pop	tempreg1
		rjmp	libmio_char_02

libmio_bchar:	cpi	XL,libmio_cols-3
		brcc	libmio_char_02
		cpi	XH,libmio_rows-5
		brcc	libmio_char_02
		subi	XL,0xfd			;X+3
		mov	libmio_cur_x,XL		;store new X
		push	ZL
		push	ZH
		push	tempreg5		;byte 1
		push	tempreg6		;byte 2
		push 	tempreg7		;byte 3
		push	tempreg8		;for stroring T-bit
		bld	tempreg8,0		;store T
		push	tempreg2		;temp value

libmio_bchar_0:	mov	ZL,tempreg1		;char
		ldi	ZH,HIGH(libmio_ctable*2)	;start char ROM
		ldi	XH,5			;10 char lines

libmio_bchar_1:	clr	tempreg5		;clear columns
		clr	tempreg6
		clr	tempreg7
		clr	tempreg2		;clear temp value

		lpm	XL,Z			;get pixel line
		sbrc	XL,7			;pixel 0
		inc	tempreg2
		bst	tempreg2,0		;copy pix
		bld	tempreg5,0
		sbrc	XL,6
		inc	tempreg2
		bst	tempreg2,0		;copy pix
		bld	tempreg5,1
		sbrc	XL,5
		inc	tempreg2
		bst	tempreg2,0		;copy pix
		bld	tempreg6,0
		sbrc	XL,4
		inc	tempreg2
		bst	tempreg2,0		;copy pix
		bld	tempreg6,1
		sbrc	XL,3
		inc	tempreg2
		bst	tempreg2,0		;copy pix
		bld	tempreg7,0
		sbrc	XL,2
		inc	tempreg2
		bst	tempreg2,0		;copy pix
		bld	tempreg7,1

		inc	ZH			;next line
		clr	tempreg2		;clear temp value

		lpm	XL,Z			;get pixel line
		sbrc	XL,7			;pixel 0
		inc	tempreg2
		bst	tempreg2,0		;copy pix
		bld	tempreg5,2
		sbrc	XL,6
		inc	tempreg2
		bst	tempreg2,0		;copy pix
		bld	tempreg5,3
		sbrc	XL,5
		inc	tempreg2
		bst	tempreg2,0		;copy pix
		bld	tempreg6,2
		sbrc	XL,4
		inc	tempreg2
		bst	tempreg2,0		;copy pix
		bld	tempreg6,3
		sbrc	XL,3
		inc	tempreg2
		bst	tempreg2,0		;copy pix
		bld	tempreg7,2
		sbrc	XL,2
		inc	tempreg2
		bst	tempreg2,0		;copy pix
		bld	tempreg7,3

libmio_bchar_d:	inc	ZH
		mov	tempreg1,tempreg5	;char1
		rcall	libmio_wchar
		adiw	YL,0x01
		mov	tempreg1,tempreg6	;char2
		rcall	libmio_wchar
		adiw	YL,0x01
		mov	tempreg1,tempreg7	;char3
		rcall	libmio_wchar
		adiw	YL,libmio_cols-2	;next line
		dec	XH
		breq	libmio_bchar_e
		rjmp	libmio_bchar_1

libmio_bchar_e:	pop	tempreg2
		bst	tempreg8,0		;restore T
		pop	tempreg8
		pop	tempreg7
		pop	tempreg6
		pop	tempreg5
		pop	ZH
		pop	ZL
		rjmp	libmio_char_02

;graphics mode
libmio_gchar:	push	ZH			;restore Z reg
		push	YL
		push	YH
		push	tempreg3
		push	tempreg4
		mov	ZL,tempreg1		;char
		lds	tempreg3,libmio_font	;get font
		sbrc	tempreg3,0		;skip if font 0
		rjmp	libmio_gchar5		;jump to fnt1 routine

		ldi	ZH,HIGH(libmio_ctable*2)	;start char ROM
		movw	XL,libmio_cur_x		;get X-position
		ldi	YH,10			;number of lines

libmio_gchar1:	lpm	YL,Z			;get pixline
		rcall	libmio_plpl
		inc	XH
		mov	XL,libmio_cur_x		;get X-position
		sbrc	ctrl,6			;bigchars
		rcall	libmio_plpl
		sbrc	ctrl,6			;bigchars
		inc	XH
		inc	ZH			;next pixline
		mov	XL,libmio_cur_x		;get X-position
		dec	YH
		brne	libmio_gchar1
		mov	XL,libmio_cur_x		;get X-position
		subi	XL,0xfa			;+6
		sbrc	ctrl,6
		subi	XL,0xfa			;+6
		mov	libmio_cur_x,XL	
libmio_gchar4:	pop	tempreg4
		pop	tempreg3
		pop	YH
		pop	YL
		pop	ZH
		pop	ZL
		pop	XH
		pop	XL
		ret

libmio_gchar5:	ldi	ZH,HIGH(libmio_ctable1*2)	;start char ROM
		andi	ZL,0x7f
		mov	XL,ZL
		ldi	ZL,LOW(libmio_ctable1*2)	;start char ROM
		add	ZL,XL
		adc	ZH,const_0
		lsl	XL
		add	ZL,XL
		adc	ZH,const_0

		movw	XL,libmio_cur_x		;get X-position
		ldi	YH,6			;number of lines
		
libmio_gchar6:	lpm	YL,Z			;get pixline
		rcall	libmio_plpl5		
		inc	XH
		mov	XL,libmio_cur_x		;get X-position
		sbrc	ctrl,6			;bigchars
		rcall	libmio_plpl5
		sbrc	ctrl,6			;bigchars
		inc	XH
		sbrc	YH,0			;skip if first nibble is used
		adiw	ZL,1
		mov	XL,libmio_cur_x		;get X-position
		dec	YH
		brne	libmio_gchar6		
		mov	XL,libmio_cur_x		;get X-position
		subi	XL,0xfc			;+4
		sbrc	ctrl,6
		subi	XL,0xfc			;+4
		mov	libmio_cur_x,XL	
		rjmp	libmio_gchar4

;-------------------------------------------------------------------------------
; plot pixline in graphics mode with font 0, YL=pixline
;-------------------------------------------------------------------------------
libmio_plpl:	lds	tempreg3,libmio_color
		push	tempreg3
		swap	tempreg3
		sbrc	YL,7			;pix 1
		swap	tempreg3
		sts	libmio_color,tempreg3
		rcall	libmio_plot		;output pix
		inc	XL			;next pos
		sbrc	ctrl,6			;no bigchar
		rcall	libmio_plot		;output pix again
		sbrc	ctrl,6			;no bigchar
		inc	XL			;next pos

		sbrc	YL,6			;pix 2
		swap	tempreg3
		sts	libmio_color,tempreg3
		rcall	libmio_plot		;output pix
		inc	XL			;next pos
		sbrc	ctrl,6			;no bigchar
		rcall	libmio_plot		;output pix again
		sbrc	ctrl,6			;no bigchar
		inc	XL			;next pos

		sbrc	YL,5			;pix 3
		swap	tempreg3
		sts	libmio_color,tempreg3
		rcall	libmio_plot		;output pix
		inc	XL			;next pos
		sbrc	ctrl,6			;no bigchar
		rcall	libmio_plot		;output pix again
		sbrc	ctrl,6			;no bigchar
		inc	XL			;next pos

		sbrc	YL,4			;pix 4
		swap	tempreg3
		sts	libmio_color,tempreg3
		rcall	libmio_plot		;output pix
		inc	XL			;next pos
		sbrc	ctrl,6			;no bigchar
		rcall	libmio_plot		;output pix again
		sbrc	ctrl,6			;no bigchar
		inc	XL			;next pos

		sbrc	YL,3			;pix 5
		swap	tempreg3
		sts	libmio_color,tempreg3
		rcall	libmio_plot		;output pix
		inc	XL			;next pos
		sbrc	ctrl,6			;no bigchar
		rcall	libmio_plot		;output pix again
		sbrc	ctrl,6			;no bigchar
		inc	XL			;next pos

		sbrc	YL,2			;pix 6
		swap	tempreg3
		sts	libmio_color,tempreg3
		rcall	libmio_plot		;output pix
		inc	XL			;next pos
		sbrc	ctrl,6			;no bigchar
		rcall	libmio_plot		;output pix again
		sbrc	ctrl,6			;no bigchar
		inc	XL			;next pos

libmio_plpl1:	pop	tempreg3		;restore color
		sts	libmio_color,tempreg3
		ret

;-------------------------------------------------------------------------------
; plot pixline in graphics mode with font 1, YL=pixline
;-------------------------------------------------------------------------------
libmio_plpl5:	lds	tempreg3,libmio_color
		sbrc	YH,0
		swap	YL
		mov	tempreg4,tempreg3
		sbrs	YL,7			;pix 1
		swap	tempreg4		
		sts	libmio_color,tempreg4
		rcall	libmio_plot		;output pix
		inc	XL			;next pos
		sbrc	ctrl,6			;no bigchar
		rcall	libmio_plot		;output pix again
		sbrc	ctrl,6			;no bigchar
		inc	XL			;next pos
		
		mov	tempreg4,tempreg3
		sbrs	YL,6			;pix 2
		swap	tempreg4		
		sts	libmio_color,tempreg4
		rcall	libmio_plot		;output pix
		inc	XL			;next pos
		sbrc	ctrl,6			;no bigchar
		rcall	libmio_plot		;output pix again
		sbrc	ctrl,6			;no bigchar
		inc	XL			;next pos

		mov	tempreg4,tempreg3
		sbrs	YL,5			;pix 3
		swap	tempreg4		
		sts	libmio_color,tempreg4
		rcall	libmio_plot		;output pix
		inc	XL			;next pos
		sbrc	ctrl,6			;no bigchar
		rcall	libmio_plot		;output pix again
		sbrc	ctrl,6			;no bigchar
		inc	XL			;next pos

		mov	tempreg4,tempreg3
		sbrs	YL,4			;pix 4
		swap	tempreg4		
		sts	libmio_color,tempreg4
		rcall	libmio_plot		;output pix
		inc	XL			;next pos
		sbrc	ctrl,6			;no bigchar
		rcall	libmio_plot		;output pix again
		sbrc	ctrl,6			;no bigchar
		inc	XL			;next pos

libmio_plpl6:	sts	libmio_color,tempreg3
		ret

;-------------------------------------------------------------------------------
; output text from ROM (Z=text,nullterminated)
;-------------------------------------------------------------------------------
libmio_orom:	push	tempreg1
libmio_orom_1:	lpm	tempreg1,Z+		;get char
		cpi	tempreg1,0		;end?
		brne	libmio_orom_2		;no
		pop	tempreg1
		ret				;the end
libmio_orom_2:	rcall	libmio_char		;write char to screen
		rjmp	libmio_orom_1		;next char

;-------------------------------------------------------------------------------
; output text from ROM (following y,x,nullterminated text)
;-------------------------------------------------------------------------------
.ifdef use_libmio_othis
libmio_othis:	pop	ZH			;get return address
		pop	ZL
		lsl	ZL			;*2 so we use bytes
		rol	ZH
		push	XL
		push	XH
		lpm	XH,Z+			;get first value (255/Y)
		cpi	XH,0xff			;no position
		breq	libmio_othis_0
		lpm	XL,Z+			;get second value
		rcall	libmio_gxy		;set position
libmio_othis_0:	pop	XH
		pop	XL
		push	tempreg1		;save reg
libmio_othis_1:	lpm	tempreg1,Z+		;get char
		cpi	tempreg1,0		;end?
		brne	libmio_othis_2		;no
		pop	tempreg1		;restore reg
		sbiw	ZL,1			;point to zero
		lsr	ZH			;/2
		ror	ZL
		adiw	ZL,1			;point to next instruction
		ijmp
libmio_othis_2:	rcall	libmio_char		;write char to screen
		rjmp	libmio_othis_1		;next char
.endif

;-------------------------------------------------------------------------------
; output text from RAM (Z=text,nullterminated)
;-------------------------------------------------------------------------------
.ifdef use_libmio_oram
libmio_oram:	push	tempreg1
libmio_oram_1:	ld	tempreg1,Z+		;get char
		cpi	tempreg1,0		;end?
		brne	libmio_oram_2		;no
		pop	tempreg1
		ret				;the end
libmio_oram_2:	call	libmio_char		;write char to screen
		rjmp	libmio_oram_1		;next char
.endif

;-------------------------------------------------------------------------------
; output text (multiple char) ZL=number
;-------------------------------------------------------------------------------
.ifdef use_libmio_mtext
libmio_mtext:	call	libmio_char		;write char to screen
		dec	ZL
		brne	libmio_mtext
		ret
.endif

;-------------------------------------------------------------------------------
; set cursor to position XL:XH
;-------------------------------------------------------------------------------
.ifdef use_libmio_gotoxy
libmio_gxy:	push	ZL
		lds	ZL,libmio_vidmode
		cpi	ZL,0x00
		breq	libmio_gxy_00
		cpi	ZL,0x07			;custom
		brne	libmio_gxy_a
		push	ZH
		lds	ZH,prg_vdrv
		ldi	ZL,vdrv_gotoxy
		sbrs	ZH,7
		icall
		pop	ZH
		pop	ZL
		ret

libmio_gxy_a:	pop	ZL
libmio_gxy_1:	movw	libmio_cur_x,XL		;save X,Y-position
		ret

libmio_gxy_00:	pop	ZL
libmio_gxy_ap:	cpi	XL,libmio_cols
		brcs	libmio_gxy_01
		ldi	XL,libmio_cols-1
libmio_gxy_01:	cpi	XH,libmio_rows
		brcs	libmio_gxy_02
		ldi	XH,libmio_rows-1
libmio_gxy_02:	movw	libmio_cur_x,XL		;save X,Y-position
		ret
.endif

;-------------------------------------------------------------------------------
; plot  XL=X XH=Y
;-------------------------------------------------------------------------------
.ifdef use_libmio_plot
libmio_plot:	push	r16
		lds	r16,libmio_vidmode
		cpi	r16,0x07		;custom
		pop	r16
		brne	libmio_iplot		;internal
		push	ZH
		push	ZL
		lds	ZH,prg_vdrv
		ldi	ZL,vdrv_plot
		sbrs	ZH,7
		icall
		pop	ZL
		pop	ZH
		ret

libmio_iplot:	push_regs
		ldi	YH,HIGH(libmio_vram)	;home position
		ldi	YL,LOW(libmio_vram)	;home position
		lds	tempreg3,libmio_clipx2	;right limit
		cp	tempreg3,XL		;greater?
		brcs	libmio_nplot		;yes
		lds	tempreg3,libmio_clipy2	;right limit
		cp	tempreg3,XH		;greater?
		brcs	libmio_nplot		;yes
		rjmp	libmio_plot0		;OK, we can plot
libmio_nplot:	pop_regs			;restore registers

libmio_plot0:	lds	tempreg4,libmio_vidmode
		cpi	tempreg4,0x07		;copy for using mode 0
		breq	libmio_plot_0
		cpi	tempreg4,0x05
		breq	libmio_plot05
		cpi	tempreg4,0x04
		breq	libmio_plot04
		andi	tempreg4,0x03
		breq	libmio_plot_0
		cpi	tempreg4,0x02
		brcc	libmio_plot1
libmio_plot05:	rjmp	libmio_plot1_0
libmio_plot1:	brne	libmio_plot2
		rjmp	libmio_plot2_0
libmio_plot2:	rjmp	libmio_plot3_0
libmio_plot04:	rjmp	libmio_plot_4

libmio_plot_0:	ldi	tempreg2,1		;bit left top

		cpi	XL,libmio_cols*2	;limit X-coord
		brcc	libmio_plot_os
		cpi	XH,libmio_rows*2	;limit Y-coord
		brcc	libmio_plot_os

		sbrc	XH,0
		lsl	tempreg2
		sbrc	XH,0
		lsl	tempreg2
		sbrc	XL,0
		lsl	tempreg2		;tempreg2 = bitmask
		lsr	XH			;Y/2
		lsr	XL			;X/2
		ldi	tempreg3,libmio_cols
		mul	XH,tempreg3
		add	YL,XL
		adc	YH,const_0
		add	YL,r0
		adc	YH,r1
		ldi	XL,LOW(libmio_cols*libmio_rows)
		ldi	XH,HIGH(libmio_cols*libmio_rows)
		add	YL,XL			;calc attr pointer
		adc	YH,XH
		ld	XL,Y			;attr at position
		lds	XH,libmio_color		;plot color
		andi	XH,0x0f			;only background is needed
		swap	XH			;swap f/b
		andi	XL,0xf0			;only background is needed
		cp	XH,XL			;background = plot color?
		breq	libmio_plot_r		;pixel must be only deleted

		lds	XH,libmio_color		;plot color
		andi	XH,0x0f			;only foreground
		or	XL,XH			;set new fg
		st	Y,XL			;write back attribute

libmio_plot_s:	ldi	XL,LOW(libmio_cols*libmio_rows)
		ldi	XH,HIGH(libmio_cols*libmio_rows)
		sub	YL,XL			;recalc char pointer
		sbc	YH,XH
		ld	XH,Y			;get char
		or	XH,tempreg2		;set pix
libmio_plot_e:	andi	XH,0x0f			;mask bits
		st	Y,XH
libmio_plot_os:	pop_regs

libmio_plot_r:	ldi	XL,LOW(libmio_cols*libmio_rows)
		ldi	XH,HIGH(libmio_cols*libmio_rows)
		sub	YL,XL			;recalc char pointer
		sbc	YH,XH
		ld	XH,Y			;get char
		com	tempreg2
		and	XH,tempreg2		;clear pix
		rjmp	libmio_plot_e

libmio_plot1_0:	mov	tempreg3,XL		;X
		andi	tempreg3,0x07		;only bits
		inc	tempreg3		;+1
		ldi	tempreg1,0x00		;bitmask
		sec				;set carry
libmio_plot1_1:	rol	tempreg1		;shift in bit
		dec	tempreg3
		brne	libmio_plot1_1
;bit mask is in tempreg1
		lsr	XL			;X/8
		lsr	XL
		lsr	XL
		mov	ZL,XL
		ldi	tempreg3,libmio_v1cols/8
		lds 	ZH,libmio_vidmode
		sbrc	ZH,2
		ldi	tempreg3,libmio_v5cols/8
		mul	XH,tempreg3
		add	YL,XL
		adc	YH,const_0
		add	YL,r0
		adc	YH,r1
;address is now in Y
		lds	XL,libmio_color		;active color
		sbrs	ZH,2
		rjmp	libmio_plot1_3
		lsr	XL
		andi	XL,0x01			;we check only bit 0
		brne	libmio_plot1_2		;set
libmio_plot1_8:	com	tempreg1
		ld	XL,Y
		and	XL,tempreg1
		st	Y,XL
		rjmp	libmio_plot_os

libmio_plot1_3:	mov	tempreg2,ZL		;X/8
		lsr	XH			;Y/8
		lsr	XH
		lsr	XH
		ldi	ZH,21
		mul	ZH,XH
		ldi	ZL,LOW(libmio_vram+2436)
		ldi	ZH,HIGH(libmio_vram+2436)
		add	ZL,r0
		adc	ZH,r1
		add	ZL,tempreg2
		adc	ZH,const_0
		lds	XL,libmio_color		;color
		andi	XL,0x0f			;mask only fg to draw
		ld	tempreg2,Z		;get attr
		swap	tempreg2
		andi	tempreg2,0x0f		;only bg is needed
		cp	tempreg2,XL
		breq	libmio_plot1_8		;we must only clear pix

libmio_plot1_4:	ld	tempreg2,Z		;get attr
		andi	tempreg2,0xf0		;only bg
		or	tempreg2,XL		;set new fg
libmio_plot1_5:	st	Z,tempreg2		;store back attr

libmio_plot1_2:	ld	XL,Y
		or	XL,tempreg1
		st	Y,XL
		rjmp	libmio_plot_os

libmio_plot2_0:	mov	tempreg3,XL		;X
		lds	tempreg2,libmio_color	;active color
		lsr	tempreg2
		andi	tempreg2,0x03		;we check only bit 0+1
		ldi	tempreg1,0x03		;bitmask
		andi	tempreg3,0x03			;only bits
libmio_plot2_1:	breq	libmio_plot2_2
		lsl	tempreg1
		lsl	tempreg1
		lsl	tempreg2
		lsl	tempreg2
		dec	tempreg3
		brne	libmio_plot2_1
libmio_plot2_2:	com	tempreg1
;bit mask is in tempreg1
;value is in tempreg2
		lsr	XL			;X/4
		lsr	XL
		ldi	tempreg3,libmio_v2cols/4
		mul	XH,tempreg3
		add	YL,XL
		adc	YH,const_0
		add	YL,r0
		adc	YH,r1
;address is now in Y	    	
		ld	XL,Y
		and	XL,tempreg1
		or	XL,tempreg2
		st	Y,XL	
		rjmp	libmio_plot_os


libmio_plot3_0:	mov	tempreg3,XL		;X
		lds	tempreg2,libmio_color	;active color
		andi	tempreg2,0x0f		;we check only bit 0-3
		lsr	tempreg2
		brcc	libmio_plot3_i
		ori	tempreg2,0x08
libmio_plot3_i:	ldi	tempreg1,0x0f		;bitmask
		andi	tempreg3,0x01		;only bits
		breq	libmio_plot3_1
		swap	tempreg1
		swap	tempreg2
libmio_plot3_1:	com	tempreg1
;bit mask is in tempreg1
;value is in tempreg2
		lsr	XL			;X/2
		ldi	tempreg3,libmio_v3cols/2
		mul	XH,tempreg3
		add	YL,XL
		adc	YH,const_0
		add	YL,r0
		adc	YH,r1
;address is now in Y
		ld	XL,Y
		and	XL,tempreg1		;mask
		or	XL,tempreg2		;new bit
		st	Y,XL
		rjmp	libmio_plot_os

libmio_plot_4:	ldi	tempreg2,1		;bit left top

		cpi	XL,libmio_cols*2	;limit X-coord
		brcc	libmio_plot4c
		cpi	XH,libmio_rows*2	;limit Y-coord
		brcc	libmio_plot4c

		sbrc	XH,0
		lsl	tempreg2
		sbrc	XH,0
		lsl	tempreg2
		sbrc	XL,0
		lsl	tempreg2		;tempreg2 = bitmask
		lsr	XH			;Y/2
		lsr	XL			;X/2
		ldi	tempreg3,libmio_cols
		mul	XH,tempreg3
		ldi	YH,HIGH(libmio_vram)	;home position
		ldi	YL,LOW(libmio_vram)	;home position
		add	YL,XL
		adc	YH,const_0
		add	YL,r0
		adc	YH,r1
		lds	XH,libmio_color		;plot color
		andi	XH,0x02			;only background is needed
		breq	libmio_plot4b
		ld	XH,Y			;get char
		or	XH,tempreg2		;set pix
libmio_plot4a:	andi	XH,0x0f			;mask bits
		st	Y,XH
libmio_plot4c:	rjmp	libmio_plot_os

libmio_plot4b:	ld	XH,Y			;get char
		com	tempreg2
		and	XH,tempreg2		;clear pix
		rjmp	libmio_plot4a
.endif


;-------------------------------------------------------------------------------
; correct char coordinates from YL,YH to ZL,ZH
;-------------------------------------------------------------------------------
libmio_corrc:	cpi	YL,libmio_cols
		brcs	libmio_corrc1
		ldi	YL,libmio_cols-1
libmio_corrc1:	cpi	ZL,libmio_cols
		brcs	libmio_corrc2
		ldi	ZL,libmio_cols-1
libmio_corrc2:	cpi	YH,libmio_rows
		brcs	libmio_corrc3
		ldi	YH,libmio_rows-1
libmio_corrc3:	cpi	ZH,libmio_rows
		brcs	libmio_corrc4
		ldi	ZH,libmio_rows-1
libmio_corrc4:	cp	ZL,YL
		brcc	libmio_corrc5
		mov	r0,ZL
		mov	ZL,YL
		mov	YL,r0
libmio_corrc5:	cp	ZH,YH
		brcc	libmio_corrc6
		mov	r0,ZH
		mov	ZH,YH
		mov	YH,r0
libmio_corrc6:	ret

;-------------------------------------------------------------------------------
; correct plot coordinates from YL,YH to ZL,ZH
;-------------------------------------------------------------------------------
libmio_corrp:	cp	ZL,YL
		brcc	libmio_corrp1
		mov	r0,ZL
		mov	ZL,YL
		mov	YL,r0
libmio_corrp1:	cp	ZH,YH
		brcc	libmio_corrp2
		mov	r0,ZH
		mov	ZH,YH
		mov	YH,r0
libmio_corrp2:	ret

;-------------------------------------------------------------------------------
; cbox YL,YH to ZL,ZH (t=0)
; ibox YL,YH to ZL,ZH (t=1)
;-------------------------------------------------------------------------------
.ifdef use_libmio_cbox
libmio_cbox:	push_regs			;save registers
		rcall	libmio_corrc		;correct coords
		movw	tempreg7,ZL		;copy end coord
		lds	tempreg1,libmio_vidmode
		cpi	tempreg1,0
		breq	libmio_cbox_1
		cpi	tempreg1,4
		breq	libmio_cbox_1
		cpi	tempreg1,1
		brne	libmio_cbox_e
		ldi	XL,LOW(libmio_vram+2436)
		ldi	XH,HIGH(libmio_vram+2436)
		ldi	ZH,0x80
		ldi	tempreg4,21
		ldi	tempreg3,16
		rjmp	libmio_cbox_2
libmio_cbox_1:	ldi	XL,LOW(libmio_vram+690)
		ldi	XH,HIGH(libmio_vram+690)
		ldi	ZL,LOW(libmio_vram)
		ldi	ZH,HIGH(libmio_vram)
		ldi	tempreg4,30
		ldi	tempreg3,23

libmio_cbox_2:	cp	YL,tempreg4
		brcc	libmio_cbox_e
		cp	tempreg7,tempreg4
		brcc	libmio_cbox_e
		cp	YH,tempreg3
		brcc	libmio_cbox_e
		cp	tempreg8,tempreg3
		brcc	libmio_cbox_e
		mov	tempreg3,YL		;copy X-coord

libmio_cbox_3:	mul	tempreg4,YH		;calc offset of topleft corner
		add	XL,r0
		adc	XH,r1
		add	XL,YL
		adc	XH,const_0
		add	ZL,r0
		adc	ZH,r1
		add	ZL,YL
		adc	ZH,const_0
		lds	r0,libmio_color
		ldi	tempreg1,0x20		;space
		mov	tempreg2,tempreg4
		dec	tempreg2
		add	tempreg2,YL
		sub	tempreg2,tempreg7
		
libmio_cbox_4:	mov	tempreg3,YL		;X1
libmio_cbox_5:	brts	libmio_cbox_6		;branch if ibox
		sbrs	ZH,7
		st	Z+,tempreg1		;write space
		rjmp	libmio_cbox_7
libmio_cbox_6:	ld	r0,X
		swap	r0
libmio_cbox_7:	st	X+,r0
		cp	tempreg3,tempreg7
		breq	libmio_cbox_8
		inc	tempreg3
		rjmp	libmio_cbox_5
libmio_cbox_8:	add	XL,tempreg2		;go to next line
		adc	XH,const_0
		add	ZL,tempreg2		;go to next line
		adc	ZH,const_0
		cp	YH,tempreg8
		breq	libmio_cbox_e
		inc	YH
		rjmp	libmio_cbox_4
libmio_cbox_e:	pop_regs
.endif

;-------------------------------------------------------------------------------
; box YL,YH to ZL,ZH
;-------------------------------------------------------------------------------
.ifdef use_libmio_box
libmio_box:	push_ptr
		push	tempreg2
		rcall	libmio_corrp
libmio_box_2:	mov	XL,YL
		mov	XH,YH
libmio_box_3:	rcall	libmio_plot
		inc	XL
		cp	XL,ZL
		brne	libmio_box_3
libmio_box_4:	rcall	libmio_plot
		inc	XH
		cp	ZH,XH
		brne	libmio_box_4
libmio_box_5:	rcall	libmio_plot
		dec	XL
		cp	XL,YL
		brne	libmio_box_5
libmio_box_6:	rcall	libmio_plot
		dec	XH
		cp	XH,YH
		brne	libmio_box_6
		pop	tempreg2
		pop_ptr
.endif	

;-------------------------------------------------------------------------------
; fbox YL,YH to ZL,ZH
;-------------------------------------------------------------------------------
.ifdef use_libmio_fbox
libmio_fbox:	push_ptr
		push	tempreg2
		rcall	libmio_corrp
libmio_fbox_2:	mov	XH,YH
libmio_fbox_3:	mov	XL,YL
libmio_fbox_4:	rcall	libmio_plot
		cp	XL,ZL
		breq	libmio_fbox_5
		inc	XL
		rjmp	libmio_fbox_4
libmio_fbox_5:	rcall	libmio_plot
		cp	ZH,XH
		breq	libmio_fbox_6
		inc	XH
		rjmp	libmio_fbox_3
.endif

libmio_fbox_6:	pop	tempreg2
libmio_fbox_7:	pop_ptr

;-------------------------------------------------------------------------------
; windowbox YL,YH to ZL,ZH
;-------------------------------------------------------------------------------
.ifdef use_libmio_wbox
libmio_wbox:	push_regs
		clt				;set for fill
		rcall	libmio_cbox		;clear box
		rcall	libmio_corrc		;sort data
		mov	tempreg1,YL		;save X-pos
		mov	XH,YH
		mov	XL,YL
		rcall	libmio_calc1		;get address
		ldi	tempreg2,0xf2		;upper left corner
		st	Y+,tempreg2		;store
		ldi	tempreg2,0xf1		;hor line
libmio_wbox1:	inc	XL
		cp	XL,ZL
		breq	libmio_wbox2
		st	Y+,tempreg2		;store
		rjmp	libmio_wbox1		;loop
		
libmio_wbox2:	ldi	tempreg2,0xf3		;upper right corner
		st	Y+,tempreg2		;store
libmio_wbox3:	mov	XL,tempreg1		;restore X
		inc	XH			;next line
		cp	XH,ZH			;last line?
		breq	libmio_wbox4		;yes
		rcall	libmio_calc1		;get address
		ldi	tempreg2,0xf0		;vertical line
		st	Y+,tempreg2
		mov	XL,ZL
		rcall	libmio_calc1		;get address
		st	Y+,tempreg2
		rjmp	libmio_wbox3
libmio_wbox4:	mov	XL,tempreg1
		rcall	libmio_calc1		;get address
		ldi	tempreg2,0xf4		;lower left corner
		st	Y+,tempreg2		;store
		ldi	tempreg2,0xf1		;hor line
libmio_wbox5:	inc	XL
		cp	XL,ZL
		breq	libmio_wbox6
		st	Y+,tempreg2		;store
		rjmp	libmio_wbox5		;loop

libmio_wbox6:	ldi	tempreg2,0xf5		;lower right corner
		st	Y+,tempreg2		;store
		pop_regs
.endif

;-------------------------------------------------------------------------------
; windowheadbox YL,YH to ZL,ZH
;-------------------------------------------------------------------------------
.ifdef use_libmio_whbox
libmio_whbox:	rcall	libmio_wbox		;draw normal windowbox
		push_regs			;save registers

		rcall	libmio_corrc		;sort data
		mov	tempreg2,ZL		;X2
		sub	tempreg2,YL		;-X1
		subi	tempreg2,1		;sub corners
		sbrs	tempreg2,7		;skip if diff <2
		brne	libmio_whbox_1		;X1!=X2
		rjmp	libmio_whbox_e		;end

libmio_whbox_1:	ldi	ZH,0xfb
		ldi	ZL,0xec
		rcall	libmio_boxl
		inc	YH
		ldi	ZH,0xfd
		ldi	ZL,0x0f
		rcall	libmio_boxl
		inc	YH
		ldi	ZH,0xee
		ldi	ZL,0xed
		rcall	libmio_boxl
libmio_whbox_e:	pop_regs

libmio_boxl:	movw	XL,YL			;copy
		rcall	libmio_gxy
		mov	tempreg1,ZH
		rcall	libmio_char
		mov	tempreg3,tempreg2
libmio_whbox_2:	mov	tempreg1,ZL		;header fields
		rcall	libmio_char
		dec	tempreg3
		brne	libmio_whbox_2
		mov	tempreg1,ZH
		inc	tempreg1
		rjmp	libmio_char
.endif

;-------------------------------------------------------------------------------
; draw line from YL,YH to ZL,ZH
; this algorithm is a little bit quick and dirty, but it works...
;-------------------------------------------------------------------------------
.ifdef use_libmio_draw
libmio_draw:	push_regs
		movw	XL,YL			;copy start pointer
		ldi	tempreg1,0xff		;set sgn(dx) negative
		sub	YL,ZL			;x1-x2
		brcc	libmio_draw01		;branch if x1<x2
		neg	tempreg1		;invert sgn(dx)
		neg	YL			;invert dx
libmio_draw01:	ldi	tempreg2,0xff		;set sgn(dy) negative
		sub	YH,ZH			;-dy
		brcc	libmio_draw02		;branch if y1<y2
		neg	tempreg2		;invert sgn(dy)
		neg	YH			;invert dy
libmio_draw02:	cp	YL,YH			;|dx|-|dy|
		brcs	libmio_draw6		;branch if |dy|>|dx|
		rjmp	libmio_draw3

;|dx|>|dy|
libmio_draw3:	mov	tempreg3,YL		;|dx|
		lsr	tempreg3		;/2
libmio_draw4:	rcall	libmio_plot		;plot pixel
		cp	XL,ZL			;the endpoint?
		breq	libmio_draw_e		;yes
		add	XL,tempreg1		;X-step
		sub	tempreg3,YH		;-|dy|
		brcc	libmio_draw4		;no overflow
		add	XH,tempreg2		;y-step
		add	tempreg3,YL
		rjmp	libmio_draw4

;|dy|>|dx|
libmio_draw6:	mov	tempreg3,YH		;|dy|
		lsr	tempreg3		;/2
libmio_draw7:	rcall	libmio_plot		;plot pixel
		cp	XH,ZH			;the endpoint?
		breq	libmio_draw_e
		add	XH,tempreg2		;y-step
		sub	tempreg3,YL
		brcc	libmio_draw7
		add	XL,tempreg1		;x-step
		add	tempreg3,YH
		rjmp	libmio_draw7

libmio_draw_e:	pop_regs
.endif

;-------------------------------------------------------------------------------
; wait for the end of visible frame
;-------------------------------------------------------------------------------
.ifdef use_libmio_sync
libmio_sync:	push	XL
		ldi	XL,240
libmio_sync_1:	cp	XL,r8
		breq	libmio_sync_1
libmio_sync_2:	cp	XL,r8
		brne	libmio_sync_2
		pop	XL
		ret
.endif

;-------------------------------------------------------------------------------
; play a note (XL=note 0..63 0xff=>noise)
;-------------------------------------------------------------------------------
libmio_note:	push_ptr
		lds	ZH,prg_sound
		sbrc	ZH,7
		rjmp	libmio_note_1
		ldi	ZL,note_entry
		icall
		rjmp	libmio_note_2

libmio_note_1:	sts	libmio_note_val,XL	;write note value
		ldi	ZH,HIGH(libmio_wtable1*2)
		sbrc	XL,6
		ldi	ZH,HIGH(libmio_wtable2*2)
		sts	libmio_wavtab,ZH
		andi	XL,0x3f			;64 notes
		lsl	XL
		ldi	ZH,HIGH(libmio_ntable*2)	;1 start envtable high
		ldi	ZL,LOW(libmio_ntable*2)	;1 start envtable low
		add	ZL,XL			;offsettable
		adc	ZH,const_0		;
		lpm	XL,Z+			;get low offset
		sts	libmio_woff_l,XL	;write
		lpm	XL,Z+			;get high offset
		sts	libmio_woff_h,XL	;write
		sts	libmio_env,const_0	;start
		sts	libmio_wpos_l,const_0
		sts	libmio_wpos_h,const_0
libmio_note_2:	pop_ptr


;------------------------------------------------------------------------------
;wait for a key
;------------------------------------------------------------------------------
libmio_waitkey:	sts	libmio_lastkey,const_0
libmio_wkey1:	lds	tempreg1,libmio_lastkey
		cpi	tempreg1,0
		breq	libmio_wkey1

		rjmp	libmio_scrs

		ret

;------------------------------------------------------------------------------
;wait for no key
;------------------------------------------------------------------------------
libmio_nokey:	push	tempreg1
libmio_nokey_1:	lds	tempreg1,libmio_keycode
		cpi	tempreg1,0
		brne	libmio_nokey_1
		pop	tempreg1
		ret

;------------------------------------------------------------------------------
;blink cursor and wait for key (return x,y in X ; adr in Y ; key in tempreg1)
;------------------------------------------------------------------------------
.ifdef use_libmio_cursor
libmio_cursor:	rcall	libmio_calc		;calculate address
		ldi	XL,LOW(690)
		add	YL,XL
		ldi	XL,HIGH(690)
		adc	YH,XL

libmio_cur1:	ld	XL,Y			;get original color
		mov	XH,XL			;copy
		swap	XH
		sts	libmio_lastkey,const_0	;clear key
libmio_cur_1:	lds	tempreg1,libmio_lastkey
		cpi	tempreg1,0
		brne	libmio_cur_e

		lds	tempreg1,libmio_frames
		cpi	tempreg1,27		;ca. half of fps
		brcc	libmio_cur_2
		st	Y,XL
		rjmp	libmio_cur_1

libmio_cur_2:	st	Y,XH
		rjmp	libmio_cur_1

libmio_cur_e:	st	Y,XL			;put back original char

.ifdef use_libmio_screenshot
		rcall	libmio_scrs
.endif
		rjmp	libmio_calc
.endif

;------------------------------------------------------------------------------
;blink cursor and wait for key (return x,y in X ; adr in Y ; key in tempreg1)
;------------------------------------------------------------------------------
.ifdef use_libmio_ecursor
libmio_ecursor:	push	r19
		ldi	r19,0xdd		;dummy
		rcall	libmio_ecalc
		sts	libmio_lastkey,const_0	;clear key
libmio_ecur_1:	lds	tempreg1,libmio_kbdstate
		andi	tempreg1,0x04		;left ctrl
		cp	tempreg1,r19
		breq	libmio_ecur_c
		cpi	tempreg1,0x00		;no
		brne	libmio_ecur_b
		call	editor_shline
		mov	r19,tempreg1
		rjmp	libmio_ecur_c
libmio_ecur_b:	call	editor_chline
		mov	r19,tempreg1
libmio_ecur_c:	lds	tempreg1,libmio_lastkey
		cpi	tempreg1,0
		brne	libmio_ecur_e
		lds	tempreg1,libmio_frames
		cpi	tempreg1,27		;ca. half of fps
		brcc	libmio_ecur_2
		ld	tempreg1,Y		;get char
		andi	tempreg1,0x7f		;set normal
		st	Y,tempreg1
		rjmp	libmio_ecur_1
libmio_ecur_2:	ld	tempreg1,Y		;get char
		ori	tempreg1,0x80		;set inverse
		st	Y,tempreg1
		rjmp	libmio_ecur_1

libmio_ecur_e:	ld	tempreg2,Y		;get char
		andi	tempreg2,0x7f		;set normal
		st	Y,tempreg2
		pop	r19
		ret
.endif

;------------------------------------------------------------------------------
;blink cursor and wait for key (return x,y in X ; adr in Y ; key in tempreg1)
;------------------------------------------------------------------------------
.ifdef use_libmio_ncursor
libmio_ncursor:	ldi	YH,HIGH(libmio_vram+11)	;home position
		ldi	YL,LOW(libmio_vram+11)	;home position
		add	YL,XL			;position+x
		adc	YH,const_0		
		sts	libmio_lastkey,const_0	;clear key
libmio_ncur_1:	lds	tempreg1,libmio_lastkey
		cpi	tempreg1,0
		brne	libmio_ncur_e
		lds	tempreg1,libmio_frames
		cpi	tempreg1,27		;ca. half of fps
		brcc	libmio_ncur_2
		ld	tempreg1,Y		;get char
		andi	tempreg1,0x7f		;set normal
		st	Y,tempreg1
		rjmp	libmio_ncur_1
libmio_ncur_2:	ld	tempreg1,Y		;get char
		ori	tempreg1,0x80		;set inverse
		st	Y,tempreg1
		rjmp	libmio_ncur_1

libmio_ncur_e:	ld	tempreg2,Y		;get char
		andi	tempreg2,0x7f		;set normal
		st	Y,tempreg2
		ret
.endif

;------------------------------------------------------------------------------
;calculate char address to Y
;------------------------------------------------------------------------------
libmio_calc:	movw	XL,libmio_cur_x		;X
libmio_calc1:	push	r0
		push	r1
		ldi	YH,libmio_cols		;chars per line
		mul	XH,YH			;store
		ldi	YH,HIGH(libmio_vram)	;home position
		ldi	YL,LOW(libmio_vram)	;home position
		add	YL,XL			;position+x
		adc	YH,const_0		
		add	YL,r0			;position+(cols*y)
		adc	YH,r1
		pop	r1
		pop	r0
		ret

;------------------------------------------------------------------------------
;calculate attributes address to Y
;------------------------------------------------------------------------------
libmio_calc2:	rcall	libmio_calc1
		push	r16
		ldi	r16,LOW(libmio_cols*libmio_rows)
		add	YL,r16
		ldi	r16,HIGH(libmio_cols*libmio_rows)
		adc	YH,r16
		pop	r16
		ret

;------------------------------------------------------------------------------
;calculate ecursor address to Y
;------------------------------------------------------------------------------
libmio_ecalc:	ldi	YH,libmio_ecols		;chars per line
		mul	XH,YH			;store
		ldi	YH,HIGH(libmio_vram+3*libmio_ecols+3)	;home position
		ldi	YL,LOW(libmio_vram+3*libmio_ecols+3)	;home position
		add	YL,XL			;position+x
		adc	YH,const_0		
		add	YL,r0			;position+(cols*y)
		adc	YH,r1
		ret

;------------------------------------------------------------------------------
;get char on cursor address
;------------------------------------------------------------------------------
.ifdef use_libmio_gchar
libmio_gchar:	rcall	libmio_calc
		ld	tempreg1,Y
		ret
.endif

;------------------------------------------------------------------------------
;set cursor to down position
;------------------------------------------------------------------------------
.ifdef use_libmio_vcur
libmio_vcur:	movw	XL,libmio_cur_x		;X
		dec	XL
		inc	XH
		rjmp	libmio_gxy		;set new cursor position
.endif

;------------------------------------------------------------------------------
;put tempreg1 to printer output
;------------------------------------------------------------------------------
libmio_ppar:	push	ZH
		call	sys_iocheck
		brts	libmio_ppar_0
		ldi	ereg,38			;io disabled
		rjmp	libmio_ppar_4

libmio_ppar_0:	ldi	ZH,0xff
		out	DDRA,ZH			;set port A0-7 as outputs
		nop
libmio_ppar_1:	sbic	PINB,2			;busy
		rjmp	libmio_ppar_1		;yes
		out	PORTA,tempreg1		;data
		ldi	ZH,100			;pause
libmio_ppar_2:	dec	ZH
		brne	libmio_ppar_2
		cbi	PORTB,1			;clear strobe (active)
		ldi	ZH,100			;pause
libmio_ppar_3:	dec	ZH
		brne	libmio_ppar_3
		sbi	PORTB,1			;set strobe (inactive)
libmio_ppar_4:	pop	ZH
		ret				;thats all

;------------------------------------------------------------------------------
;output screenshot
;------------------------------------------------------------------------------
libmio_scrs:	push	tempreg1		;save tempreg1, w need for checking
		lds	tempreg1,libmio_kflags	;get key flags
		sbrc	tempreg1,1		;screenshot-flag
		rjmp	libmio_scrs_1
		pop	tempreg1		;restore regs
		ret

libmio_scrs_e:	pop	ZH
		pop	ZL
		pop	YH
		pop	YL
		pop	XH
		pop	XL
		pop	tempreg1		;restore regs
		clr	tempreg1
		ret

libmio_scrs_1:	push	XL
		push	XH
		push	YL
		push	YH
		push	ZL
		push	ZH
		andi	tempreg1,0xfd		;clear screenshot-flag
		sts	libmio_kflags,tempreg1	;write back
		lds	XL,libmio_vidmode	;video mode

		mov	tempreg1,XL		;vmode
		rcall	libmio_scrs_c		;output videomode
		lds	tempreg1,libmio_border	;border
		rcall	libmio_scrs_cc		;output border
		lds	tempreg1,libmio_estatus	;color of status line
		rcall	libmio_scrs_cc		;out

		ldi	YL,LOW(libmio_palette)
		ldi	YH,HIGH(libmio_palette)
		ldi	XH,16
libmio_scrs_2:	ld	tempreg1,Y+
		swap	tempreg1
		rcall	libmio_scrs_c
		dec	XH
		brne	libmio_scrs_2
		rcall	libmio_sysnl
		ldi	YL,LOW(libmio_vram)
		ldi	YH,HIGH(libmio_vram)

;text
		cpi	XL,0			;textmode?
		brne	libmio_scrs_02		;no
		ldi	ZH,libmio_rows*2	;number of charlines
libmio_scrs_01:	ldi	ZL,libmio_cols		;number of chars per line
		rcall	libmio_scrs_li
		dec	ZH
		brne	libmio_scrs_01
		rjmp	libmio_scrs_e

;editor
libmio_scrs_02:	cpi	XL,8			;editormode?
		brne	libmio_scrs_10		;no
		ldi	ZH,0x03			;number of fix charlines
libmio_scrs_06:	ldi	ZL,libmio_ecols		;number of chars per line
libmio_scrs_07:	rcall	libmio_scrs_li
		dec	ZH
		brne	libmio_scrs_06
		ldi	ZL,libmio_ecols		;calc startadr of listing 
		lds	ZH,libmio_edstart
		mul	ZL,ZH
		add	YL,r0
		adc	YH,r1
		ldi	ZH,0x10			;number of edit charlines
libmio_scrs_08:	ldi	ZL,libmio_ecols		;number of chars per line
libmio_scrs_09:	ld	tempreg1,Y
;		andi	tempreg1,0x7f		;mask invert bit
		swap	tempreg1		;high nibble first
		rcall	libmio_scrs_c
		ld	tempreg1,Y+
		andi	tempreg1,0x7f		;mask invert bit
		rcall	libmio_scrs_c
		dec	ZL
		brne	libmio_scrs_09
		call	libmio_sysnl
		dec	ZH
		brne	libmio_scrs_08
		rjmp	libmio_scrs_e

;graph 1
libmio_scrs_10:	cpi	XL,1
		brne	libmio_scrs_20
		ldi	ZH,131			;number of pixlines+attrlines
libmio_scrs_11:	ldi	ZL,21			;number of chars per line
		rcall	libmio_scrs_li
		dec	ZH
		brne	libmio_scrs_11
		rjmp	libmio_scrs_e

;graph 2
libmio_scrs_20:	cpi	XL,2
		brne	libmio_scrs_30
		ldi	ZH,84			;number of pixlines
libmio_scrs_21:	ldi	ZL,30			;number of chars per line
		rcall	libmio_scrs_li
		dec	ZH
		brne	libmio_scrs_21
		rjmp	libmio_scrs_e

;graph 3
libmio_scrs_30:	cpi	XL,3
		brne	libmio_scrs_40
		ldi	ZH,58			;number of pixlines
libmio_scrs_31:	ldi	ZL,42			;number of chars per line
		rcall	libmio_scrs_li
		dec	ZH
		brne	libmio_scrs_31
		rjmp	libmio_scrs_e

;mode 4 (user 1)
libmio_scrs_40:	cpi	XL,4			;user 1?
		brne	libmio_scrs_50		;no
libmio_scrs_41:	ldi	ZH,87			;number of charlines
		rjmp	libmio_scrs_01

;graph 5
libmio_scrs_50:	cpi	XL,5			;user 1?
		brne	libmio_scrs_60		;no
		ldi	ZH,64			;number of pixlines
libmio_scrs_51:	ldi	ZL,16			;number of chars per line
		rcall	libmio_scrs_li
		dec	ZH
		brne	libmio_scrs_51
		rjmp	libmio_scrs_e

;mode 6 (user 2)
libmio_scrs_60:	cpi	XL,6			;user 2?
		breq	libmio_scrs_41		;no

;mode 7 (driver)
		cpi	XL,7			;custom
		brne	libmio_scrs_80
		ldi	ZL,0x0e
		ldi	ZH,0xe4
		lpm	tempreg1,Z		;library code
		swap	tempreg1
		rcall	libmio_scrs_c
		lpm	tempreg1,Z+
		rcall	libmio_scrs_c
		lpm	tempreg1,Z		;library version
		swap	tempreg1
		rcall	libmio_scrs_c
		lpm	tempreg1,Z+
		rcall	libmio_scrs_c
		call	libmio_sysnl
		ldi	ZH,92			;number of pixlines
libmio_scrs_71:	ldi	ZL,30			;number of chars per line
		rcall	libmio_scrs_li
		dec	ZH
		brne	libmio_scrs_71
libmio_scrs_80:	rjmp	libmio_scrs_e		;no valid mode

libmio_scrs_cc:	swap	tempreg1		;
		lsr	tempreg1
		andi	tempreg1,0x07		;mask bits

libmio_scrs_c:	andi	tempreg1,0x0f		;mask bits
		subi	tempreg1,0xd0		;+30
		cpi	tempreg1,0x3a		;>9
		brcs	libmio_scrs_c1		;jump, if not
		subi	tempreg1,0xf9		;+6
libmio_scrs_c1:	jmp	libmio_sysput

libmio_scrs_li:	ld	tempreg1,Y
		swap	tempreg1		;high nibble first
		rcall	libmio_scrs_c
		ld	tempreg1,Y+
		rcall	libmio_scrs_c
		dec	ZL
		brne	libmio_scrs_li
		jmp	libmio_sysnl

;------------------------------------------------------------------------------
;enable/disable chargepump
;------------------------------------------------------------------------------
.ifdef use_libmio_chp
libmio_chp:	push	XH
		ldi	XH,0x02			;disable output
		cpi	XL,0			;is zero
		breq	libmio_chp1
		ori	XH,0x40			;enable output
libmio_chp1:	out	TCCR0A,XH
		pop	XH
		ret
.endif


;------------------------------------------------------------------------------
;put newline to serial output
;------------------------------------------------------------------------------
libmio_sysnl:	sbis	GPIOR0,0		;sysbaud-flag
		rjmp	libmio_sernl2

libmio_sernl:	push	tempreg1
		ldi	tempreg1,0x0d		;CR
		sbic	GPIOR0,6		;skip if no CR
		rcall	libmio_pser
		ldi	tempreg1,0x0a		;
		sbic	GPIOR0,7		;skip if no LF
		rcall	libmio_pser
		pop	tempreg1
		ret

;------------------------------------------------------------------------------
;put newline to serial output2
;------------------------------------------------------------------------------
libmio_sernl2:	push	tempreg1
		ldi	tempreg1,0x0d		;CR
		sbic	GPIOR0,6		;skip if no CR
		rcall	libmio_pser2
		ldi	tempreg1,0x0a		;
		sbic	GPIOR0,7		;skip if no LF
		rcall	libmio_pser2
		pop	tempreg1
		ret

;------------------------------------------------------------------------------
;put tempreg1 to serial output 1
;------------------------------------------------------------------------------
libmio_sysput:	sbis	GPIOR0,0		;sysbaud-flag
		rjmp	libmio_pser2

libmio_pser:	push	XH
libmio_pser_1:	lds	XH,libmio_tcnt		;counter
		cpi	XH,0x00			;ready?
		brne	libmio_pser_1		;loop
		sts	libmio_tdat,tempreg1	;store data
		dec	XH			;prepare 0xff
		sts	libmio_tcnt,XH		;store start
		pop	XH
		ret				;thats all

;------------------------------------------------------------------------------
;put tempreg1 to serial output 2
;------------------------------------------------------------------------------
libmio_pser2:	sbis	GPIOR0,2		;skip if P-type
		ret

		push	tempreg2
libmio_pser2a:	lds	tempreg2,UCSR1A
		sbrs	tempreg2,5
		rjmp	libmio_pser2a
		sts	UDR1,tempreg1
		pop	tempreg2
		ret

;------------------------------------------------------------------------------
;get tempreg1 from serial input
;------------------------------------------------------------------------------
libmio_sysget:	sbis	GPIOR0,0		;sysbaud-flag
		rjmp	libmio_gser2

libmio_gser:	push	XH
libmio_gser_1:	lds	XH,libmio_rcnt		;counter
		cpi	XH,0x00			;ready?
		brne	libmio_gser_1		;loop
		dec	XH			;prepare 0xff
		sts	libmio_rcnt,XH		;store start
libmio_gser_2:	lds	XH,libmio_rcnt		;counter
		cpi	XH,0x00			;ready?
		brne	libmio_gser_2		;loop
		lds	tempreg1,libmio_rdat	;get char
libmio_pconv1:	pop	XH
		ret				;thats all

;------------------------------------------------------------------------------
;get tempreg1 from serial input 2
;------------------------------------------------------------------------------
libmio_gser2:	clr	tempreg1		;return zero
		sbis	GPIOR0,2		;skip if P-type
		ret				;return if only one USART

libmio_gser2a:	lds	tempreg1,UCSR1A
		sbrs	tempreg1,7
		rjmp	libmio_gser2a
		lds	tempreg1,UDR1
		ret

;------------------------------------------------------------------------------
;get tempreg1 from serial input 2 (XL=baudrate selector)
;------------------------------------------------------------------------------
libmio_baud2:	sbis	GPIOR0,2		;skip if P-type
		ret

libmio_baud:	push_ptr
		mov	XH,XL			;copy
		lsl	XH
		andi	XH,0x0e
		ldi	ZL,LOW(libmio_brate*2)
		ldi	ZH,HIGH(libmio_brate*2)
		add	ZL,XH
		adc	ZH,const_0
		lpm	XH,Z+
		sts	UBRR1H,XH
		lpm	XH,Z+
		sts	UBRR1L,XH
		lds	XH,UCSR1C		;copy
		andi	XH,0x06
		bst	XL,4			;stopp-bits
		bld	XH,3			;0=1, 1=2
		sbrs	XL,5			;parity
		rjmp	libmio_baud_np		;no parity
		ori	XH,0x20			;enable parity
		bst	XL,6			;parity select bit
		bld	XH,4			;0=even, 1=odd
		rjmp	libmio_baud_1

libmio_baud_np:	sbrc	XL,6			;bits-2
		andi	XH,0xfb			;0=7/8, 1=5/6

libmio_baud_1:	sbrc	XL,7
		andi	XH,0xfd			;0=8bit 1=7bit
		sts	UCSR1C,XH
		pop_ptr

libmio_brate:
		.db	8,35			;1200
		.db	4,17			;2400
		.db	2,8			;4800
		.db	1,3			;9600
		.db	0,129			;19200
		.db	0,80			;31250
		.db	0,64			;38400
		.db	0,42			;57600

;------------------------------------------------------------------------------
;get tempreg1 from serial input with break
;------------------------------------------------------------------------------
libmio_sysgetb:	sbis	GPIOR0,0		;sysbaud-flag
		rjmp	libmio_gserb2

libmio_gserb:	push	XH
libmio_gserb_1:	lds	XH,libmio_rcnt		;counter
		cpi	XH,0x00			;ready?
		brne	libmio_gserb_1		;loop
		dec	XH			;prepare 0xff
		sts	libmio_rcnt,XH		;store start
libmio_gserb_2:	lds	tempreg1,libmio_keycode	;get keycode
		cpi	tempreg1,0xed		;ESC
		breq	libmio_gserb_4		;end if ESC key is pressed
		lds	XH,libmio_rcnt		;counter
		cpi	XH,0x00			;ready?
		brne	libmio_gserb_2		;loop
		lds	tempreg1,libmio_rdat	;get char
libmio_gserb_3:	pop	XH
		ret				;thats all
libmio_gserb_4:	ldi	tempreg1,0x1b		;ESC
		pop	XH
		ret				;thats all

;------------------------------------------------------------------------------
;get tempreg1 from serial input 2
;------------------------------------------------------------------------------
libmio_gserb2:	clr	tempreg1		;return zero
		sbis	GPIOR0,2		;skip if P-type
		ret

libmio_gser2ba:	lds	tempreg1,libmio_keycode	;get keycode
		cpi	tempreg1,0xed		;ESC
		breq	libmio_gserb_4		;end if ESC key is pressed
		lds	tempreg1,UCSR1A
		sbrs	tempreg1,7
		rjmp	libmio_gser2ba
		lds	tempreg1,UDR1
		ret

;-------------------------------------------------------------------------------
; draw menu from Z and exit with a valid key in tempreg1 
;-------------------------------------------------------------------------------
libmio_menu:	push_ptr
;		rcall	libmio_nokey
		push	r18
		movw	r0,ZL				;save ptr
		adiw	ZL,30				;point Z to shifting flag
		lpm	r18,Z				;get shifting and key flag
		ldi	YL,LOW(libmio_vram+1350)	;attr address
		ldi	YH,HIGH(libmio_vram+1350)
		ldi	ZL,LOW(libmio_menu_at*2)
		ldi	ZH,HIGH(libmio_menu_at*2)
		rcall	libmio_menu_fc			;copy from Flash
		sts	libmio_lastkey,const_0
libmio_menu_1:	movw	ZL,r0
		rcall	libmio_scrs
		ldi	YL,LOW(libmio_vram+660)		;char address
		ldi	YH,HIGH(libmio_vram+660)
		sbrs	r18,0
		rjmp	libmio_menu_2			;no shifting
		lds	XH,libmio_kbdstate
		sbrc	XH,2				;left ctrl
		adiw	ZL,32				;ctrl level
libmio_menu_2:	rcall	libmio_menu_fc			;copy from flash
libmio_menu_4:	lds	tempreg1,libmio_lastkey
		cpi	tempreg1,0			;no key
		breq	libmio_menu_1
		ldi	XL,0x01
		cpi	tempreg1,0xed			;ESC
		breq	libmio_menu_10
		ldi	XL,0x02
		cpi	tempreg1,0xf1			;F1
		breq	libmio_menu_10
		ldi	XL,0x03
		cpi	tempreg1,0xf2			;F2
		breq	libmio_menu_10
		ldi	XL,0x04
		cpi	tempreg1,0xf3			;F3
		breq	libmio_menu_10
		ldi	XL,0x05
		cpi	tempreg1,0xf4			;F4
		breq	libmio_menu_10
		sbrc	r18,1				;key enable flag
		rjmp	libmio_menu_12			;branch if all keys enabled
		rjmp	libmio_menu_1			;no valid key

libmio_menu_10:	lds	XH,libmio_kbdstate
		sbrc	XH,2
		subi	XL,0xfb				;+5 for ctrl
		mov	tempreg1,XL			;copy result
libmio_menu_12:	pop	r18
		pop_ptr

;copy from flash
libmio_menu_fc:	ldi	XH,30
libmio_menu_f1:	lpm	XL,Z+
		st	Y+,XL
		dec	XH
		brne	libmio_menu_f1
		ret

;color table
libmio_menu_at:	.db	0x4e,0x4e,0x4e,0x4e,0x4e,0x4e
		.db	0xc0,0xc0,0xc0,0xc0,0xc0,0xc0
		.db	0x80,0x80,0x80,0x80,0x80,0x80
		.db	0x2e,0x2e,0x2e,0x2e,0x2e,0x2e
		.db	0x6e,0x6e,0x6e,0x6e,0x6e,0x6e

;only draw menu
libmio_domenu:	push_ptr
		ldi	YL,LOW(libmio_vram+660)		;char address
		ldi	YH,HIGH(libmio_vram+660)
		rcall	libmio_menu_fc			;copy from flash
		ldi	YL,LOW(libmio_vram+1350)	;attr address
		ldi	YH,HIGH(libmio_vram+1350)
		ldi	ZL,LOW(libmio_menu_at*2)
		ldi	ZH,HIGH(libmio_menu_at*2)
		rcall	libmio_menu_fc			;copy from Flash
		pop_ptr

;-------------------------------------------------------------------------------
; set config for I2C and SPI
;-------------------------------------------------------------------------------
libmio_setconf:	in	XH,GPIOR0
libmio_setcf1:	ldi	XL,0x5c		;f/4
		sbrc	XH,3
		ldi	XL,0x5f		;f/128
		out	SPCR,XL
		out	SPSR,const_0
		ldi	XL,17		;speed
		sts	TWBR,XL		
		ldi	XL,0x00		;/1
		sbrs	XH,4
		ldi	XL,0x01		;/4
		sts	TWSR,XL
		ldi	XL,3
		lds	XH,PCMSK0
		sbrc	XH,0
		ldi	XL,6
		rjmp	libmio_baud2	;set ser2 baudrate

;-------------------------------------------------------------------------------
; pseudo-random generator (external call)
;-------------------------------------------------------------------------------
libmio_psrg:	mov	XL,r8		;get vline_l
libmio_psrg1:	cp	XL,r8		;wait for change
		breq	libmio_psrg1
		lds	XL,libmio_rand3
		lds	XH,libmio_rand2
		ret

;-------------------------------------------------------------------------------
; copy char table to RAM (and convert) 
;-------------------------------------------------------------------------------
libmio_copytab:	push_regs
		ldi	XL,LOW(libmio_vram)	;base address
		ldi	XH,HIGH(libmio_vram)
		ldi	YL,LOW(690)		;bytes to write
		ldi	YH,HIGH(690)
libmio_copyt_1:	st	X+,const_0		;set to zero
		sbiw	YL,1
		brne	libmio_copyt_1
;		rjmp	libmio_copyt_9

		ldi	tempreg1,0		;first char
libmio_copyt_2:	ldi	tempreg2,0		;first pixline
libmio_copyt_3:	rcall	libmio_copyti
		inc	tempreg1
		cpi	tempreg1,64
		brne	libmio_copyt_2
libmio_copyt_9:	pop_regs

libmio_exx:	sbrs	tempreg3,7
		rjmp	libmio_exx_1
		push	tempreg1
		mov	tempreg1,YL		;set data to 0
		mov	YL,YH			;set data to 1
		mov	YH,tempreg1
		pop	tempreg1
libmio_exx_1:	or	tempreg4,YL
		lsl	tempreg3
		ret
;innner loop
libmio_copyti:	ldi	ZL,LOW(libmio_ctable*2)	;character table
		ldi	ZH,HIGH(libmio_ctable*2)
		add	ZH,tempreg2		;add pixline offset
		add	ZL,tempreg1		;add char
		lpm	tempreg3,Z		;get pixline
		lds 	YL,libmio_color
		mov	YH,YL
		swap	YH
		andi	YL,0xf0
		andi	YH,0xf0
		ldi	ZL,3
libmio_copyti1:	clr	tempreg4
		rcall	libmio_exx
		swap	tempreg4
		rcall	libmio_exx
		swap	tempreg4
		st	X+,tempreg4
		dec	ZL
		brne	libmio_copyti1
		inc	tempreg2
		cpi	tempreg2,10
		brne	libmio_copyti
		ret

;-------------------------------------------------------------------------------
; copy char table to RAM (and convert) for
;-------------------------------------------------------------------------------
libmio_cptab:	push_ptr
		push	tempreg1
		ldi	XL,0x80			;base address
		ldi	XH,0x07
		ldi	ZL,LOW(libmio_ctable*2)	;character table
		ldi	ZH,HIGH(libmio_ctable*2)
		ldi	YL,0x0a			;number of charlines
libmio_cptab_1:	ldi	YH,0x80			;number of bytes
libmio_cptab_2:	lpm	tempreg1,Z+
		st	X+,tempreg1
		dec	YH
		brne	libmio_cptab_2
		adiw	ZL,0x30
		adiw	ZL,0x30
		adiw	ZL,0x20
		dec	YL
		brne	libmio_cptab_1

		pop	tempreg1
		pop_ptr

;-------------------------------------------------------------------------------
; set volume to XH
;-------------------------------------------------------------------------------
libmio_setvol:	push	ZH
		push	ZL
		andi	XH,0x0f
		ldi	ZL,LOW(libmio_etable*2+15)
		ldi	ZH,HIGH(libmio_etable*2+15)
		sub	ZL,XH
		lpm	ZH,Z
		lsl	ZH
		dec	ZH
		sts	libmio_volume,ZH
		brtc	libmio_setvol1
		sts	libmio_mvolume,ZH
libmio_setvol1:	pop	ZL
		pop	ZH
		ret

;-------------------------------------------------------------------------------
; set video on/off
;-------------------------------------------------------------------------------
libmio_vidoff:	push	XL
		in	XL,GPIOR2
		ori	XL,0x20
libmio_vidoff1:	out	GPIOR2,XL
		pop	XL
		ret

libmio_vidon:	push	XL
		in	XL,GPIOR2
		andi	XL,0xdf
		rjmp	libmio_vidoff1

;-------------------------------------------------------------------------------
; copy char matrices in mode 6
; tempreg1 = src char
; tempreg2 = dest char
; tempreg3 = color
;-------------------------------------------------------------------------------
libmio_cpbits6:	push_regs
		ldi	XL,LOW(libmio_vram+690)
		ldi	XH,HIGH(libmio_vram+690)
		ldi	tempreg1,30
		andi	tempreg2,0x3f
		mul	tempreg1,tempreg2
		add	XL,r0
		adc	XH,r1
		lds	r0,libmio_color
		sts	libmio_color,tempreg3
		clr	tempreg2
		call	libmio_copyti
		sts	libmio_color,r0
		pop_regs

;-------------------------------------------------------------------------------
; copy char matrices in mode 4
; tempreg1 = src char
; tempreg2 = dest char
;-------------------------------------------------------------------------------
libmio_cpbits4:	push_regs
		mov	ZL,tempreg1		;src char
		ldi	ZH,HIGH(libmio_ctable*2)
		ldi	XL,0x80			;offset
		ldi	XH,0x07
		andi	tempreg2,0x7f
		add	XL,tempreg2
		ldi	tempreg2,0x0a
		ldi	tempreg3,0x80
libmio_cpb4_1:	lpm	tempreg1,Z
		st	X,tempreg1
		inc	ZH
		add	XL,tempreg3
		adc	XH,const_0
		dec	tempreg2
		brne	libmio_cpb4_1
		pop_regs

;-------------------------------------------------------------------------------
; include additional functions
;-------------------------------------------------------------------------------
.include 	"libmio/bcopy.asm"
.include 	"libmio/circle.asm"
.include	"libmio/alert.asm"
.include	"libmio/sprites.asm"
.include	"libmio/scroll.asm"
.include	"libmio/xmodem.asm"
.include	"libmio/sq.asm"

