;################################################################################
;#										#
;# Z80 emulator		 							#
;# copyright (c) 2009-2011 Joerg Wolfram (joerg@jcwolfram.de)			#
;#										#
;#										#
;# This program is free software; you can redistribute it and/or		#
;# modify it under the terms of the GNU General Public License			#
;# as published by the Free Software Foundation; either version 3		#
;# of the License, or (at your option) any later version.			#
;#										#
;# This program is distributed in the hope that it will be useful,		#
;# but WITHOUT ANY WARRANTY; without even the implied warranty of		#
;# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the GNU		#
;# General Public License for more details.					#
;#										#
;# You should have received a copy of the GNU General Public			#
;# License along with this library; if not, write to the			#
;# Free Software Foundation, Inc., 59 Temple Place - Suite 330,			#
;# Boston, MA 02111-1307, USA.							#
;#										#
;################################################################################

.macro emu_pop3
		movw	YL,SP_L
		adiw	YL,2
		movw	SP_L,YL
		andi	YH,STACKA
		subi	YH,0xfe
		ld	@1,-Y
		ld	@0,-Y
.endm

.macro emu_push3
		movw	YL,SP_L
		sbiw	YL,2
		movw	SP_L,YL
		andi	YH,STACKA
		subi	YH,0xfe
		st	Y+,@0
		st	Y+,@1
.endm

.macro emu_exch3
		movw	YL,SP_L
		andi	YH,STACKA
		subi	YH,0xfe
		ld	r0,Y+
		ld	r1,Y+
		st	-Y,@1
		st	-Y,@0
		movw	@0,r0
.endm

;------------------------------------------------------------------------------
;init registers and start
;------------------------------------------------------------------------------
emu_start:	ldi	PC_L,0
		ldi	PC_H,0
		movw	reg_f,PC_L
		movw	reg_c,PC_L
		movw	reg_e,PC_L
		movw	reg_l,PC_L
		movw	reg_ixl,PC_L
		movw	reg_iyl,PC_L
		movw	SP_L,PC_L
		sts	libmio_nmi,const_0
		out	GPIOR0,const_0		;N flag
		ldi	XL,LOW(z80emu_ram)
		ldi	XH,HIGH(z80emu_ram)
		ldi	ZL,32
emu_start_1:	st	X+,const_0
		dec	ZL
		brne	emu_start_1
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x01		LD BC,nnnn
;------------------------------------------------------------------------------
emu_code_01:	getpcword	reg_c,reg_b		;19 get word
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x02		LD (BC),A
;------------------------------------------------------------------------------
emu_code_02:	putmembyte	reg_a,reg_c,reg_b
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x03		INC BC
;------------------------------------------------------------------------------
emu_code_03:	z80_inc16	reg_c,reg_b
		rjmp	emu_loop		;2 end of command

;------------------------------------------------------------------------------
; 0x04		INC B
;------------------------------------------------------------------------------
emu_code_04:	z80_increg	reg_b
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x05		DEC B
;------------------------------------------------------------------------------
emu_code_05:	z80_decreg	reg_b
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x06		LD	B,nn
;------------------------------------------------------------------------------
emu_code_06:	getpcbyte	reg_b
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x07		RLC	A
;------------------------------------------------------------------------------
emu_code_07:	lsr	reg_f			;shift out carry
		mov	r16,reg_a
		lsl	r16
		rol	reg_a
		rol	reg_f
		andi	reg_f,flags_clrh	;1 clear H flag
		rjmp	emu_loopnc		;2 end of command

;------------------------------------------------------------------------------
; 0x08		EX 	AF,AF'
;------------------------------------------------------------------------------
emu_code_08:	movw	r0,reg_f		;2 save AF
		lds	reg_f,zreg_f1
		lds	reg_a,zreg_a1
		sts	zreg_f1,r0
		sts	zreg_a1,r1
		rjmp	emu_loop		;2 end of command

;------------------------------------------------------------------------------
; 0x09		ADD	HL,BC
;------------------------------------------------------------------------------
emu_code_09:	z80_add16	reg_c,reg_b
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x0a		LD	A,(BC)
;------------------------------------------------------------------------------
emu_code_0a:	getmembyte reg_a,reg_c,reg_b
		rjmp	emu_loop		;2

;------------------------------------------------------------------------------
; 0x0b		DEC	BC
;------------------------------------------------------------------------------
emu_code_0b:	z80_dec16	reg_c,reg_b
		rjmp	emu_loop		;2

;------------------------------------------------------------------------------
; 0x0c		INC C
;------------------------------------------------------------------------------
emu_code_0c:	z80_increg	reg_c
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x0d		DEC 	C
;------------------------------------------------------------------------------
emu_code_0d:	z80_decreg	reg_c
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x0e		LD	C,nn
;------------------------------------------------------------------------------
emu_code_0e:	getpcbyte	reg_c
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x0f		RRC	A
;------------------------------------------------------------------------------
emu_code_0f:	lsr	reg_f
		mov	r16,reg_a
		lsr	r16
		ror	reg_a
		rol	reg_f
		andi	reg_f,flags_clrh	;1 clear H flag
		rjmp	emu_loopnc		;2 end of command

;------------------------------------------------------------------------------
; 0x10		DJNZ	nn
;------------------------------------------------------------------------------
emu_code_10:	set
		sub	reg_b,const_1		;decrement B
		brne	emu_code_10a
		clt
emu_code_10a:	rjmp	emu_jrels		;jump if flag is set

;------------------------------------------------------------------------------
; 0x11		LD 	DE,nnnn
;------------------------------------------------------------------------------
emu_code_11:	getpcword	reg_e,reg_d		;19 get word
		z80_loop

;------------------------------------------------------------------------------
; 0x12		LD 	(DE),A
;------------------------------------------------------------------------------
emu_code_12:	putmembyte	reg_a,reg_e,reg_d
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x13		INC DE
;------------------------------------------------------------------------------
emu_code_13:	z80_inc16	reg_e,reg_d
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x14		INC D
;------------------------------------------------------------------------------
emu_code_14:	z80_increg	reg_d
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x15		DEC D
;------------------------------------------------------------------------------
emu_code_15:	z80_decreg	reg_d
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x16		LD	D,nn
;------------------------------------------------------------------------------
emu_code_16:	getpcbyte	reg_d
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x17		RL	A
;------------------------------------------------------------------------------
emu_code_17:	lsr	reg_f		;1 get C flag
		rol	reg_a
		rol	reg_f
		andi	reg_f,flags_clrh	;1 clear H flag
		rjmp	emu_loopnc		;2 end of command

;------------------------------------------------------------------------------
; 0x18		jr	nn
;------------------------------------------------------------------------------
emu_code_18:	set				;1 activate
		rjmp	emu_jrels		;2 jump relative

;------------------------------------------------------------------------------
; 0x19		ADD	HL,DE
;------------------------------------------------------------------------------
emu_code_19:	z80_add16	reg_e,reg_d
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x1a		LD	A,(DE)
;------------------------------------------------------------------------------
emu_code_1a:	getmembyte	reg_a,reg_e,reg_d
		rjmp	emu_loop		;2

;------------------------------------------------------------------------------
; 0x1b		DEC	DE
;------------------------------------------------------------------------------
emu_code_1b:	z80_dec16	reg_e,reg_d
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x1c		INC	E
;------------------------------------------------------------------------------
emu_code_1c:	z80_increg	reg_e
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x1d		DEC	E
;------------------------------------------------------------------------------
emu_code_1d:	z80_decreg	reg_e
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x1e		LD	E,nn
;------------------------------------------------------------------------------
emu_code_1e:	getpcbyte	reg_e
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x1f		RR	A
;------------------------------------------------------------------------------
emu_code_1f:	lsr	reg_f		;1 get C flag
		ror	reg_a
		rol	reg_f
		andi	reg_f,flags_clrh	;1 clear H flag
		rjmp	emu_loopnc		;2 end of command

;------------------------------------------------------------------------------
; 0x20		jrnz	nn
;------------------------------------------------------------------------------
emu_code_20:	bst	reg_f,aflag_z		;copy Z flag
		rjmp	emu_jrelc		;jump if T is cleared

;------------------------------------------------------------------------------
; 0x21		LD 	HL,nnnn
;------------------------------------------------------------------------------
emu_code_21:	getpcword	reg_l,reg_h
		z80_loop

;------------------------------------------------------------------------------
; 0x22		LD 	(nnnn),HL
;------------------------------------------------------------------------------
emu_code_22:	getpcword	XL,XH
		putmemword	reg_l,reg_h,XL,XH
		rjmp	emu_loop		;2 end of command

;------------------------------------------------------------------------------
; 0x23		INC	HL
;------------------------------------------------------------------------------
emu_code_23:	z80_inc16	reg_l,reg_h
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x24		INC	H
;------------------------------------------------------------------------------
emu_code_24:	z80_increg	reg_h
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x25		DEC	H
;------------------------------------------------------------------------------
emu_code_25:	z80_decreg	reg_h
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x26		LD	H,nn
;------------------------------------------------------------------------------
emu_code_26:	getpcbyte	reg_h
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x27		DA	A
;------------------------------------------------------------------------------
emu_code_27:	mov	r16,reg_a		;1 copy A
		andi	r16,0x0f		;1 mask bits
		sbic	GPIOR0,0		;1 skip if add was last
		rjmp	emu_code_27s		;2 last was sub
		cpi	r16,0x0a		;1 check if we must add
		sbrs	reg_f,aflag_h		;1 skip if H set
		brcs	emu_code_27a		;1/2
		subi	reg_a,0xfa		;1 +6
emu_code_27a:	mov	r16,reg_a		;1 copy A
		andi	r16,0xf0		;1 mask bits
		cpi	r16,0x0a		;1 check if we must add
		clr	r19			;1
		brcs	emu_code_27b		;1/2
		ldi	r19,0x60		;1 +60
emu_code_27b:	add	reg_a,r19
		brcc	emu_code_27c		;no carry
		ori	reg_f,1			;set carry
emu_code_27c:	rjmp	emu_code_27x		;2 end of command

emu_code_27s:	sbrc	reg_f,aflag_h		;1 skip if no H flag
		subi	reg_a,0x06		;1
		sbrc	reg_f,aflag_c		;1/2 skip if no carry
		subi	reg_a,0x60		;1
emu_code_27x:	mov	ZL,reg_a		;1
		ldi	ZH,HIGH(emu_flags_log*2);1
		lpm	reg_f,Z			;1
		rjmp	emu_loop		;2

;------------------------------------------------------------------------------
; 0x28		jrz	nn
;------------------------------------------------------------------------------
emu_code_28:	bst	reg_f,aflag_z		;copy Z flag
		rjmp	emu_jrels		;jump if T is set

;------------------------------------------------------------------------------
; 0x29		ADD	HL,HL
;------------------------------------------------------------------------------
emu_code_29:	z80_add16	reg_l,reg_h
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x2a		LD 	HL,(nnnn)
;------------------------------------------------------------------------------
emu_code_2a:	getpcword	XL,XH
		getmemword	reg_l,reg_h,XL,XH
		rjmp	emu_loop		;2 end of command

;------------------------------------------------------------------------------
; 0x2b		DEC	HL
;------------------------------------------------------------------------------
emu_code_2b:	z80_dec16	reg_l,reg_h
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x2c		INC	L
;------------------------------------------------------------------------------
emu_code_2c:	z80_increg	reg_l
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x2d		DEC	L
;------------------------------------------------------------------------------
emu_code_2d:	z80_decreg	reg_l
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x2e		LD	L,nn
;------------------------------------------------------------------------------
emu_code_2e:	getpcbyte	reg_l
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x2f		CPL
;------------------------------------------------------------------------------
emu_code_2f:	com	reg_a			;1 ones complement
		ori	reg_f,flags_seth	;1 set H flag
		z80_setnflag			;1 set N flag
		rjmp	emu_loop		;2 end of command

;------------------------------------------------------------------------------
; 0x30		jrnc	nn
;------------------------------------------------------------------------------
emu_code_30:	bst	reg_f,aflag_c		;copy C flag
		rjmp	emu_jrelc		;jump if T is cleared

;------------------------------------------------------------------------------
; 0x31		LD 	SP,nnnn
;------------------------------------------------------------------------------
emu_code_31:	getpcword	SP_L,SP_H
		z80_loop

;------------------------------------------------------------------------------
; 0x32		LD 	(nnnn),a
;------------------------------------------------------------------------------
emu_code_32:	getpcword	XL,XH
		putmembyte	reg_a,XL,XH
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x33		INC	SP
;------------------------------------------------------------------------------
emu_code_33:	z80_inc16	SP_L,SP_H
		rjmp	emu_loop		;2 end of command

;------------------------------------------------------------------------------
; 0x34		INC	(HL)
;------------------------------------------------------------------------------
emu_code_34:	getmembyte	r16,reg_l,reg_h
		z80_increg	r16
		rjmp	emu_storem

;------------------------------------------------------------------------------
; 0x35		DEC	(HL)
;------------------------------------------------------------------------------
emu_code_35:	getmembyte	r16,reg_l,reg_h
		z80_decreg	r16
		rjmp	emu_storem

;------------------------------------------------------------------------------
; 0x36		LD 	(HL),nn
;------------------------------------------------------------------------------
emu_code_36:	getpcbyte	r16
		rjmp	emu_storem

;------------------------------------------------------------------------------
; 0x37		SCF
;------------------------------------------------------------------------------
emu_code_37:	andi	reg_f,0xdf		;clear H
		z80_clrnflag			;clear N
		ori	reg_f,flags_setc	;set carry
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x38		jrc	nn
;------------------------------------------------------------------------------
emu_code_38:	bst	reg_f,aflag_c		;copy C flag
		rjmp	emu_jrels		;jump if T is set

;------------------------------------------------------------------------------
; 0x39		ADD	HL,SP
;------------------------------------------------------------------------------
emu_code_39:	z80_add16	SP_L,SP_H
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x3a		LD	A,(nnnn)
;------------------------------------------------------------------------------
emu_code_3a:	getpcword	XL,XH
		getmembyte	reg_a,XL,XH
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x3b		DEC	SP
;------------------------------------------------------------------------------
emu_code_3b:	z80_dec16	SP_L,SP_H
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x3c		INC	A
;------------------------------------------------------------------------------
emu_code_3c:	z80_increg	reg_a
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x3d		DEC	A
;------------------------------------------------------------------------------
emu_code_3d:	z80_decreg	reg_a
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x3e		LD	A,nn
;------------------------------------------------------------------------------
emu_code_3e:	getpcbyte	reg_a
		rjmp	emu_loop		;2 end of command (set flags)

;------------------------------------------------------------------------------
; 0x3f		CCF
;------------------------------------------------------------------------------
emu_code_3f:	bst	reg_f,aflag_c		;get C
		bld	reg_f,aflag_h		;copy to H
		eor	reg_f,const_1		;toggle C flag
		rjmp	emu_loopnc		;goto loop

;------------------------------------------------------------------------------
; 0x40		LD	B,B
;------------------------------------------------------------------------------
emu_code_40:	rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x41		LD	B,C
;------------------------------------------------------------------------------
emu_code_41:	mov	reg_b,reg_c
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x42		LD	B,D
;------------------------------------------------------------------------------
emu_code_42:	mov	reg_b,reg_d
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x43		LD	B,E
;------------------------------------------------------------------------------
emu_code_43:	mov	reg_b,reg_e
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x44		LD	B,H
;------------------------------------------------------------------------------
emu_code_44:	mov	reg_b,reg_h
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x45		LD	B,L
;------------------------------------------------------------------------------
emu_code_45:	mov	reg_b,reg_l
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x46		LD	B,(HL)
;------------------------------------------------------------------------------
emu_code_46:
.if hl_mode == 1
		getmembyte	reg_b,reg_l,reg_h
		rjmp	emu_loop		;goto loop
.endif
.if hl_mode == 2
		rcall	emu_mread
		mov	reg_b,r16
		rjmp	emu_loop		;goto loop
.endif

;------------------------------------------------------------------------------
; 0x47		LD	B,A
;------------------------------------------------------------------------------
emu_code_47:	mov	reg_b,reg_a
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x48		LD	C,B
;------------------------------------------------------------------------------
emu_code_48:	mov	reg_c,reg_b
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x49		LD	C,C
;------------------------------------------------------------------------------
emu_code_49:	rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x4a		LD	C,D
;------------------------------------------------------------------------------
emu_code_4a:	mov	reg_c,reg_d
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x4b		LD	C,E
;------------------------------------------------------------------------------
emu_code_4b:	mov	reg_c,reg_e
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x4c		LD	C,H
;------------------------------------------------------------------------------
emu_code_4c:	mov	reg_c,reg_h
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x4d		LD	C,L
;------------------------------------------------------------------------------
emu_code_4d:	mov	reg_c,reg_l
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x4e		LD	C,(HL)
;------------------------------------------------------------------------------
emu_code_4e:
.if hl_mode == 1
		getmembyte	reg_c,reg_l,reg_h
		rjmp	emu_loop		;goto loop
.endif
.if hl_mode == 2
		rcall	emu_mread
		mov	reg_c,r16
		rjmp	emu_loop		;goto loop
.endif

;------------------------------------------------------------------------------
; 0x4f		LD	C,A
;------------------------------------------------------------------------------
emu_code_4f:	mov	reg_c,reg_a
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x50		LD	D,B
;------------------------------------------------------------------------------
emu_code_50:	mov	reg_d,reg_b
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x51		LD	D,C
;------------------------------------------------------------------------------
emu_code_51:	mov	reg_d,reg_c
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x52		LD	D,D
;------------------------------------------------------------------------------
emu_code_52:	rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x53		LD	D,E
;------------------------------------------------------------------------------
emu_code_53:	mov	reg_d,reg_e
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x54		LD	D,H
;------------------------------------------------------------------------------
emu_code_54:	mov	reg_d,reg_h
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x55		LD	D,L
;------------------------------------------------------------------------------
emu_code_55:	mov	reg_d,reg_l
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x56		LD	D,(HL)
;------------------------------------------------------------------------------
emu_code_56:
.if hl_mode == 1
		getmembyte	reg_d,reg_l,reg_h
		rjmp	emu_loop		;goto loop
.endif
.if hl_mode == 2
		rcall	emu_mread
		mov	reg_d,r16
		rjmp	emu_loop		;goto loop
.endif

;------------------------------------------------------------------------------
; 0x57		LD	D,A
;------------------------------------------------------------------------------
emu_code_57:	mov	reg_d,reg_a
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x58		LD	E,B
;------------------------------------------------------------------------------
emu_code_58:	mov	reg_e,reg_b
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x59		LD	E,C
;------------------------------------------------------------------------------
emu_code_59:	mov	reg_e,reg_c
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x5a		LD	E,D
;------------------------------------------------------------------------------
emu_code_5a:	mov	reg_e,reg_d
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x5b		LD	E,E
;------------------------------------------------------------------------------
emu_code_5b:	rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x5c		LD	E,H
;------------------------------------------------------------------------------
emu_code_5c:	mov	reg_e,reg_h
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x5d		LD	E,L
;------------------------------------------------------------------------------
emu_code_5d:	mov	reg_e,reg_l
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x5e		LD	E,(HL)
;------------------------------------------------------------------------------
emu_code_5e:
.if hl_mode == 1
		getmembyte	reg_e,reg_l,reg_h
		rjmp	emu_loop		;goto loop
.endif
.if hl_mode == 2
		rcall	emu_mread
		mov	reg_e,r16
		rjmp	emu_loop		;goto loop
.endif

;------------------------------------------------------------------------------
; 0x5f		LD	E,A
;------------------------------------------------------------------------------
emu_code_5f:	mov	reg_e,reg_a
		rjmp	emu_loop		;goto loop


;------------------------------------------------------------------------------
; 0x60		LD	H,B
;------------------------------------------------------------------------------
emu_code_60:	mov	reg_h,reg_b
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x61		LD	H,C
;------------------------------------------------------------------------------
emu_code_61:	mov	reg_h,reg_c
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x62		LD	H,D
;------------------------------------------------------------------------------
emu_code_62:	mov	reg_h,reg_d
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x63		LD	H,E
;------------------------------------------------------------------------------
emu_code_63:	mov	reg_h,reg_e
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x64		LD	H,H
;------------------------------------------------------------------------------
emu_code_64:	rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x65		LD	H,L
;------------------------------------------------------------------------------
emu_code_65:	mov	reg_h,reg_l
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x66		LD	H,(HL)
;------------------------------------------------------------------------------
emu_code_66:
.if hl_mode == 1
		getmembyte	reg_h,reg_l,reg_h
		rjmp	emu_loop		;goto loop
.endif
.if hl_mode == 2
		rcall	emu_mread
		mov	reg_h,r16
		rjmp	emu_loop		;goto loop
.endif

;------------------------------------------------------------------------------
; 0x67		LD	H,A
;------------------------------------------------------------------------------
emu_code_67:	mov	reg_h,reg_a
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x68		LD	L,B
;------------------------------------------------------------------------------
emu_code_68:	mov	reg_l,reg_b
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x69		LD	L,C
;------------------------------------------------------------------------------
emu_code_69:	mov	reg_l,reg_c
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x6a		LD	L,D
;------------------------------------------------------------------------------
emu_code_6a:	mov	reg_l,reg_d
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x6b		LD	L,E
;------------------------------------------------------------------------------
emu_code_6b:	mov	reg_l,reg_e
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0x6c		LD	L,H
;------------------------------------------------------------------------------
emu_code_6c:	mov	reg_l,reg_h
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x6d		LD	L,L
;------------------------------------------------------------------------------
emu_code_6d:	rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x6e		LD	L,(HL)
;------------------------------------------------------------------------------
emu_code_6e:
.if hl_mode == 1
		getmembyte	reg_l,reg_l,reg_h
		rjmp	emu_loop		;goto loop
.endif
.if hl_mode == 2
		rcall	emu_mread
		mov	reg_l,r16
		rjmp	emu_loop		;goto loop
.endif

;------------------------------------------------------------------------------
; 0x6f		LD	L,A
;------------------------------------------------------------------------------
emu_code_6f:	mov	reg_l,reg_a
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x70		LD	(HL),B
;------------------------------------------------------------------------------
emu_code_70:	mov	r16,reg_b
		rjmp	emu_storem
;------------------------------------------------------------------------------
; 0x71		LD	(HL),C
;------------------------------------------------------------------------------
emu_code_71:	mov	r16,reg_c
		rjmp	emu_storem

;------------------------------------------------------------------------------
; 0x72		LD	(HL),D
;------------------------------------------------------------------------------
emu_code_72:	mov	r16,reg_d
		rjmp	emu_storem

;------------------------------------------------------------------------------
; 0x73		LD	(HL),E
;------------------------------------------------------------------------------
emu_code_73:	mov	r16,reg_e
		rjmp	emu_storem

;------------------------------------------------------------------------------
; 0x74		LD	(HL),H
;------------------------------------------------------------------------------
emu_code_74:	mov	r16,reg_h
		rjmp	emu_storem

;------------------------------------------------------------------------------
; 0x75		LD	(HL),L
;------------------------------------------------------------------------------
emu_code_75:	mov	r16,reg_l
		rjmp	emu_storem

;------------------------------------------------------------------------------
; 0x76		HALT
;------------------------------------------------------------------------------
emu_code_76:;	sts	z80_break,const_1	;set break flag
		rjmp	emu_loop		;goto loop (ignore code)

;------------------------------------------------------------------------------
; 0x77		LD	(HL),A
;------------------------------------------------------------------------------
emu_code_77:	mov	r16,reg_a
		rjmp	emu_storem

;------------------------------------------------------------------------------
; 0x78		LD	A,B
;------------------------------------------------------------------------------
emu_code_78:	mov	reg_a,reg_b
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x79		LD	A,C
;------------------------------------------------------------------------------
emu_code_79:	mov	reg_a,reg_c
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x7a		LD	A,D
;------------------------------------------------------------------------------
emu_code_7a:	mov	reg_a,reg_d
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x7b		LD	A,E
;------------------------------------------------------------------------------
emu_code_7b:	mov	reg_a,reg_e
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x7c		LD	A,H
;------------------------------------------------------------------------------
emu_code_7c:	mov	reg_a,reg_h
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x7d		LD	A,L
;------------------------------------------------------------------------------
emu_code_7d:	mov	reg_a,reg_l
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x7e		LD	A,(HL)
;------------------------------------------------------------------------------
emu_code_7e:
.if hl_mode == 1
		getmembyte	reg_a,reg_l,reg_h
		rjmp	emu_loop		;goto loop
.endif
.if hl_mode == 2
		rcall	emu_mread
		mov	reg_a,r16
		rjmp	emu_loop		;goto loop
.endif

;------------------------------------------------------------------------------
; 0x7f		LD	A,A
;------------------------------------------------------------------------------
emu_code_7f:	rjmp	emu_loop		;goto loop



;-------------------------------------------------------------------------------
; clock cycles:
; RAM:		17
; ROM:		18
;-------------------------------------------------------------------------------
emu_storem:
.if hl_mode == 1
		putmembyte	r16,reg_l,reg_h
		rjmp	emu_loop		;goto loop
.endif
.if hl_mode == 2
		rjmp	emu_mwrite
.endif

emu_loop:
		getpcbyte_m1 ZL			;17/18
		ldi	ZH,HIGH(emu_jtab)	;1
		ijmp				;2

emu_loopa:
		in	reg_f,SREG		;get status
emu_loopnc:	z80_clrnflag			;clear N flag
		getpcbyte_m1 ZL			;12/13
		ldi	ZH,HIGH(emu_jtab)	;1
		ijmp				;2

emu_loops:
		in	reg_f,SREG		;get status
		z80_setnflag			;set N flag
		getpcbyte_m1 ZL			;12/13
		ldi	ZH,HIGH(emu_jtab)	;1
		ijmp				;2


;------------------------------------------------------------------------------
; jump relative if T is set
;
;------------------------------------------------------------------------------
emu_jrels:	getpcbyte ZL			;get next PC byte
		brtc	emu_jrels_1		;branch if no jump
		subi	ZL,0x80			;1
		subi	PC_L,0x80		;1
		sbc	PC_H,const_0		;1
		add	PC_L,ZL			;1
		adc	PC_H,const_0		;1
emu_jrels_1:
		getpcbyte_m1 ZL			;12/13
		ldi	ZH,HIGH(emu_jtab)	;1
		ijmp				;2

;------------------------------------------------------------------------------
; jump relative if T is not set
;
;------------------------------------------------------------------------------
emu_jrelc:	getpcbyte ZL			;get next PC byte
		brts	emu_jrelc_1		;branch if no jump
		subi	ZL,0x80			;1
		subi	PC_L,0x80		;1
		sbc	PC_H,const_0		;1
		add	PC_L,ZL			;1
		adc	PC_H,const_0		;1
emu_jrelc_1:
		getpcbyte_m1 ZL			;12/13
		ldi	ZH,HIGH(emu_jtab)	;1
		ijmp				;2

;------------------------------------------------------------------------------
; jump absolute if T is set
;------------------------------------------------------------------------------
emu_jmps:	getpcword XL,XH			;get jump destination
		brtc	emu_jmps_1
		movw	PC_L,XL
emu_jmps_1:
		getpcbyte_m1 ZL			;12/13 get next PC byte
		ldi	ZH,HIGH(emu_jtab)	;1
		ijmp				;2

;------------------------------------------------------------------------------
; jump absolute if T is cleared
;------------------------------------------------------------------------------
emu_jmpc:	getpcword XL,XH			;get jump destination
		brts	emu_jmpc_1
		movw	PC_L,XL
emu_jmpc_1:
		getpcbyte_m1 ZL			;12/13 get next PC byte
		ldi	ZH,HIGH(emu_jtab)	;1
		ijmp				;2

;------------------------------------------------------------------------------
; call if T is set
;------------------------------------------------------------------------------
emu_calls:	getpcword r0,r1			;get next two PC bytes
		brtc	emu_calls_1		;branch if no jump
.if stack_mode == 3
		emu_push3	PC_L,PC_H
.else
		movw	XL,SP_L			;2 get stack pointer
		sbiw	XL,2			;2 decrement SP
		movw	SP_L,XL			;2 store back SP
		putmembyte PC_L,XL,XH			;store PCL to stack
		adiw	XL,1
		putmembyte PC_H,XL,XH			;store PCH to stack
.endif
		movw	PC_L,r0

emu_calls_1:
		getpcbyte_m1 ZL			;12/13
		ldi	ZH,HIGH(emu_jtab)	;1
		ijmp				;2

;------------------------------------------------------------------------------
; call if T is cleared
;------------------------------------------------------------------------------
emu_callc:	getpcword r0,r1			;get next two PC bytes
		brts	emu_callc_1		;branch if no jump
.if stack_mode == 3
		emu_push3	PC_L,PC_H
.else
		movw	XL,SP_L			;2 get stack pointer
		sbiw	XL,2			;2 decrement SP
		movw	SP_L,XL			;2 store back SP
		putmembyte PC_L,XL,XH			;store PCL to stack
		adiw	XL,1
		putmembyte PC_H,XL,XH			;store PCH to stack
.endif
		movw	PC_L,r0

emu_callc_1:
		getpcbyte_m1 ZL			;12/13
		ldi	ZH,HIGH(emu_jtab)	;1
		ijmp				;2

;------------------------------------------------------------------------------
;RST
;------------------------------------------------------------------------------
emu_rst:	mov	r0,r16			;address low
		mov	r1,const_0
.if stack_mode == 3
		emu_push3	PC_L,PC_H
.else
		movw	XL,SP_L			;2 get stack pointer
		sbiw	XL,2			;2 decrement SP
		movw	SP_L,XL			;2 store back SP
		putmembyte PC_L,XL,XH			;store PCL to stack
		adiw	XL,1
		putmembyte PC_H,XL,XH			;store PCH to stack
.endif
		movw	PC_L,r0
		getpcbyte_m1 ZL			;12/13
		ldi	ZH,HIGH(emu_jtab)	;1
		ijmp				;2

;------------------------------------------------------------------------------
; ret if T is set
;------------------------------------------------------------------------------
emu_rets:	brtc	emu_rets_1		;branch if no jump
.if stack_mode == 3
		emu_pop3	PC_L,PC_H
.else
		movw	XL,SP_L			;get SP
		getmembyte PC_L,XL,XH
		adiw	XL,1			;2
		getmembyte PC_H,XL,XH
		adiw	XL,1			;2
		movw	SP_L,XL			;2 store back SP 
.endif
emu_rets_1:
		getpcbyte_m1 ZL			;12/13
		ldi	ZH,HIGH(emu_jtab)	;1
		ijmp				;2

;------------------------------------------------------------------------------
; ret if T is cleared
;------------------------------------------------------------------------------
emu_retc:	brts	emu_retc_1		;branch if no jump
.if stack_mode == 3
		emu_pop3	PC_L,PC_H
.else
		movw	XL,SP_L			;get SP
		getmembyte PC_L,XL,XH
		adiw	XL,1			;2
		getmembyte PC_H,XL,XH
		adiw	XL,1			;2
		movw	SP_L,XL			;2 store back SP 
.endif
emu_retc_1:
		getpcbyte_m1 ZL			;12/13
		ldi	ZH,HIGH(emu_jtab)	;1
		ijmp				;2


;------------------------------------------------------------------------------
; set flags after logical operations
;------------------------------------------------------------------------------
emu_and:	mov	ZL,reg_a
		ldi	ZH,HIGH(emu_flags_log*2);1
		lpm	reg_f,Z			;3
		z80_clrnflag			;1 clear N flag
		getpcbyte_m1 ZL			;12/13
		ldi	ZH,HIGH(emu_jtab)	;1
		ijmp				;2

emu_or:		mov	ZL,reg_a
		ldi	ZH,HIGH(emu_flags_log*2);1
		lpm	reg_f,Z			;3
		z80_clrnflag			;1 clear N flag
		andi	reg_f,flags_clrh	;1 clear H flag
		getpcbyte_m1 ZL			;12/13
		ldi	ZH,HIGH(emu_jtab)	;1
		ijmp				;2


; the level1 jump table
;------------------------------------------------------------------------------
.org (pc+255) & 0xff00

emu_jtab:	rjmp	emu_loop		;2	0x00	NOP
		rjmp	emu_code_01
		rjmp	emu_code_02
		rjmp	emu_code_03
		rjmp	emu_code_04
		rjmp	emu_code_05
		rjmp	emu_code_06
		rjmp	emu_code_07
		rjmp	emu_code_08
		rjmp	emu_code_09
		rjmp	emu_code_0a
		rjmp	emu_code_0b
		rjmp	emu_code_0c
		rjmp	emu_code_0d
		rjmp	emu_code_0e
		rjmp	emu_code_0f

		rjmp	emu_code_10
		rjmp	emu_code_11
		rjmp	emu_code_12
		rjmp	emu_code_13
		rjmp	emu_code_14
		rjmp	emu_code_15
		rjmp	emu_code_16
		rjmp	emu_code_17
		rjmp	emu_code_18
		rjmp	emu_code_19
		rjmp	emu_code_1a
		rjmp	emu_code_1b
		rjmp	emu_code_1c
		rjmp	emu_code_1d
		rjmp	emu_code_1e
		rjmp	emu_code_1f

		rjmp	emu_code_20
		rjmp	emu_code_21
		rjmp	emu_code_22
		rjmp	emu_code_23
		rjmp	emu_code_24
		rjmp	emu_code_25
		rjmp	emu_code_26
		rjmp	emu_code_27
		rjmp	emu_code_28
		rjmp	emu_code_29
		rjmp	emu_code_2a
		rjmp	emu_code_2b
		rjmp	emu_code_2c
		rjmp	emu_code_2d
		rjmp	emu_code_2e
		rjmp	emu_code_2f

		rjmp	emu_code_30
		rjmp	emu_code_31
		rjmp	emu_code_32
		rjmp	emu_code_33
		rjmp	emu_code_34
		rjmp	emu_code_35
		rjmp	emu_code_36
		rjmp	emu_code_37
		rjmp	emu_code_38
		rjmp	emu_code_39
		rjmp	emu_code_3a
		rjmp	emu_code_3b
		rjmp	emu_code_3c
		rjmp	emu_code_3d
		rjmp	emu_code_3e
		rjmp	emu_code_3f

		rjmp	emu_code_40
		rjmp	emu_code_41
		rjmp	emu_code_42
		rjmp	emu_code_43
		rjmp	emu_code_44
		rjmp	emu_code_45
		rjmp	emu_code_46
		rjmp	emu_code_47
		rjmp	emu_code_48
		rjmp	emu_code_49
		rjmp	emu_code_4a
		rjmp	emu_code_4b
		rjmp	emu_code_4c
		rjmp	emu_code_4d
		rjmp	emu_code_4e
		rjmp	emu_code_4f

		rjmp	emu_code_50
		rjmp	emu_code_51
		rjmp	emu_code_52
		rjmp	emu_code_53
		rjmp	emu_code_54
		rjmp	emu_code_55
		rjmp	emu_code_56
		rjmp	emu_code_57
		rjmp	emu_code_58
		rjmp	emu_code_59
		rjmp	emu_code_5a
		rjmp	emu_code_5b
		rjmp	emu_code_5c
		rjmp	emu_code_5d
		rjmp	emu_code_5e
		rjmp	emu_code_5f

		rjmp	emu_code_60
		rjmp	emu_code_61
		rjmp	emu_code_62
		rjmp	emu_code_63
		rjmp	emu_code_64
		rjmp	emu_code_65
		rjmp	emu_code_66
		rjmp	emu_code_67
		rjmp	emu_code_68
		rjmp	emu_code_69
		rjmp	emu_code_6a
		rjmp	emu_code_6b
		rjmp	emu_code_6c
		rjmp	emu_code_6d
		rjmp	emu_code_6e
		rjmp	emu_code_6f

		rjmp	emu_code_70
		rjmp	emu_code_71
		rjmp	emu_code_72
		rjmp	emu_code_73
		rjmp	emu_code_74
		rjmp	emu_code_75
		rjmp	emu_code_76
		rjmp	emu_code_77
		rjmp	emu_code_78
		rjmp	emu_code_79
		rjmp	emu_code_7a
		rjmp	emu_code_7b
		rjmp	emu_code_7c
		rjmp	emu_code_7d
		rjmp	emu_code_7e
		rjmp	emu_code_7f

		rjmp	emu_code_80
		rjmp	emu_code_81
		rjmp	emu_code_82
		rjmp	emu_code_83
		rjmp	emu_code_84
		rjmp	emu_code_85
		rjmp	emu_code_86
		rjmp	emu_code_87
		rjmp	emu_code_88
		rjmp	emu_code_89
		rjmp	emu_code_8a
		rjmp	emu_code_8b
		rjmp	emu_code_8c
		rjmp	emu_code_8d
		rjmp	emu_code_8e
		rjmp	emu_code_8f

		rjmp	emu_code_90
		rjmp	emu_code_91
		rjmp	emu_code_92
		rjmp	emu_code_93
		rjmp	emu_code_94
		rjmp	emu_code_95
		rjmp	emu_code_96
		rjmp	emu_code_97
		rjmp	emu_code_98
		rjmp	emu_code_99
		rjmp	emu_code_9a
		rjmp	emu_code_9b
		rjmp	emu_code_9c
		rjmp	emu_code_9d
		rjmp	emu_code_9e
		rjmp	emu_code_9f

		rjmp	emu_code_a0
		rjmp	emu_code_a1
		rjmp	emu_code_a2
		rjmp	emu_code_a3
		rjmp	emu_code_a4
		rjmp	emu_code_a5
		rjmp	emu_code_a6
		rjmp	emu_code_a7
		rjmp	emu_code_a8
		rjmp	emu_code_a9
		rjmp	emu_code_aa
		rjmp	emu_code_ab
		rjmp	emu_code_ac
		rjmp	emu_code_ad
		rjmp	emu_code_ae
		rjmp	emu_code_af

		rjmp	emu_code_b0
		rjmp	emu_code_b1
		rjmp	emu_code_b2
		rjmp	emu_code_b3
		rjmp	emu_code_b4
		rjmp	emu_code_b5
		rjmp	emu_code_b6
		rjmp	emu_code_b7
		rjmp	emu_code_b8
		rjmp	emu_code_b9
		rjmp	emu_code_ba
		rjmp	emu_code_bb
		rjmp	emu_code_bc
		rjmp	emu_code_bd
		rjmp	emu_code_be
		rjmp	emu_code_bf

		rjmp	emu_code_c0
		rjmp	emu_code_c1
		rjmp	emu_code_c2
		rjmp	emu_code_c3
		rjmp	emu_code_c4
		rjmp	emu_code_c5
		rjmp	emu_code_c6
		rjmp	emu_code_c7
		rjmp	emu_code_c8
		rjmp	emu_code_c9
		rjmp	emu_code_ca
		rjmp	emu_code_cb
		rjmp	emu_code_cc
		rjmp	emu_code_cd
		rjmp	emu_code_ce
		rjmp	emu_code_cf

		rjmp	emu_code_d0
		rjmp	emu_code_d1
		rjmp	emu_code_d2
		rjmp	emu_code_d3
		rjmp	emu_code_d4
		rjmp	emu_code_d5
		rjmp	emu_code_d6
		rjmp	emu_code_d7
		rjmp	emu_code_d8
		rjmp	emu_code_d9
		rjmp	emu_code_da
		rjmp	emu_code_db
		rjmp	emu_code_dc
		rjmp	emu_code_dd
		rjmp	emu_code_de
		rjmp	emu_code_df

		rjmp	emu_code_e0
		rjmp	emu_code_e1
		rjmp	emu_code_e2
		rjmp	emu_code_e3
		rjmp	emu_code_e4
		rjmp	emu_code_e5
		rjmp	emu_code_e6
		rjmp	emu_code_e7
		rjmp	emu_code_e8
		rjmp	emu_code_e9
		rjmp	emu_code_ea
		rjmp	emu_code_eb
		rjmp	emu_code_ec
		rjmp	emu_code_ed
		rjmp	emu_code_ee
		rjmp	emu_code_ef

		rjmp	emu_code_f0
		rjmp	emu_code_f1
		rjmp	emu_code_f2
		rjmp	emu_code_f3
		rjmp	emu_code_f4
		rjmp	emu_code_f5
		rjmp	emu_code_f6
		rjmp	emu_code_f7
		rjmp	emu_code_f8
		rjmp	emu_code_f9
		rjmp	emu_code_fa
		rjmp	emu_code_fb
		rjmp	emu_code_fc
		rjmp	emu_code_fd
		rjmp	emu_code_fe
		rjmp	emu_code_ff


;------------------------------------------------------------------------------
; 0x80		ADD	B
;------------------------------------------------------------------------------
emu_code_80:	reg_add	reg_b
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0x81		ADD	C
;------------------------------------------------------------------------------
emu_code_81:	reg_add	reg_c
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0x82		ADD	D
;------------------------------------------------------------------------------
emu_code_82:	reg_add	reg_d
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0x83		ADD	E
;------------------------------------------------------------------------------
emu_code_83:	reg_add	reg_e
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0x84		ADD	H
;------------------------------------------------------------------------------
emu_code_84:	reg_add	reg_h
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0x85		ADD	L
;------------------------------------------------------------------------------
emu_code_85:	reg_add	reg_l
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0x86		ADD	(HL)
;------------------------------------------------------------------------------
emu_code_86:	getmembyte r16,reg_l,reg_h
		reg_add	r16
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0x87		ADD	A
;------------------------------------------------------------------------------
emu_code_87:	reg_add	reg_a
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0x88		ADC	B
;------------------------------------------------------------------------------
emu_code_88:	reg_adc	reg_b
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0x89		ADC	C
;------------------------------------------------------------------------------
emu_code_89:	reg_adc	reg_c
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0x8a		ADC	D
;------------------------------------------------------------------------------
emu_code_8a:	reg_adc	reg_d
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0x8b		ADC	E
;------------------------------------------------------------------------------
emu_code_8b:	reg_adc	reg_e
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0x8c		ADC	H
;------------------------------------------------------------------------------
emu_code_8c:	reg_adc	reg_h
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0x8d		ADC	L
;------------------------------------------------------------------------------
emu_code_8d:	reg_adc	reg_l
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0x8e		ADC	(HL)
;------------------------------------------------------------------------------
emu_code_8e:	getmembyte r16,reg_l,reg_h
		reg_adc	r16
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0x8f		ADC	A
;------------------------------------------------------------------------------
emu_code_8f:	reg_adc	reg_a
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0x90		SUB	B
;------------------------------------------------------------------------------
emu_code_90:	reg_sub	reg_b
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0x91		SUB	C
;------------------------------------------------------------------------------
emu_code_91:	reg_sub	reg_c			;sub C
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0x92		SUB	D
;------------------------------------------------------------------------------
emu_code_92:	reg_sub	reg_d			;sub D
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0x93		SUB	E
;------------------------------------------------------------------------------
emu_code_93:	reg_sub	reg_e
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0x94		SUB	H
;------------------------------------------------------------------------------
emu_code_94:	reg_sub	reg_h			;sub H
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0x95		SUB	L
;------------------------------------------------------------------------------
emu_code_95:	reg_sub	reg_l			;sub L
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0x96		SUB	(HL)
;------------------------------------------------------------------------------
emu_code_96:	getmembyte r16,reg_l,reg_h
		reg_sub r16
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0x97		SUB	A
;------------------------------------------------------------------------------
emu_code_97:	reg_sub	reg_a
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0x98		SBC	B
;------------------------------------------------------------------------------
emu_code_98:	reg_sbc	reg_b
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0x99		SBC	C
;------------------------------------------------------------------------------
emu_code_99:	reg_sbc	reg_c
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0x9a		SBC	D
;------------------------------------------------------------------------------
emu_code_9a:	reg_sbc	reg_d
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0x9b		SBC	E
;------------------------------------------------------------------------------
emu_code_9b:	reg_sbc	reg_e
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0x9c		SBC	H
;------------------------------------------------------------------------------
emu_code_9c:	reg_sbc	reg_h
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0x9d		SBC	L
;------------------------------------------------------------------------------
emu_code_9d:	reg_sbc	reg_l
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0x9e		SBC	(HL)
;------------------------------------------------------------------------------
emu_code_9e:	getmembyte r16,reg_l,reg_h
		reg_sbc	r16
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0x9f		SBC	A
;------------------------------------------------------------------------------
emu_code_9f:	reg_sbc	reg_a
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0xa0		AND	B
;------------------------------------------------------------------------------
emu_code_a0:	and	reg_a,reg_b
		rjmp	emu_and

;------------------------------------------------------------------------------
; 0xa1		AND	C
;------------------------------------------------------------------------------
emu_code_a1:	and	reg_a,reg_c
		rjmp	emu_and

;------------------------------------------------------------------------------
; 0xa2		AND	D
;------------------------------------------------------------------------------
emu_code_a2:	and	reg_a,reg_d
		rjmp	emu_and

;------------------------------------------------------------------------------
; 0xa3		AND	E
;------------------------------------------------------------------------------
emu_code_a3:	and	reg_a,reg_e
		rjmp	emu_and

;------------------------------------------------------------------------------
; 0xa4		AND	H
;------------------------------------------------------------------------------
emu_code_a4:	and	reg_a,reg_h
		rjmp	emu_and

;------------------------------------------------------------------------------
; 0xa5		AND	L
;------------------------------------------------------------------------------
emu_code_a5:	and	reg_a,reg_l
		rjmp	emu_and

;------------------------------------------------------------------------------
; 0xa6		AND	(HL)
;------------------------------------------------------------------------------
emu_code_a6:	getmembyte r16,reg_l,reg_h
		and	reg_a,r16
		rjmp	emu_and

;------------------------------------------------------------------------------
; 0xa7		AND	A
;------------------------------------------------------------------------------
emu_code_a7:	and	reg_a,reg_a
		rjmp	emu_and

;------------------------------------------------------------------------------
; 0xa8		XOR	B
;------------------------------------------------------------------------------
emu_code_a8:	eor	reg_a,reg_b
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xa9		XOR	C
;------------------------------------------------------------------------------
emu_code_a9:	eor	reg_a,reg_c
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xaa		XOR	D
;------------------------------------------------------------------------------
emu_code_aa:	eor	reg_a,reg_d
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xab		XOR	E
;------------------------------------------------------------------------------
emu_code_ab:	eor	reg_a,reg_e
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xac		XOR	H
;------------------------------------------------------------------------------
emu_code_ac:	eor	reg_a,reg_h
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xad		XOR	L
;------------------------------------------------------------------------------
emu_code_ad:	eor	reg_a,reg_l
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xae		XOR	(HL)
;------------------------------------------------------------------------------
emu_code_ae:	getmembyte r16,reg_l,reg_h
		eor	reg_a,r16
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xaf		XOR	A
;------------------------------------------------------------------------------
emu_code_af:	eor	reg_a,reg_a			;get A
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xb0		OR	B
;------------------------------------------------------------------------------
emu_code_b0:	or	reg_a,reg_b
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xb1		OR	C
;------------------------------------------------------------------------------
emu_code_b1:	or	reg_a,reg_c
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xb2		OR	D
;------------------------------------------------------------------------------
emu_code_b2:	or	reg_a,reg_d
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xb3		OR	E
;------------------------------------------------------------------------------
emu_code_b3:	or	reg_a,reg_e
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xb4		OR	H
;------------------------------------------------------------------------------
emu_code_b4:	or	reg_a,reg_h
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xb5		OR	L
;------------------------------------------------------------------------------
emu_code_b5:	or	reg_a,reg_l
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xb6		OR	(HL)
;------------------------------------------------------------------------------
emu_code_b6:	getmembyte r16,reg_l,reg_h
		or	reg_a,r16
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xb7		OR	A
;------------------------------------------------------------------------------
emu_code_b7:	or	reg_a,reg_a
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xb8		CMP	B
;------------------------------------------------------------------------------
emu_code_b8:	cp	reg_a,reg_b		;sub A
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0xb9		CMP	C
;------------------------------------------------------------------------------
emu_code_b9:	cp	reg_a,reg_c		;sub A
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0xba		CMP	D
;------------------------------------------------------------------------------
emu_code_ba:	cp	reg_a,reg_d		;sub A
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0xbb		CMP	E
;------------------------------------------------------------------------------
emu_code_bb:	cp	reg_a,reg_e		;sub A
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0xbc		CMP	H
;------------------------------------------------------------------------------
emu_code_bc:	cp	reg_a,reg_h		;sub A
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0xbd		CMP	L
;------------------------------------------------------------------------------
emu_code_bd:	cp	reg_a,reg_l		;sub A
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0xbe		CMP	(HL)
;------------------------------------------------------------------------------
emu_code_be:	getmembyte r16,reg_l,reg_h
		cp	reg_a,r16		;sbc M
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0xbf		CMP	A
;------------------------------------------------------------------------------
emu_code_bf:	cp	reg_a,reg_a		;sub A
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0xc0		RNZ
;------------------------------------------------------------------------------
emu_code_c0:	bst	reg_f,aflag_z		;copy Z flag
		rjmp	emu_retc		;return if not set

;------------------------------------------------------------------------------
; 0xc1		POP	BC
;------------------------------------------------------------------------------
emu_code_c1:
.if stack_mode == 1 
		z80_pop		reg_c,reg_b
		rjmp	emu_loop
.endif
.if stack_mode == 2
		ldi	YL,8			;pointer to reg_c
		rjmp	emu_pop
.endif
.if stack_mode == 3
		emu_pop3	reg_c,reg_b
		rjmp	emu_loop
.endif


;------------------------------------------------------------------------------
; 0xc2		JPNZ	nnnn
;------------------------------------------------------------------------------
emu_code_c2:	bst	reg_f,aflag_z			;copy Z flag
		rjmp	emu_jmpc

;------------------------------------------------------------------------------
; 0xc3		JMP	nnnn
;------------------------------------------------------------------------------
emu_code_c3:	getpcword r16,r17
		movw	PC_L,r16
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0xc4		CANZ	nnnn
;------------------------------------------------------------------------------
emu_code_c4:	bst	reg_f,aflag_z		;copy Z flag
		rjmp	emu_callc		;call if not set

;------------------------------------------------------------------------------
; 0xc5		PUSH	BC
;------------------------------------------------------------------------------
emu_code_c5:
.if stack_mode == 1 
		z80_push	reg_c,reg_b
		rjmp	emu_loop
.endif
.if stack_mode == 2
		ldi	YL,8			;pointer to reg_c
		rjmp	emu_push
.endif
.if stack_mode == 3
		emu_push3	reg_c,reg_b
		rjmp	emu_loop
.endif

;------------------------------------------------------------------------------
; 0xc6		ADD	nn
;------------------------------------------------------------------------------
emu_code_c6:	getpcbyte r16
		reg_add	r16
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0xc7		RST	00H
;------------------------------------------------------------------------------
emu_code_c7:	ldi	r16,0x00
		rjmp	emu_rst			;goto loop

;------------------------------------------------------------------------------
; 0xc8		RZ
;------------------------------------------------------------------------------
emu_code_c8:	bst	reg_f,aflag_z		;copy Z flag
		rjmp	emu_rets		;return if set

;------------------------------------------------------------------------------
; 0xc9		RET
;------------------------------------------------------------------------------
emu_code_c9:	set
		rjmp	emu_rets		;goto loop

;------------------------------------------------------------------------------
; 0xca		JPZ	nnnn
;------------------------------------------------------------------------------
emu_code_ca:	bst	reg_f,aflag_z		;copy Z flag
		rjmp	emu_jmps

;------------------------------------------------------------------------------
; 0xcb		bit commands
;------------------------------------------------------------------------------
emu_code_cb:	jmp	emu1_sel		;jump to alt interpreter

;------------------------------------------------------------------------------
; 0xcc		CAZ	nnnn
;------------------------------------------------------------------------------
emu_code_cc:	bst	reg_f,aflag_z		;copy Z flag
		rjmp	emu_calls		;call if set

;------------------------------------------------------------------------------
; 0xcd		CALL	nnnn
;------------------------------------------------------------------------------
emu_code_cd:	set
		rjmp	emu_calls		;goto loop

;------------------------------------------------------------------------------
; 0xce		ADC	nn
;------------------------------------------------------------------------------
emu_code_ce:	getpcbyte r16
		reg_adc	r16
		rjmp	emu_loopa		;2 goto loop

;------------------------------------------------------------------------------
; 0xcf		RST	08H
;------------------------------------------------------------------------------
emu_code_cf:	ldi	r16,0x08
		rjmp	emu_rst			;goto loop

;------------------------------------------------------------------------------
; 0xd0		RNC
;------------------------------------------------------------------------------
emu_code_d0:	bst	reg_f,aflag_c		;copy C flag
		rjmp	emu_retc		;return if not set

;------------------------------------------------------------------------------
; 0xd1		POP	DE
;------------------------------------------------------------------------------
emu_code_d1:
.if stack_mode == 1 
		z80_pop		reg_e,reg_d
		rjmp	emu_loop
.endif
.if stack_mode == 2
		ldi	YL,10			;pointer to reg_c
		rjmp	emu_pop
.endif
.if stack_mode == 3
		emu_pop3	reg_e,reg_d
		rjmp	emu_loop
.endif

;------------------------------------------------------------------------------
; 0xd2		JPNC	nnnn
;------------------------------------------------------------------------------
emu_code_d2:	bst	reg_f,aflag_c		;copy C flag
		rjmp	emu_jmpc		;jump if cleared

;------------------------------------------------------------------------------
; 0xd3		OUT	nn
;------------------------------------------------------------------------------
emu_code_d3:	getpcbyte r18			;address
		mov	r19,reg_a		;HIGH address
		mov	r16,reg_a
		call	emu_out
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0xd4		CANC	nnnn
;------------------------------------------------------------------------------
emu_code_d4:	bst	reg_f,aflag_c		;copy C flag
		rjmp	emu_callc		;jump if cleared

;------------------------------------------------------------------------------
; 0xd5		PUSH	DE
;------------------------------------------------------------------------------
emu_code_d5:
.if stack_mode == 1 
		z80_push	reg_e,reg_d
		rjmp	emu_loop
.endif
.if stack_mode == 2
		ldi	YL,10			;pointer to reg_e
		rjmp	emu_push
.endif
.if stack_mode == 3
		emu_push3	reg_e,reg_d
		rjmp	emu_loop
.endif

;------------------------------------------------------------------------------
; 0xd6		SUB	nn
;------------------------------------------------------------------------------
emu_code_d6:	getpcbyte r16
		reg_sub	r16
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0xd7		RST	10H
;------------------------------------------------------------------------------
emu_code_d7:	ldi	r16,0x10
		rjmp	emu_rst			;goto loop

;------------------------------------------------------------------------------
; 0xd8		RC
;------------------------------------------------------------------------------
emu_code_d8:	bst	reg_f,aflag_c		;copy C flag
		rjmp	emu_rets		;return if cleared

;------------------------------------------------------------------------------
; 0xd9		EXX
;------------------------------------------------------------------------------
emu_code_d9:	movw	r0,reg_c		;2 change BC
		lds	reg_c,zreg_c1
		lds	reg_b,zreg_b1
		sts	zreg_b1,r1
		sts	zreg_c1,r0

		movw	r0,reg_e		;2 change DE
		lds	reg_e,zreg_e1
		lds	reg_d,zreg_d1
		sts	zreg_d1,r1
		sts	zreg_e1,r0

		movw	r0,reg_l		;2 change HL
		lds	reg_l,zreg_l1
		lds	reg_h,zreg_h1
		sts	zreg_h1,r1
		sts	zreg_l1,r0
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0xda		JPC	nnnn
;------------------------------------------------------------------------------
emu_code_da:	bst	reg_f,aflag_c		;copy C flag
		rjmp	emu_jmps		;jump if set

;------------------------------------------------------------------------------
; 0xdb		IN	nn
;------------------------------------------------------------------------------
emu_code_db:	getpcbyte r18			;address
		mov	r19,reg_a		;HIGH address
		call	emu_in
		mov	reg_a,r16
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0xdc		CAC	nnnn
;------------------------------------------------------------------------------
emu_code_dc:	bst	reg_f,aflag_c		;copy C flag
		rjmp	emu_calls		;call if set

;------------------------------------------------------------------------------
; 0xdd		IX commands
;------------------------------------------------------------------------------
emu_code_dd:	jmp	emu2_sel		;goto loop

;------------------------------------------------------------------------------
; 0xde		SBC	nn
;------------------------------------------------------------------------------
emu_code_de:	getpcbyte r16
		reg_sbc	r16
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0xdf		RST	18H
;------------------------------------------------------------------------------
emu_code_df:	ldi	r16,0x18
		rjmp	emu_rst			;goto loop

;------------------------------------------------------------------------------
; 0xe0		RPO
;------------------------------------------------------------------------------
emu_code_e0:	bst	reg_f,aflag_p		;copy PV flag
		rjmp	emu_retc		;return if not set

;------------------------------------------------------------------------------
; 0xe1		POP	HL
;------------------------------------------------------------------------------
emu_code_e1:
.if stack_mode == 1 
		z80_pop		reg_l,reg_h
		rjmp	emu_loop
.endif
.if stack_mode == 2
		ldi	YL,22			;pointer to reg_l
		rjmp	emu_pop
.endif
.if stack_mode == 3
		emu_pop3	reg_l,reg_h
		rjmp	emu_loop
.endif

;------------------------------------------------------------------------------
; 0xe2		JPPO nnnn
;------------------------------------------------------------------------------
emu_code_e2:	bst	reg_f,aflag_p		;copy PV flag
		rjmp	emu_jmpc		;jump if cleared


;------------------------------------------------------------------------------
; 0xe3		EX (SP),HL
;------------------------------------------------------------------------------
emu_code_e3:
.if	stack_mode == 1
		movw	XL,SP_L
		movw	r0,reg_l
		getmemword	reg_l,reg_h,XL,XH
		putmemword	r0,r1,XL,XH
		rjmp	emu_loop
.endif
.if	stack_mode == 2
		ldi	YL,22
		rjmp	emu_ex16
.endif
.if	stack_mode == 3
		emu_exch3	reg_l,reg_h
		rjmp	emu_loop
.endif

;------------------------------------------------------------------------------
; 0xe4		CAPO nnnn
;------------------------------------------------------------------------------
emu_code_e4:	bst	reg_f,aflag_p		;copy PV flag
		rjmp	emu_callc		;call if cleared

;------------------------------------------------------------------------------
; 0xe5		PUSH HL
;------------------------------------------------------------------------------
emu_code_e5:
.if stack_mode == 1 
		z80_push	reg_l,reg_h
		rjmp	emu_loop
.endif
.if stack_mode == 2
		ldi	YL,22			;pointer to reg_l
		rjmp	emu_push
.endif
.if stack_mode == 3
		emu_push3	reg_l,reg_h
		rjmp	emu_loop
.endif

;------------------------------------------------------------------------------
; 0xe6		AND nn
;------------------------------------------------------------------------------
emu_code_e6:	getpcbyte r16
		and	reg_a,r16
		rjmp	emu_and

;------------------------------------------------------------------------------
; 0xe7		RST	20H
;------------------------------------------------------------------------------
emu_code_e7:	ldi	r16,0x20
		rjmp	emu_rst			;goto loop

;------------------------------------------------------------------------------
; 0xe8		RPE
;------------------------------------------------------------------------------
emu_code_e8:	bst	reg_f,aflag_p		;copy PV flag
		rjmp	emu_rets		;return if set

;------------------------------------------------------------------------------
; 0xe9		JMP M
;------------------------------------------------------------------------------
emu_code_e9:	mov	PC_L,reg_l
		mov	PC_h,reg_h
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0xea		JPPE nnnn
;------------------------------------------------------------------------------
emu_code_ea:	bst	reg_f,aflag_p		;copy PV flag
		rjmp	emu_jmps		;jump if set

;------------------------------------------------------------------------------
; 0xeb		EX 	DE,HL
;------------------------------------------------------------------------------
emu_code_eb:	movw	r0,reg_e
		movw	reg_e,reg_l
		movw	reg_l,r0
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0xec		CAPE nnnn
;------------------------------------------------------------------------------
emu_code_ec:	bst	reg_f,aflag_p		;copy PV flag
		rjmp	emu_calls		;call if set

;------------------------------------------------------------------------------
; 0xed		transport ext
;------------------------------------------------------------------------------
emu_code_ed:	jmp	emu6_sel

;------------------------------------------------------------------------------
; 0xee		XOR n
;------------------------------------------------------------------------------
emu_code_ee:	getpcbyte r16
		eor	reg_a,r16
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xef		RST	28H
;------------------------------------------------------------------------------
emu_code_ef:	ldi	r16,0x28
		rjmp	emu_rst			;goto loop

;------------------------------------------------------------------------------
; 0xf0		RP
;------------------------------------------------------------------------------
emu_code_f0:	bst	reg_f,aflag_s		;copy S flag
		rjmp	emu_retc		;return if not set

;------------------------------------------------------------------------------
; 0xf1		POP AF
;------------------------------------------------------------------------------
emu_code_f1:
.if stack_mode == 1
		z80_pop		reg_f,reg_a
		rjmp	emu_loop
.endif
.if stack_mode == 2
		ldi	YL,20			;pointer to reg_f
		rjmp	emu_pop
.endif
.if stack_mode == 3
		emu_pop3	reg_f,reg_a
		rjmp	emu_loop
.endif

;------------------------------------------------------------------------------
; 0xf2		JPP nnnn
;------------------------------------------------------------------------------
emu_code_f2:	bst	reg_f,aflag_s		;copy S flag
		rjmp	emu_jmpc		;jump if cleared

;------------------------------------------------------------------------------
; 0xf3		DI
;------------------------------------------------------------------------------
emu_code_f3:	;sts	z80_iflag,const_0	;disable INT
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0xf4		CAP nnnn
;------------------------------------------------------------------------------
emu_code_f4:	bst	reg_f,aflag_s		;copy S flag
		rjmp	emu_callc		;call if not set

;------------------------------------------------------------------------------
; 0xf5		PUSH AF
;------------------------------------------------------------------------------
emu_code_f5:
.if stack_mode == 1 
		z80_push	reg_f,reg_a
		rjmp	emu_loop
.endif
.if stack_mode == 2
		ldi	YL,20			;pointer to reg_f
		rjmp	emu_push
.endif
.if stack_mode == 3
		emu_push3	reg_f,reg_a
		rjmp	emu_loop
.endif

;------------------------------------------------------------------------------
; 0xf6		OR n
;------------------------------------------------------------------------------
emu_code_f6:	getpcbyte r16
		or	reg_a,r16
		rjmp	emu_or

;------------------------------------------------------------------------------
; 0xf7		RST	30H
;------------------------------------------------------------------------------
emu_code_f7:	ldi	r16,0x30
		rjmp	emu_rst			;goto loop

;------------------------------------------------------------------------------
; 0xf8		RM
;------------------------------------------------------------------------------
emu_code_f8:	bst	reg_f,aflag_s		;copy S flag
		rjmp	emu_rets		;return if set

;------------------------------------------------------------------------------
; 0xf9		LD	SP,HL
;------------------------------------------------------------------------------
emu_code_f9:	mov	SP_L,reg_l
		mov	SP_H,reg_h
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0xfa		JPM nnnn
;------------------------------------------------------------------------------
emu_code_fa:	bst	reg_f,aflag_s		;copy S flag
		rjmp	emu_jmps		;jump if set

;------------------------------------------------------------------------------
; 0xfb		EI (ignored)
;------------------------------------------------------------------------------
emu_code_fb:	;sts	z80_iflag,const_1	;enable INT
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0xfc		CAM nnnn
;------------------------------------------------------------------------------
emu_code_fc:	bst	reg_f,aflag_s		;copy S flag
		rjmp	emu_calls		;return if set

;------------------------------------------------------------------------------
; 0xfd		IY commands
;------------------------------------------------------------------------------
emu_code_fd:	jmp	emu3_sel		;goto loop

;------------------------------------------------------------------------------
; 0xfe		CMP nn
;------------------------------------------------------------------------------
emu_code_fe:	getpcbyte r16
		cp	reg_a,r16		;sbc M
		rjmp	emu_loops		;2 goto loop

;------------------------------------------------------------------------------
; 0xff		RST	38H
;------------------------------------------------------------------------------
emu_code_ff:	ldi	r16,0x38
		rjmp	emu_rst			;goto loop

;------------------------------------------------------------------------------
; push (Y)
;------------------------------------------------------------------------------
emu_push:	ldi	r16,2
		sub	SP_L,r16
		sbc	SP_H,const_0
		ldi	YH,0
		ld	r16,Y+			;LOW register
		ld	r17,Y			;HIGH register
		putmemword	r16,r17,SP_L,SP_H
		rjmp	emu_loop

;------------------------------------------------------------------------------
; pop (Y)
;------------------------------------------------------------------------------
emu_pop:	ldi	YH,0
		getmemword	r16,r17,SP_L,SP_H
		st	Y+,r16			;LOW register
		st	Y,r17			;HIGH register
		ldi	r16,2
		add	SP_L,r16
		adc	SP_H,const_0
		rjmp	emu_loop

;------------------------------------------------------------------------------
; EX (Y),(SP)
;------------------------------------------------------------------------------
emu_ex16:	ldi	YH,0
		ld	r0,Y			;low reg
		ldd	r1,Y+1			;high reg
		movw	XL,SP_L
		getmembyte	r16,XL,XH
		putmembyte	r0,XL,XH
		adiw	XL,1
		st	Y,r16
		getmembyte	r16,XL,XH
		putmembyte	r1,XL,XH
		std	Y+1,r16
		rjmp	emu_loop

emu_mwrite:	putmembyte	r16,reg_l,reg_h
		rjmp	emu_loop

emu_mread:	getmembyte	r16,reg_l,reg_h
		ret

