;################################################################################
;#										#
;# fsys - file system library							#
;# copyright (c) 2011 Joerg Wolfram (joerg@jcwolfram.de)			#
;#										#
;# This library is free software; you can redistribute it and/or		#
;# modify it under the terms of the GNU  General Public				#
;# License as published by the Free Software Foundation; either			#
;# version 3 of the License, or (at your option) any later version.		#
;#										#
;# This library is distributed in the hope that it will be useful,		#
;# but WITHOUT ANY WARRANTY; without even the implied warranty of		#
;# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU		#
;# General Public License for more details.					#
;#										#
;# You should have received a copy of the GNU  General Public			#
;# License along with this library; if not, write to the			#
;# Free Software Foundation, Inc., 59 Temple Place - Suite 330,			#
;# Boston, MA 02111-1307, USA.							#
;#										#
;################################################################################

;------------------------------------------------------------------------------
; init filesystem and write messages
;------------------------------------------------------------------------------
fsys_init:	ldi	XL,0x07
		sts	vsys_color,XL
		out	RAMPZ,const_1
		ldi	XL,0
		ldi	XH,0
		out	SPCR,const_0
		sbi	SPI_SPORT,SPI_SPIN	;set port pins
		sbi	SPI_SDDR,SPI_SPIN
		sbi	MOSI_DDR,MOSI_PIN
		sbi	SCK_DDR,SCK_PIN
		sts	fsys_tapenr,const_0	;start with tape 0
		sts	fsys_status,const_0	;card not available
		sts	fsys_maxtape,const_0	;card not available

fsys_init_0:	ldi	ZL,LOW(fsys_buf2)
		ldi	ZH,HIGH(fsys_buf2)
		ldi	XH,10
		ldi	XL,0x20
fsys_init_0a:	st	Z+,XL
		dec	XH
		brne	fsys_init_0a

fsys_init_1:	nop
		nop
		nop
		sbiw	XL,1
		brne	fsys_init_1

		rcall	fsys_sdi		;check for SD card

		ldi	XL,1
		ldi	XH,22
		call	sys_gotoxy

		rcall	fsys_pstat		;view type
		lds	XL,fsys_status		;status
		sbrs	XL,7
		ret				;no format check

		rcall	fsys_nfind		;find image (new version)

		ldi	XL,13
		ldi	XH,22
		sts	vsys_xpos,XL
		sts	vsys_ypos,XH

		in	r16,SPCR
		sbrs	r16,0
		rjmp	fsys_init_2
		ldi	r19,'*'
		rcall	sys_char

fsys_init_2:	rcall	fsys_fstat		;format status
		lds	r16,fsys_status
;		call	sys_hexout
		ret


;------------------------------------------------------------------------------
; wait for SPI is ready for next data
; write XL to SPI
;------------------------------------------------------------------------------
fsys_write_ff2:	rcall	fsys_write_ff
fsys_write_ff:	ldi	XL,0xff			;dummy data
fsys_wspi:	out	SPDR,XL
fsys_wait:	in	XL,SPSR
		sbrs	XL,SPIF
		rjmp	fsys_wait
		in	XL,SPDR
		ret

fsys_write_00:	clr	XL
		rjmp	fsys_wspi


;------------------------------------------------------------------------------
; print out filesystem status
;------------------------------------------------------------------------------
fsys_pstat:	lds	XL,fsys_status
		ldi	ZL,LOW(fsys_txt1*2)
		ldi	ZH,HIGH(fsys_txt1*2)
		cpi	XL,0x01			;card error
		breq	fsys_ppstat
		adiw	ZL,12
		cpi	XL,0x81			;MMC
		breq	fsys_ppstat
		adiw	ZL,12
		cpi	XL,0x82			;SDV1
		breq	fsys_ppstat
		adiw	ZL,12
		cpi	XL,0x83			;SDV2
		breq	fsys_ppstat
		adiw	ZL,12
		cpi	XL,0x84			;SDHC
		breq	fsys_ppstat
		adiw	ZL,12
		cpi	XL,0x02			;ACMD41 ERR
		breq	fsys_ppstat
		adiw	ZL,12
		cpi	XL,0x03			;CMD1 ERR
		breq	fsys_ppstat
		adiw	ZL,12
		cpi	XL,0x04			;PATTERN ERROR
		breq	fsys_ppstat
		adiw	ZL,12
		cpi	XL,0x05			;CMD0 FAIL
		breq	fsys_ppstat
		adiw	ZL,12
fsys_ppstat:	call	sys_flashtext
fsys_pstat_e:	ret


fsys_txt1:	.db "CARD ERROR ",0x00		;CARD ERR
		.db "MMC-CARD   ",0x00		;MMC-CARD
		.db "SDV1-CARD  ",0x00		;SDV1-CARD
		.db "SDV2-CARD  ",0x00		;SDV2-CARD
		.db "SDHC-CARD  ",0x00		;SDHC-CARD
		.db "ERR-ACMD41 ",0x00		;E-ACMD41
		.db "ERR-CMD1   ",0x00		;E-CMD1
		.db "ERR-CMD8   ",0x00		;E-CMD8
		.db "ERR-CMD0   ",0x00		;E-CMD0
		.db "NO CARD    ",0x00		;NO CARD


;------------------------------------------------------------------------------
; print out filesystem status2
;------------------------------------------------------------------------------
fsys_fstat:	lds	XL,fsys_status
		ldi	ZL,LOW(fsys_txt2*2)
		ldi	ZH,HIGH(fsys_txt2*2)
		sbrs	XL,7
		rjmp	fsys_fstat_e
		lds	XL,vsys_mode
		andi	XL,0x07
		inc	XL
		ldi	XH,16
		mul	XL,XH
		add	ZL,r0
		adc	ZH,r1
		rjmp	fsys_ppstat

fsys_fstat_e:	ldi	XL,16
		dec	r16
		mul	XL,r16
		ldi	ZL,LOW(fsys_txt3*2)
		ldi	ZH,HIGH(fsys_txt3*2)
		add	ZL,r0
		adc	ZH,r1
		rjmp	fsys_ppstat


fsys_txt2:	.db "NO IMAGE FOUND ",0x00	;NO IMAGE
		.db "AX81-ZX81 IMAGE",0x00	;AX-IMAGE
		.db "AX81-ZX81 IMAGE",0x00	;AX-IMAGE
		.db "AX81-ZX81 IMAGE",0x00	;AX-IMAGE
		.db "AX81-ZX80 IMAGE",0x00	;AX-IMAGE
		.db "AX81-JACE IMAGE",0x00	;AX-IMAGE
		.db "AX81-SPEC IMAGE",0x00	;AX-IMAGE
		.db "AX81-DMMY IMAGE",0x00	;AX-IMAGE
		.db "AX81-DMMY IMAGE",0x00	;AX-IMAGE

fsys_txt3:	.db "NO MBR FOUND   ",0x00	;no MBR
		.db "NO FAT16 PART. ",0x00	;no FAT 16 Partition
		.db "NO 512B SECTORS",0x00	;wrong geometry
		.db "IMG NOT IN DIR ",0x00	;image not found in dir
		.db "WRONG IMAGE    ",0x00	;wrong image information

;------------------------------------------------------------------------------
; write command XH to SPI
; XL=return byte (FF=time out)
;------------------------------------------------------------------------------
fsys_cmd0:	clr	r16			;clear address field
		clr	r17
		movw	r18,r16

fsys_cmd:	cbi	SPI_SPORT,SPI_SPIN	;CS active
		rcall	fsys_write_ff2		;16 dummy clocks
		andi	XH,0x3f			;limit
		ori	XH,0x40			;set bit 6
		out	SPDR,XH			;CMD
		andi	XH,0x3f			;clear bit 6
		rcall	fsys_wait
		out	SPDR,r19		;address MSB
		rcall	fsys_wait
		out	SPDR,r18
		rcall	fsys_wait
		out	SPDR,r17
		rcall	fsys_wait
		out	SPDR,r16		;address LSB
		rcall	fsys_wait
		ldi	XL,0x95			;dummy checksum
		cpi	XH,0x08
		brne	fsys_cmd_01
		ldi	XL,0x87
fsys_cmd_01:	rcall	fsys_wspi
		rjmp	fsys_wait_nff		;wait for non 0xff byte


fsys_xcmd:	cbi	SPI_SPORT,SPI_SPIN	;CS active
		rcall	fsys_write_ff2		;16 dummy clocks
		mov	XL,XH
		andi	XL,0x3f			;limit
		ori	XL,0x40			;set bit 6
		rcall	fsys_wspi
		rcall	fsys_saddr		;out sector address
		rjmp	fsys_wait_nff		;wait for non 0xff byte


;------------------------------------------------------------------------------
; wait for start block mtoken
;------------------------------------------------------------------------------
fsys_waitsbt:	push	ZL
		push	ZH
		ldi	ZL,0			;timeout
		ldi	ZH,0

fsys_waitsbt1:	cpi	XL,0xfe			;start block token
		brne	fsys_waitsbt2
		pop	ZH
		pop	ZL
		ret				;SBT found

fsys_waitsbt2:	sbiw	ZL,1			;timeout counter
		breq	fsys_waitsbt3		;branch if timeout is reached
		rcall	fsys_write_ff
		rjmp	fsys_waitsbt1		;next try

fsys_waitsbt3:	pop	ZH
		pop	ZL
		ret

;------------------------------------------------------------------------------
; init SD card and determine type
;------------------------------------------------------------------------------
fsys_sdi:	;init SPI to f/128
		ldi	XL,0x53			;f/128
		out	SPCR,XL
		ldi	XL,0x04			;timeout factor

		;80 dummy pulses with CS=HI
fsys_sdi_00:	ldi	XH,10
		sbi	SPI_SPORT,SPI_SPIN	;CS inactive
fsys_sdi_01:	rcall	fsys_write_ff		;128 clock pulses
		dec	XH
		brne	fsys_sdi_01

		ldi	YL,0x08			;CMD0 retries
		ldi	YH,0x00

fsys_sdi_02:	ldi	XH,0			;CMD0
		rcall	fsys_cmd0		;send CMD0
		cpi	XL,0xff
		breq	fsys_sdi_02a
		sbrc	XL,0			;skip if idle state
		rjmp	fsys_sdi_03		;yes->OK
fsys_sdi_02a:	sbiw	YL,1			;retry counter
		brne	fsys_sdi_02		;retry CMD 0

fsys_sdi_err:	sbi	SPI_SPORT,SPI_SPIN	;CS inactive
		ldi	XL,0x00			;CMD0 error
		sts	fsys_status,XL		;card not available
		ret				;exit

		;check for SDv2 with CMD8
fsys_sdi_03:	ldi	r16,0xaa		;test pattern
		ldi	r17,0x01
		clr	r18
		clr	r19
		ldi	XH,8			;CMD8
		rcall	fsys_cmd		;send CMD8
		sbrc	XL,2			;illegal?
		rjmp	fsys_sdi_04		;yes
		rcall	fsys_write_ff2
		rcall	fsys_write_ff2
		cpi	XL,0xaa			;test pattern
		brne	fsys_sdi_err1		;fail
		ldi	XL,0x90			;SDv2
		sts	fsys_status,XL		;write status
		rjmp	fsys_sdi_05		;goto wait idle

		;check for MMC / SD
fsys_sdi_04:	ldi	XH,55			;CMD55
		rcall	fsys_cmd0		;send CMD55
		ldi	XH,1			;ACMD41
		rcall	fsys_cmd0		;send ACMD41
		sbrc	XL,2			;illegal?
		rjmp	fsys_sdi_20		;MMC
		ldi	XL,0x82			;SDv1
		sts	fsys_status,XL		;write status

		;wait for card ready
fsys_sdi_05:	ldi	YL,0
		ldi	YH,0x80

fsys_sdi_06:	ldi	XH,55			;CMD55
		rcall	fsys_cmd0		;send CMD55
		lds	XL,fsys_status		;get stat
		sbrc	XL,4			;check V2
		ldi	r19,0x40
		ldi	XH,1			;ACMD41
		rcall	fsys_cmd		;send ACMD41
		sbrs	XL,0			;skip if idle state
		rjmp	fsys_sdi_10		;yes->OK
		sbiw	YL,1			;loops
		brne	fsys_sdi_06

fsys_sdi_07:	ldi	r16,0x02		;ACMD41 err
		sts	fsys_status,r16
		sbi	SPI_SPORT,SPI_SPIN	;CS inactive
		ret

fsys_sdi_err1:	ldi	XL,0x04
		sts	fsys_status,XL		;card error
		sbi	SPI_SPORT,SPI_SPIN	;CS inactive
		ret				;exit

fsys_sdi_10:	lds	XL,fsys_status		;get stat
		sbrs	XL,4			;
		rjmp	fsys_sdi_end		;SDv1 card
		ldi	XL,0x83			;set SDv2
		sts	fsys_status,XL		;write status

		;now check for SDHC or V2
		ldi	XH,58			;CMD58
		rcall	fsys_cmd0		;send CMD58
		rcall	fsys_write_ff
		push	XL
		rcall	fsys_write_ff
		rcall	fsys_write_ff
		rcall	fsys_write_ff
		pop	XL
		sbrs	XL,6
		rjmp	fsys_sdi_end		;is SDv2 card
		ldi	XL,0x84			;SDHC card
		sts	fsys_status,XL		;write status

fsys_sdi_end:	clr	r16			;512 Bytes block size
		ldi	r17,2
		clr	r18
		clr	r19
		ldi	XH,16			;CMD16
		rcall	fsys_cmd		;send CMD16
		sbi	SPI_SPORT,SPI_SPIN	;CS inactive
		ldi	XL,0x50			;f/4
		lds	XH,vsys_kbdstate
		sbrc	XH,4			;skip if no ALT key
		ldi	XL,0x51			;f/16
		out	SPCR,XL
		ret				;exit

fsys_sdi_20:	ldi	YL,0
		ldi	YH,0x80

fsys_sdi_21:	ldi	XH,1			;CMD1
		rcall	fsys_cmd0		;send CMD1
		sbrs	XL,0			;skip if idle state
		rjmp	fsys_sdi_22		;yes->OK
		sbiw	YL,1			;loops
		brne	fsys_sdi_21		
		ldi	r16,0x03		;CMD1 err
		sts	fsys_status,r16
		sbi	SPI_SPORT,SPI_SPIN	;CS inactive
		ret

fsys_sdi_22:	ldi	XL,0x81			;MMC card
		sts	fsys_status,XL		;write status
		rjmp	fsys_sdi_end


;------------------------------------------------------------------------------
; calc sector address to r16..r19
; fsys_tapenr	= tape
; fsys_filenr	= file
; fsys_sectnr	= sector
;------------------------------------------------------------------------------
fsys_sectaddr0:	sts	fsys_sectnr,const_0	;first sector
fsys_sectaddr:	push	XL
		push	XH
		push	YL
		push	YH
		push	ZL
		push	ZH
		lds	XL,fsys_sectnr
		lds	XH,fsys_offset_l
		add	XL,XH
		sts	fsys_sectnr0,XL
		lds	XL,fsys_offset_m
		adc	XL,const_0
		sts	fsys_sectnr1,XL
		lds	XL,fsys_offset_h
		adc	XL,const_0
		sts	fsys_sectnr2,XL
		lds	XL,fsys_offset_e
		adc	XL,const_0
		sts	fsys_sectnr3,XL

		lds	YL,fsys_tapenr		;tape number
		ldi	ZL,LOW(1056)
		mul	YL,ZL
		movw	XL,r0
		ldi	ZL,HIGH(1056)
		mul	YL,ZL
		clr	YL
		clr	YH
		add	XH,r0
		adc	YL,r1
		adc	YH,const_0
		ldi	ZL,0x21
		lds	r0,fsys_filenr
		mul	ZL,r0
		add	XL,r0
		adc	XH,r1
		adc	YL,const_0
		adc	YH,const_0
		rcall	fsys_addsec
		pop	ZH
		pop	ZL
		pop	YH
		pop	YL
		pop	XH
		pop	XL
		ret

;------------------------------------------------------------------------------
; read name
; fileio_ram+32	= tape
; fileio_ram+33	= file
;------------------------------------------------------------------------------
fsys_rname:	sts	fsys_sectnr,const_0
		ldi	XH,17
		rcall	fsys_xcmd
		rcall	fsys_waitsbt		;wait for start block token

		ldi	YL,LOW(fsys_buf2)
		ldi	YH,HIGH(fsys_buf2)
		ldi	XH,12			;fname length + len
fsys_rname_5:	rcall	fsys_write_ff
;		ldi	XL,'+'
		st	Y+,XL
		dec	XH
		brne	fsys_rname_5
		ldi	XH,251			;(510/2)+2CRC
fsys_rname_6:	rcall	fsys_write_ff2
		dec	XH
		brne	fsys_rname_6
		sbi	SPI_SPORT,SPI_SPIN	;CS inactive
		;change complete to space if first char ist > 127
		ldi	YL,LOW(fsys_buf2)
		ldi	YH,HIGH(fsys_buf2)
		ld	XL,Y
		cpi	XL,0xff
		brne	fsys_name_e
		ldi	XH,10			;fname length + len
		ldi	XL,' '
fsys_rname_7:	st	Y+,XL
		dec	XH
		brne	fsys_rname_7
fsys_name_e:	ret


;------------------------------------------------------------------------------
; delete current file (overwrite first sector with 0xff
;------------------------------------------------------------------------------
fsys_delfile:	sts	fsys_sectnr,const_0
		ldi	XH,24
		call	fsys_xcmd
		ldi	XL,0xfe			;start block token
		call	fsys_wspi

		ldi	XH,0
fsys_delfile_1:	call	fsys_write_ff2
		dec	XH
		brne	fsys_delfile_1
		call	fsys_write_ff2		;dummy CRC

		rcall	fsys_wait_nff		;wait for non 0xff byte
		sbi	SPI_SPORT,SPI_SPIN
		ret


;------------------------------------------------------------------------------
; search for valid image
;------------------------------------------------------------------------------
fsys_nfind:	clr	r16
		rcall	fsys_rdmbr		;read MBE
		cpi	r16,0x00
		brne	fsys_nfind_err
		rcall	fsys_rbs		;read Bootsector
		cpi	r16,0x00
		brne	fsys_nfind_err
		rcall	fsys_rdir		;read root directory
		cpi	r16,0x00
		brne	fsys_nfind_err
		rcall	fsys_rsize
		cpi	r16,0x00
		brne	fsys_nfind_err
		ret

fsys_nfind_err:	;call	sys_hexout
		sts	fsys_status,const_0
		ret


;------------------------------------------------------------------------------
; read sector
; fsys_tapenr	= tape
; fsys_filenr	= file
; fsys_sectnr	= sector
;------------------------------------------------------------------------------
fsys_rsector:	ldi	XH,17
		rcall	fsys_xcmd
		rcall	fsys_waitsbt		;wait for start block token
		ldi	XH,0x00
		ldi	YL,LOW(fsys_buffer)
		ldi	YH,HIGH(fsys_buffer)
fsys_sread_4:	rcall	fsys_write_ff
		st	Y+,XL
		rcall	fsys_write_ff
		st	Y+,XL
		dec	XH
		brne	fsys_sread_4
		rcall	fsys_write_ff2		;dummy CRC read
		sbi	SPI_SPORT,SPI_SPIN	;de-activate
		ret

;-----------------------------------------------------------------------------
; read sector to altered buffer (use zx_ram at startup)
;-----------------------------------------------------------------------------
fsys_asread:	cbi	SPI_SPORT,SPI_SPIN	;activate
		rcall	fsys_write_ff2		;16 dummy clocks
		ldi	XL,0x51			;read (17) + 64
		rcall	fsys_wspi
		rcall	fsys_saddr0		;sector address
		clr	r0
fsys_asread_2:	rcall	fsys_write_ff
		cpi	XL,0xff
		brne	fsys_asread_3
		dec	r0
		brne	fsys_asread_2
fsys_asread_3:	rcall	fsys_waitsbt		;wait for start block token
		ldi	XH,0x00
		ldi	YL,LOW(zx_ram)
		ldi	YH,HIGH(zx_ram)
fsys_asread_4:	rcall	fsys_write_ff
		st	Y+,XL
		rcall	fsys_write_ff
		st	Y+,XL
		dec	XH
		brne	fsys_asread_4
		rcall	fsys_write_ff2
		sbi	SPI_SPORT,SPI_SPIN	;de-activate
		ret

;------------------------------------------------------------------------------
; write sector
; fsys_tapenr	= tape
; fsys_filenr	= file
; fsys_sectnr	= sector
;------------------------------------------------------------------------------
fsys_wsector:	ldi	XH,24			;write single sector
		rcall	fsys_xcmd
		ldi	XL,0xfe			;start block token
		rcall	fsys_wspi
		ldi	XH,0x00
		ldi	YL,LOW(fsys_buffer)
		ldi	YH,HIGH(fsys_buffer)
fsys_swrite_4:	ld	XL,Y+
		rcall	fsys_wspi
		ld	XL,Y+
		rcall	fsys_wspi
		dec	XH
		brne	fsys_swrite_4
		rcall	fsys_write_ff2
fsas_swrite_5:	rcall	fsys_write_ff
		cpi	XL,0xff
		brne	fsas_swrite_5
		sbi	SPI_SPORT,SPI_SPIN	;de-activate
		ret


;-----------------------------------------------------------------------------
; read master boot record
;-----------------------------------------------------------------------------
fsys_rdmbr:	sts	fsys_sectnr3,const_0
		sts	fsys_sectnr2,const_0
		sts	fsys_sectnr1,const_0
		sts	fsys_sectnr0,const_0
		rcall	fsys_asread		;read sector into buffer
		lds	XL,zx_ram+0x1fe	;check MBR signature
		lds	XH,zx_ram+0x1ff
		cpi	XL,0x55
		brne	fsys_rdmbr_err
		cpi	XH,0xaa
		breq	fsys_rdmbr_0
fsys_rdmbr_err:	ldi	r16,0x01		;NO MBR
		ret

fsys_rdmbr_0:	ldi	YL,LOW(zx_ram+0x1c2)
		ldi	YH,HIGH(zx_ram+0x1c2)
		ldi	XH,4
		clr	r16			;OK
fsys_rdmbr_1:	ld	XL,Y
		cpi	XL,0x06			;FAT16
		breq	fsys_rdmbr_3
		adiw	YL,16
		dec	XH
		brne	fsys_rdmbr_1
		ldi	r16,0x02		;NO FAT16 partition
fsys_rdmbr_2:	ret
		;set address of boot sector
fsys_rdmbr_3:	ldd	XH,Y+4
		sts	fsys_sectnr0,XH
		ldd	XH,Y+5
		sts	fsys_sectnr1,XH
		ldd	XH,Y+6
		sts	fsys_sectnr2,XH
		ldd	XH,Y+7
		sts	fsys_sectnr3,XH
		ret

;-----------------------------------------------------------------------------
; read boot sector
;-----------------------------------------------------------------------------
fsys_rbs:	rcall	fsys_asread		;read sector into buffer
		lds	XL,zx_ram+0x0b
		cpi	XL,0x00			;only 512 bytes/sector allowed
		brne	fsys_rbs_err
		lds	XL,zx_ram+0x0c
		cpi	XL,0x02			;only 512 bytes/sector allowed
		breq	fsys_rbs_1

fsys_rbs_err:	ldi	r16,0x03		;wrong geometry
		ret

fsys_rbs_1:	lds	XL,zx_ram+0x0d		;sectors per cluster
		sts	fsys_clsec,XL		;store

		lds	XL,zx_ram+0x0e		;reserved sectors
		lds	XH,zx_ram+0x0f
		clr	YL
		clr	YH
		rcall	fsys_addsec		;add reserved sectors

		push	r16
		push	r17
		push	r18

		lds	r16,zx_ram+0x16	;sectors per FAT
		lds	r17,zx_ram+0x17
		lds	r18,zx_ram+0x10	;number of FAT copies

		mul	r16,r18
		movw	XL,r0
		clr	YL
		clr	YH
		mul	r17,r18
		add	XH,r0
		adc	YL,r1
		adc	YH,const_0
		pop	r18
		pop	r17
		pop	r16
		rcall	fsys_addsec		;add FAT sectors, we got the address of root directory
		ret

;-----------------------------------------------------------------------------
; read root directory
;-----------------------------------------------------------------------------
fsys_rdir:	rcall	fsys_asread		;read sector into buffer
		ldi	YL,LOW(zx_ram)
		ldi	YH,HIGH(zx_ram)
fsys_rdir_1:	ldi	ZL,LOW(fsys_dirent*2)
		ldi	ZH,HIGH(fsys_dirent*2)
		lds	XL,vsys_mode
		andi	XL,0x07
		ldi	XH,12			;bytes per entry
		mul	XL,XH
		add	ZL,r0
		adc	ZH,r1
		ld	XL,Y
		cpi	XL,0x00			;no entry
		breq	fsys_rdir_err
		push	YL
		push	YH
		ldi	XH,11			;bytes to compare
fsys_rdir_2:	elpm	r0,Z+			;get ROM data
		ld	r1,Y+			;get buffer data
		cp	r0,r1			;compare
		breq	fsys_rdir_3		;branch if equal
		pop	YH
		pop	YL
		adiw	YL,32			;next entry
		cpi	YH,HIGH(zx_ram)+2	;all done?
		brcs	fsys_rdir_1		;branch if not

fsys_rdir_err:	ldi	r16,0x04		;file not found
		ret

fsys_rdir_3:	dec	XH			;bytes counter
		brne	fsys_rdir_2		;next char
		pop	YH
		pop	YL

		push	r16
		push	r17
		push	r18

		ldd	r16,Y+0x1a		;offset
		ldd	r17,Y+0x1b
		ldi	r18,2
		sub	r16,r18
		sbc	r17,const_0

		lds	r18,fsys_clsec		;sectors per cluster
		mul	r16,r18
		movw	XL,r0
		clr	YL
		clr	YH
		mul	r17,r18
		add	XH,r0
		adc	YL,r1
		adc	YH,const_0
		ldi	r17,32			;sectors for directory
		cp	r17,r18			;check if cluster size is bigger
		brcc	fsys_rdir_4
		mov	r17,r18
fsys_rdir_4:	add	XL,r17
		adc	XH,const_0
		adc	YL,const_0
		adc	YH,const_0
		pop	r18
		pop	r17
		pop	r16
		rcall	fsys_addsec		;add offset
;		rcall	fsys_viewoffset
		ret

fsys_dirent:
		.db	"AX81_X81BIN",0x00
		.db	"AX81_X81BIN",0x00
		.db	"AX81_X81BIN",0x00
		.db	"AX81_X80BIN",0x00
		.db	"AX81_ACEBIN",0x00
		.db	"AX81_SPCBIN",0x00
		.db	"AX81_DMYBIN",0x00
		.db	"AX81_DMYBIN",0x00

;-----------------------------------------------------------------------------
; read file size
;-----------------------------------------------------------------------------
fsys_rsize:	rcall	fsys_asread		;read sector into buffer
		lds	XL,zx_ram+11
		sts	fsys_maxtape,XL
		lds	XL,ZX_ram+0		;check image start bytes
		cpi	XL,0xa6
		brne	fsys_rsize_err
		lds	XL,ZX_ram+1
		cpi	XL,0xbd
		brne	fsys_rsize_err
		lds	XL,ZX_ram+2
		cpi	XL,0x26
		brne	fsys_rsize_err
		lds	XL,ZX_ram+3
		cpi	XL,0x3d
		brne	fsys_rsize_err
;		mov	r16,XL
;		rcall	sys_hexout
		ldi	XL,0x01			;offset=1 (head sector)
		ldi	XH,0x00
		ldi	YL,0x00
		ldi	YH,0x00

		rcall	fsys_addsec		;add offset
		lds	XL,fsys_sectnr0
		sts	fsys_offset_l,XL
		lds	XL,fsys_sectnr1
		sts	fsys_offset_m,XL
		lds	XL,fsys_sectnr2
		sts	fsys_offset_h,XL
		lds	XL,fsys_sectnr3
		sts	fsys_offset_e,XL
		clr	r16
		ret

fsys_rsize_err:	ldi	r16,0x05		;wrong image
		ret

;-----------------------------------------------------------------------------
; add sector offset
;-----------------------------------------------------------------------------
fsys_addsec:	push	r16
		lds	r16,fsys_sectnr0
		add	r16,XL
		sts	fsys_sectnr0,r16

		lds	r16,fsys_sectnr1
		adc	r16,XH
		sts	fsys_sectnr1,r16

		lds	r16,fsys_sectnr2
		adc	r16,YL
		sts	fsys_sectnr2,r16

		lds	r16,fsys_sectnr3
		adc	r16,YH
		sts	fsys_sectnr3,r16
		pop	r16
		ret


;-----------------------------------------------------------------------------
; out sector address + dummy CRC
;-----------------------------------------------------------------------------
fsys_saddr:	rcall	fsys_sectaddr		;calculate sector number
fsys_saddr0:	lds	XL,fsys_status		;fs status
		sbrs	XL,2			;skip if SDHC
		rjmp	fsys_saddr_1		;no SDHC
		lds	XL,fsys_sectnr3
		rcall	fsys_wspi
		lds	XL,fsys_sectnr2
		rcall	fsys_wspi
		lds	XL,fsys_sectnr1
		rcall	fsys_wspi
		lds	XL,fsys_sectnr0
		rcall	fsys_wspi
		ldi	XL,0x95			;dummy crc
		rcall	fsys_wspi
		ret


fsys_saddr_1:	lds	XL,fsys_sectnr2
		lds	XH,fsys_sectnr1
		lsl	XH
		rol	XL
		rcall	fsys_wspi
		mov	XL,XH
		lsr	XL
		lds	XH,fsys_sectnr0
		lsl	XH
		rol	XL
		rcall	fsys_wspi
		mov	XL,XH
		rcall	fsys_wspi
		clr	XL
		rcall	fsys_wspi
		ldi	XL,0x95			;dummy crc
		rjmp	fsys_wspi

;-----------------------------------------------------------------------------
; wait for 0xff output
;-----------------------------------------------------------------------------
fsys_wait_nff:	push	ZL
		push	ZH
		ldi	ZL,0			;max. 1024
		ldi	ZH,4
fsys_wait_ff_1:	call	fsys_write_ff
		cpi	XL,0xff
		brne	fsys_wait_ff_2
		sbiw	ZL,1
		brne	fsys_wait_ff_1
fsys_wait_ff_2:	pop	ZH
		pop	ZL
		ret


fsys_viewoffset1:
		push	r16
		lds	r16,fsys_offset_e
		call	sys_hexout
		lds	r16,fsys_offset_h
		call	sys_hexout
		lds	r16,fsys_offset_m
		call	sys_hexout
		lds	r16,fsys_offset_l
		call	sys_hexout
		pop	r16
		ret

fsys_viewoffset:
		push	r16
		lds	r16,fsys_sectnr3
		call	sys_hexout
		lds	r16,fsys_sectnr2
		call	sys_hexout
		lds	r16,fsys_sectnr1
		call	sys_hexout
		lds	r16,fsys_sectnr0
		call	sys_hexout
		pop	r16
		call	vsys_waitkey
		ret
