;################################################################################
;#										#
;# UPROG2 universal programmer for linux					#
;#										#
;# copyright (c) 2012-2016 Joerg Wolfram (joerg@jcwolfram.de)			#
;#										#
;#										#
;# This program is free software; you can redistribute it and/or		#
;# modify it under the terms of the GNU General Public License			#
;# as published by the Free Software Foundation; either version 2		#
;# of the License, or (at your option) any later version.			#
;#										#
;# This program is distributed in the hope that it will be useful,		#
;# but WITHOUT ANY WARRANTY; without even the implied warranty of		#
;# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the GNU		#
;# General Public License for more details.					#
;#										#
;# You should have received a copy of the GNU General Public			#
;# License along with this library; if not, write to the			#
;# Free Software Foundation, Inc., 59 Temple Place - Suite 330,			#
;# Boston, MA 02111-1307, USA.							#
;#										#
;################################################################################
.equ			SWD32_RST		= SIG1
.equ			SWD32_CLOCK		= SIG2
.equ			SWD32_DATA		= SIG3
.equ			SWD32_TRIGGER		= SIG5

.equ			SWD32_ZERO_R		= 0				;none
.equ			SWD32_ONE_R		= SIG3_OR			;data
.equ			SWD32_ZERO		= SIG1_OR			;reset
.equ			SWD32_ONE		= (SIG3_OR + SIG1_OR)		;data + reset
.equ			SWD32_CLK		= SIG2_OR			;only clock


;SW-DP registers
.equ			SWD32_READ_IDCODE	= 0xa5
.equ			SWD32_WRITE_ABORT	= 0x81

.equ			SWD32_READ_CTRL		= 0xb1
.equ			SWD32_WRITE_CTRL	= 0x95

.equ			SWD32_READ_STAT		= 0xa9		;b5
.equ			SWD32_WRITE_SELECT	= 0x8d		;8d

.equ			SWD32_READ_RDBUFF	= 0xbd		;read buffer


; AHB-AP registers

.equ			SWD32_READ_CSW		= 0xe1
.equ			SWD32_WRITE_CSW		= 0xc5

.equ			SWD32_READ_TAR		= 0xf5
.equ			SWD32_WRITE_TAR		= 0xd1

.equ			SWD32_READ_IDR		= 0xed
.equ			SWD32_WRITE_BASE	= 0xc9

.equ			SWD32_READ_DRW		= 0xf9
.equ			SWD32_WRITE_DRW		= 0xdd

			
;-------------------------------------------------------------------------------
; init
; R19 bit 4 = release reset
;-------------------------------------------------------------------------------
swd32_init:		out		CTRLPORT,const_0
			sbi		CTRLDDR,SWD32_RST
			sbi		CTRLDDR,SWD32_CLOCK
			sbi		CTRLDDR,SWD32_DATA

			;we do a connect under reset
			call		api_vcc_on		;power on
			ldi		ZL,200
			ldi		ZH,0
			call		api_wait_ms

			sbrc		r19,4
			sbi		CTRLPORT,SWD32_RST	;release reset


swd32_init_1:		cbi		CTRLPORT,SWD32_CLOCK
			cbi		CTRLPORT,SWD32_DATA
			clr		ZL
			rcall		swd32_w0_1

			ldi		r16,0x41		;timeout

			rcall		swd32_reginit_reset	;set registers for faster output
			sbrc		r19,4
			rcall		swd32_reginit		;set registers for faster output
			ldi		YL,0
			ldi		YH,1

			;now get chip id
			ldi		r24,0
			ldi		r25,4			;1024 tries
swd32_init_2:		rcall		swd32_reset		;reset state machine

swd32_init_2a:		ldi		XL,SWD32_READ_IDCODE	;read ID code
			rcall		swd32_read_dap
			
			sts		0x100,XL
			cpi		XL,0x04			;no ack -> exit
			breq		swd32_init_3
			rcall		swd32_wait_1ms		;next try
			sbiw		r24,1
			brne		swd32_init_2
			rjmp		swd32_init_err

swd32_init_3:		st		Y+,r20			;return ID
			st		Y+,r21
			st		Y+,r22
			st		Y+,r23


			ldi		ZL,LOW(swd32_data_init1*2)
			ldi		ZH,HIGH(swd32_data_init1*2)

			rcall		swd32_read_ctrlstat			

			;DebugPortStart
			ldi		r24,5
swd32_init_3a:		rcall		swd32_write_dap_table
			dec		r24
			brne		swd32_init_3a

			;debug core start
			ldi		r24,3
swd32_init_3b:		rcall		swd32_write_dap_table
			dec		r24
			brne		swd32_init_3b
		
			;set reset vector catch
			ldi		r24,3
swd32_init_3c:		rcall		swd32_write_dap_table
			dec		r24
			brne		swd32_init_3c
			
;			rcall		swd32_read_drwx

			sbi		CTRLPORT,SWD32_RST	;release reset

			ldi		ZL,250
			ldi		ZH,0
			call		api_wait_ms

			rcall		swd32_reginit		;set registers for faster output

			jmp		main_loop_ok
		
swd32_init_err:		jmp		main_loop

swd32_data_init1:	;DebugPortStart
			.db SWD32_WRITE_ABORT,	0x00,	0x00,0x00,0x00,0x1e	;clear all errors
			.db SWD32_WRITE_SELECT,	0x00,	0x00,0x00,0x00,0x00	;switch to Bank 0x00			
			.db SWD32_WRITE_CTRL,	0x00,	0x50,0x00,0x00,0x00	;power up debug interface
			.db SWD32_WRITE_CTRL,	0x00,	0x54,0x00,0x00,0x00	;request debug reset
			.db SWD32_WRITE_CTRL,	0x00,	0x50,0x00,0x0F,0x00	;init AP transfer mode

			;DebugCoreStart
			.db SWD32_WRITE_CSW,	0x00,	0x23,0x00,0x00,0x02	;32 bit access
			.db SWD32_WRITE_TAR,	0x00,	0xE0,0x00,0xED,0xF0	;DHCSR
			.db SWD32_WRITE_DRW,	0x00,	0xA0,0x5F,0x00,0x09	;halt CPU and enable debug
			
			;set reset vector catch
			.db SWD32_WRITE_TAR,	0x00,	0xE0,0x00,0xED,0xFC	;DEMCR
			.db SWD32_WRITE_DRW,	0x00,	0x01,0x00,0x00,0x01	;enable reset vector catch
			.db SWD32_WRITE_DRW,	0x00,	0x01,0x00,0x00,0x01	;enable reset vector catch		

;-------------------------------------------------------------------------------
; reset and goto idle state
;-------------------------------------------------------------------------------
swd32_reset:		sbi		CTRLPORT,SWD32_DATA
			sbi		CTRLDDR,SWD32_DATA
			;min 50 clocks with TMS high
			ldi		XH,56
swd32_reset_1:		out		CTRLPORT,r14		;one
			dec		XH
			out		CTRLPIN,r15		;clock inactive
			brne		swd32_reset_1

			cpi		r18,0xde		;skip switch to SWD mode if not needed 
			breq		swd32_reset_nsw

			rcall		swd32_switch_swd

swd32_reset_nsw:
			;goto run-test-idle
			cbi		CTRLPORT,SWD32_DATA
			ldi		XH,16
swd32_reset_4:		out		CTRLPORT,r13		;zero
			dec		XH
			out		CTRLPIN,r15		;clock inactive
			brne		swd32_reset_4
			ret

			;switch to SWD mode
swd32_switch_swd:	ldi		XH,16
			ldi		ZH,0xE7			;switch code (LSB first)
			ldi		ZL,0x9E
swd32_switch_swd_2:	mov		XL,r14			;one
			sbrs		ZL,0
			mov		XL,r13			;zero
			out		CTRLPORT,XL
			lsr		ZH
			ror		ZL
			out		CTRLPIN,r15		;clock inactive
			dec		XH
			brne		swd32_switch_swd_2
			ret


;-------------------------------------------------------------------------------
; write
; P1-P3 = Address
; P4=pages
;-------------------------------------------------------------------------------
swd32_write:		mov		r25,r16			;address l
			mov		r6,r17			;address m
			mov		r7,r18			;address h
			ldi		YL,0
			ldi		YH,1

swd32_write_1:		clr		r24
swd32_write_2:		movw		r20,r24
			movw		r22,r6
			ldi		XL,SWD32_WRITE_TAR
			rcall		swd32_write_dap
			
			call		gen_r32

			ldi		XL,SWD32_WRITE_DRW
			rcall		swd32_write_dap
			
			subi		r24,0xfc		;+4
			brne		swd32_write_2

			add		r25,const_1
			adc		r6,const_0
			adc		r7,const_0

			dec		r19
			brne		swd32_write_1
			
			sbiw		YL,4			;set back pointer
			call		gen_r32

			ldi		XL,SWD32_WRITE_DRW
			rcall		swd32_write_dap		;write last word again
			
			jmp		main_loop_ok


;-------------------------------------------------------------------------------
; fast write
; P1-P3 = Address
; P4=pages
;-------------------------------------------------------------------------------
swd32_fwrite:	;	rjmp		swd32_write
			ldi		YL,0
			ldi		YH,1
			mov		r25,r19
		
swd32_fwrite_1:		clr		r20
			mov		r21,r16			;address l
			mov		r22,r17			;address m
			mov		r23,r18			;address h
			ldi		XL,SWD32_WRITE_TAR
			rcall		swd32_write_dap


			ldi		r24,0x40
swd32_fwrite_2:		call		gen_r32
			ldi		XL,SWD32_WRITE_DRW
			rcall		swd32_write_dap
			
			dec		r24
			brne		swd32_fwrite_2
			
			add		r16,const_1
			adc		r17,const_0			
			adc		r18,const_0			
			
			dec		r25
			brne		swd32_fwrite_1

			jmp		main_loop_ok


;-------------------------------------------------------------------------------
; read
; P1-P3 = Address
; P4=pages
;-------------------------------------------------------------------------------
swd32_read:		mov		r25,r16			;address l
			mov		r6,r17			;address m
			mov		r7,r18			;address h
			ldi		YL,0
			ldi		YH,1
			
swd32_read_1:		clr		r24
swd32_read_2:		movw		r20,r24
			movw		r22,r6
			ldi		XL,SWD32_WRITE_TAR
			rcall		swd32_write_dap

			rcall		swd32_read_drwx		;dummy value
			st		Y+,r20
			st		Y+,r21
			st		Y+,r22
			st		Y+,r23
			
			subi		r24,0xfc		;+4
			brne		swd32_read_2
			
			add		r25,const_1
			adc		r6,const_0
			adc		r7,const_0

			dec		r19
			brne		swd32_read_1
			jmp		main_loop_ok

;-------------------------------------------------------------------------------
; fast read with incremented address
; P1-P3 = Address
; P4=pages
;-------------------------------------------------------------------------------
swd32_fread:	;	rjmp		swd32_read
			ldi		YL,0
			ldi		YH,1
			mov		r25,r19
		
swd32_fread_1:		clr		r20
			mov		r21,r16			;address l
			mov		r22,r17			;address m
			mov		r23,r18			;address h
			ldi		XL,SWD32_WRITE_TAR
			rcall		swd32_write_dap

			ldi		XL,SWD32_READ_DRW
			rcall		swd32_read_dap		;first dummy read
			
			ldi		r24,0x3f

swd32_fread_2:		ldi		XL,SWD32_READ_DRW
			rcall		swd32_read_dap		;data read
			call		gen_w32
		
			dec		r24
			brne		swd32_fread_2
			
			ldi		XL,SWD32_READ_RDBUFF
			rcall		swd32_read_dap		;last value read
			call		gen_w32
			
			add		r16,const_1
			adc		r17,const_0			
			adc		r18,const_0			
			
			dec		r25
			brne		swd32_fread_1

			jmp		main_loop_ok


;-------------------------------------------------------------------------------
; nread
; P1-P4 = Address
; byte 0=num
;-------------------------------------------------------------------------------
swd32_nread:		ldi		YL,0
			ldi		YH,1
			ld		r24,Y
			
swd32_nread_1:		movw		r20,r16
			movw		r22,r18
			ldi		XL,SWD32_WRITE_TAR
			rcall		swd32_write_dap

			rcall		swd32_read_drwx		;dummy value
			st		Y+,r20
			st		Y+,r21
			st		Y+,r22
			st		Y+,r23
			
			ldi		XL,4
			add		r16,XL
			adc		r17,const_0
			adc		r18,const_0
			adc		r19,const_0
			
			dec		r24			;lw counter
			brne		swd32_nread_1
			jmp		main_loop_ok


;-------------------------------------------------------------------------------
; write long
;-------------------------------------------------------------------------------
swd32_wlong:		sbi		CTRLPORT,SWD32_TRIGGER	
			cbi		CTRLPORT,SWD32_TRIGGER	
			rcall		swd32_wvdata
			sts		0x100,XL
			jmp		main_loop_ok

swd32_wword:		ldi		ZL,LOW(swd32_wword_data*2)
			ldi		ZH,HIGH(swd32_wword_data*2)
			rcall		swd32_read_ctrlstat			
			rcall		swd32_write_dap_table
			rcall		swd32_wvdata
			rcall		swd32_write_dap_table
			sts		0x100,XL
			jmp		main_loop_ok

swd32_wbyte:		ldi		ZL,LOW(swd32_wbyte_data*2)
			ldi		ZH,HIGH(swd32_wbyte_data*2)
			rcall		swd32_read_ctrlstat			
			rcall		swd32_write_dap_table
			rcall		swd32_wvdata
			rcall		swd32_write_dap_table
			sts		0x100,XL
			jmp		main_loop_ok


swd32_wvdata:		ldi		YL,0
			ldi		YH,1

			ldi		XL,SWD32_WRITE_TAR
			ld		r20,Y+
			ld		r21,Y+
			ld		r22,Y+
			ld		r23,Y+			
			rcall		swd32_write_dap

			ldi		XL,SWD32_WRITE_DRW
			ld		r20,Y+
			ld		r21,Y+
			ld		r22,Y+
			ld		r23,Y+
			rjmp		swd32_write_dap



swd32_wword_data:	.db SWD32_WRITE_CSW,	0x00,	0x23,0x00,0x00,0x01	;16 bit access
			.db SWD32_WRITE_CSW,	0x00,	0x23,0x00,0x00,0x02	;32 bit access
swd32_wbyte_data:	.db SWD32_WRITE_CSW,	0x00,	0x23,0x00,0x00,0x00	;8 bit access
			.db SWD32_WRITE_CSW,	0x00,	0x23,0x00,0x00,0x02	;32 bit access


;-------------------------------------------------------------------------------
; go
; P1-P4 = Address
;-------------------------------------------------------------------------------
swd32_cont:		ldi		YL,0
			ldi		YH,1
			rjmp		swd32_go_1

swd32_go:		ldi		YL,0
			ldi		YH,1

			;write SP/PC	
			rcall		swd32_wsp		;write stack pointer			
swd32_go_1:		rcall		swd32_wpc		;write pc			
			rcall		swd32_readregs
			
swd32_sgo:		ldi		ZL,LOW(swd32_data_go*2)
			ldi		ZH,HIGH(swd32_data_go*2)
			ldi		r24,3
swd32_go_3:		rcall		swd32_write_dap_table
			dec		r24
			brne		swd32_go_3

			jmp		main_loop_ok			
			
swd32_data_go:		.db SWD32_WRITE_TAR,	0x00,	0xE0,0x00,0xED,0xF0	;DHCSR
			.db SWD32_WRITE_DRW,	0x00,	0xA0,0x5F,0x00,0x09	;run CPU (clear halt)
			.db SWD32_WRITE_DRW,	0x00,	0xA0,0x5F,0x00,0x09	;run CPU (clear halt)

;-------------------------------------------------------------------------------
; prepare
; P1-P4 = Address
;-------------------------------------------------------------------------------
swd32_prepare:		ldi		YL,0
			ldi		YH,1

			;write SP/PC	
			rcall		swd32_wsp		;write stack pointer			
			rcall		swd32_wpc		;write pc			
			rcall		swd32_readregs
			
			jmp		main_loop_ok
			
;-------------------------------------------------------------------------------
; unlocking and mass erase
;-------------------------------------------------------------------------------
swd32_exit_debug:	ldi		ZL,LOW(swd32_data_exit*2)
			ldi		ZH,HIGH(swd32_data_exit*2)
			ldi		r24,3
swd32_exit_1:		rcall		swd32_write_dap_table
			dec		r24
			brne		swd32_exit_1

			cbi		CTRLPORT,SWD32_RST	;set reset

			rcall		swd32_wait_1ms
			
			sbi		CTRLPORT,SWD32_RST	;set reset
			
			jmp		main_loop_ok			


swd32_data_exit:	.db SWD32_WRITE_TAR,	0x00,	0xE0,0x00,0xED,0xF0	;DHCSR
			.db SWD32_WRITE_DRW,	0x00,	0xA0,0x5F,0x00,0x00	;run CPU (clear halt+debug)
			.db SWD32_WRITE_DRW,	0x00,	0xA0,0x5F,0x00,0x00	;run CPU (clear halt+debug)
		
;-------------------------------------------------------------------------------
; flash command etc
;-------------------------------------------------------------------------------
swd32_cmd3:		ldi		r20,0x00		;cmd word is at 0x20400C00
			ldi		r21,0x0c
			ldi		r22,0x40
			ldi		r23,0x20
			rjmp		swd32_cmd_0

swd32_cmd2:		ldi		r20,0x00		;cmd word is at 0x18000c00
			ldi		r21,0x0c
			ldi		r22,0x00
			ldi		r23,0x18
			rjmp		swd32_cmd_0

swd32_cmd1:		ldi		r20,0x00		;cmd word is at 0x20000000
			ldi		r21,0x00
			ldi		r22,0x00
			ldi		r23,0x20
			rjmp		swd32_cmd_0
			
swd32_cmd:		ldi		r20,0x00		;cmd word is at 0x20000c00
			ldi		r21,0x0c
			ldi		r22,0x00
			ldi		r23,0x20
			rjmp		swd32_cmd_0

swd32_cmd_0:		call		save_w32		;store address
			ldi		XL,SWD32_WRITE_TAR
			rcall		swd32_write_dap
			
			movw		r20,r16			;store data
			movw		r22,r18
			ldi		XL,SWD32_WRITE_DRW
			rcall		swd32_write_dap
			rcall		swd32_read_drwx		;dummy readout

			ldi		r24,0
			ldi		r25,8			;shorten??? (was 20)

swd32_cmd_1:		sbiw		r24,1
			breq		swd32_cmd_to
			ldi		ZL,1
			ldi		ZH,0
			call		api_wait_ms

			call		restore_r32		;restore address
			ldi		XL,SWD32_WRITE_TAR
			rcall		swd32_read_drwx		;readout

			cpi		r20,0x00		;check lowest byte
			brne		swd32_cmd_1		;wait until cmd word is zero
			call		gen_wres		;write result to memory
			jmp		main_loop_ok

swd32_cmd_to:		ldi		r16,0x41
			jmp		main_loop

;-------------------------------------------------------------------------------
; write stack pointer value
;-------------------------------------------------------------------------------
swd32_wsp:		adiw		YL,4
			ldi		r25,2
swd32_wsp_1:		ldi		ZL,LOW(swd32_data_wsp*2)
			ldi		ZH,HIGH(swd32_data_wsp*2)
			sbiw		YL,4

			ldi		r24,3
swd32_wsp_2:		rcall		swd32_write_dap_table
			dec		r24
			brne		swd32_wsp_2

			ld		r20,Y+			;get stored SP value
			ld		r21,Y+
			ld		r22,Y+
			ld		r23,Y+

			ldi		XL,SWD32_WRITE_DRW	;write SP
			rcall		swd32_write_dap

			dec		r25
			brne		swd32_wsp_1
			ret

swd32_data_wsp:		.db SWD32_WRITE_TAR,	0x00,	0xE0,0x00,0xED,0xF4	;DCRSR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x01,0x00,0x0D	;write, r13=SP
			.db SWD32_WRITE_TAR,	0x00,	0xE0,0x00,0xED,0xF8	;DCRDR

;-------------------------------------------------------------------------------
; write PC value
;-------------------------------------------------------------------------------
swd32_wpc:		adiw		YL,4
			ldi		r25,2
swd32_wpc_1:		ldi		ZL,LOW(swd32_data_wpc*2)
			ldi		ZH,HIGH(swd32_data_wpc*2)
			sbiw		YL,4

			ldi		r24,3
swd32_wpc_2:		rcall		swd32_write_dap_table
			dec		r24
			brne		swd32_wpc_2

			ld		r20,Y+			;get stored PC value
			ld		r21,Y+
			ld		r22,Y+
			ld		r23,Y+

			ldi		XL,SWD32_WRITE_DRW	;write PC
			rcall		swd32_write_dap

			dec		r25
			
			brne		swd32_wpc_1
			ret

swd32_data_wpc:		.db SWD32_WRITE_TAR,	0x00,	0xE0,0x00,0xED,0xF4	;DCRSR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x01,0x00,0x0F	;write, r15=PC/debug return address
			.db SWD32_WRITE_TAR,	0x00,	0xE0,0x00,0xED,0xF8	;DCRDR


;-------------------------------------------------------------------------------
; write Rx value
; P4=num
;-------------------------------------------------------------------------------
swd32_wreg:		ldi		r25,2

swd32_wreg_1:		ldi		YL,0
			ldi		YH,1
			ldi		ZL,LOW(swd32_data_wreg*2)
			ldi		ZH,HIGH(swd32_data_wreg*2)

			rcall		swd32_write_dap_table

			ldi		XL,SWD32_WRITE_DRW	;write REG ADDR
			rcall		swd32_dnext

			rcall		swd32_write_dap_table

			ldi		XL,SWD32_WRITE_DRW	;write REG
			rcall		swd32_dnext

			dec		r25
			
			brne		swd32_wreg_1
			jmp		main_loop_ok
			

swd32_data_wreg:	.db SWD32_WRITE_TAR,	0x00,	0xE0,0x00,0xED,0xF4	;DCRSR
			.db SWD32_WRITE_TAR,	0x00,	0xE0,0x00,0xED,0xF8	;DCRDR


;-------------------------------------------------------------------------------
; read registers for debug
; r19 bit 4 = do not read data at PC
;-------------------------------------------------------------------------------
swd32_readregs:		ldi		YL,0
			ldi		YH,1
			
			ldi		r24,0
swd32_readregs_1:	ldi		ZL,LOW(swd32_data_rregs*2)
			ldi		ZH,HIGH(swd32_data_rregs*2)
			rcall		swd32_write_dap_table			;DCRSR
			ldi		XL,SWD32_WRITE_DRW
			clr		r23
			clr		r22
			clr		r21
			mov		r20,r24
			rcall		swd32_write_dap			
			
			rcall		swd32_write_dap_table			;DCRDR
			rcall		swd32_read_to_buf			;read data

			inc		r24
			cpi		r24,0x10
			brne		swd32_readregs_1

			sbiw		YL,4
			sbrc		r19,4
			jmp		main_loop_ok

			;now read data at PC
			ld		r20,Y+
			ld		r21,Y+
			ld		r22,Y+
			ld		r23,Y+
			
			andi		r20,0xfc
			push		r20
			push		r21
			push		r22
			push		r23
			ldi		XL,SWD32_WRITE_TAR
			rcall		swd32_write_dap			
			rcall		swd32_read_to_buf			;read data

			pop		r23
			pop		r22
			pop		r21
			pop		r20

			ldi		XL,4
			add		r20,XL
			adc		r21,const_0
			adc		r22,const_0
			adc		r23,const_0
			ldi		XL,SWD32_WRITE_TAR
			rcall		swd32_write_dap			
			rcall		swd32_read_to_buf			;read data
		
			jmp		main_loop_ok
		
swd32_data_rregs:	.db SWD32_WRITE_TAR,	0x00,	0xE0,0x00,0xED,0xF4	;DCRSR
			.db SWD32_WRITE_TAR,	0x00,	0xE0,0x00,0xED,0xF8	;DCRDR
				
;-------------------------------------------------------------------------------
; single step for debug
;-------------------------------------------------------------------------------
swd32_step:		ldi		XL,SWD32_WRITE_TAR
			ldi		r23,0xE0
			ldi		r22,0x00
			ldi		r21,0xED
			ldi		r20,0xF0		;step
			rcall		swd32_write_dap
		
			ldi		XL,SWD32_WRITE_DRW
			ldi		r23,0xA0
			ldi		r22,0x5F
			ldi		r21,0x00
			ldi		r20,0x0D		;step
			rcall		swd32_write_dap

swd32_step_wait:	rcall		swd32_read_drwx		;read
			andi		r20,0x02		;halt bit
			breq		swd32_step_wait						
									
			rjmp		swd32_readregs		;done		

swd32_read_to_buf:	rcall		swd32_read_drwx2	;read
swd32_read_to_buf1:	st		Y+,r20
			st		Y+,r21
			st		Y+,r22
			st		Y+,r23
			ret

swd32_readback:		ldi		YL,0
			ldi		YH,1
			rjmp		swd32_read_to_buf	;get version


swd32_halt:		ldi		XL,SWD32_WRITE_TAR
			ldi		r23,0xE0
			ldi		r22,0x00
			ldi		r21,0xED
			ldi		r20,0xF0		;step
			rcall		swd32_write_dap
		
			ldi		XL,SWD32_WRITE_DRW
			ldi		r23,0xA0
			ldi		r22,0x5F
			ldi		r21,0x00
			ldi		r20,0x0B		;step
			rcall		swd32_write_dap

			rjmp	swd32_step_wait

;-------------------------------------------------------------------------------
; helper functions
;-------------------------------------------------------------------------------
swd32_wdwt:		ldi		XL,SWD32_WRITE_TAR
			ldi		r23,0xE0		;DWT
			ldi		r22,0x00
			ldi		r21,0x10
			mov		r20,r24
			rjmp		swd32_write_dap

swd32_wzero:		ldi		XL,SWD32_WRITE_DRW
			ldi		r23,0x00
			ldi		r22,0x00
			ldi		r21,0x00
			ldi		r20,0x00
			rjmp		swd32_write_dap

swd32_dnext:		ld		r20,Y+			;get stored PC value
			ld		r21,Y+
			ld		r22,Y+
			ld		r23,Y+
			rjmp		swd32_write_dap


;-------------------------------------------------------------------------------
; set breakpoint 0 for RAM
; PAR = ADDR
;-------------------------------------------------------------------------------
swd32_setbrk:		ldi		r24,0x00
			rcall		swd32_wdwt
			ldi		XL,SWD32_WRITE_DRW
			ldi		r23,0x00
			ldi		r22,0x00
			ldi		r21,0x00
			ldi		r20,0x01
			rcall		swd32_write_dap
			rcall		swd32_readback

			ldi		r24,0x20
			rcall		swd32_wdwt

			ldi		XL,SWD32_WRITE_DRW
			mov		r23,r19
			mov		r22,r18
			mov		r21,r17
			mov		r20,r16
			rcall		swd32_write_dap
			rcall		swd32_read_to_buf

			ldi		r24,0x24
			rcall		swd32_wdwt
			rcall		swd32_wzero
			rcall		swd32_read_to_buf
			
			ldi		r24,0x28
			rcall		swd32_wdwt
			ldi		XL,SWD32_WRITE_DRW
			ldi		r23,0x00
			ldi		r22,0x00
			ldi		r21,0x00
			ldi		r20,0x04
			rcall		swd32_write_dap
			rcall		swd32_read_to_buf

			jmp		main_loop_ok

;-------------------------------------------------------------------------------
; clear breakpoint 0
;-------------------------------------------------------------------------------
swd32_clrbrk:		ldi		r24,0x28
			rcall		swd32_wdwt
			rcall		swd32_wzero
			
			jmp		main_loop_ok
			

;-------------------------------------------------------------------------------
; check if in debug mode (bit 17 = halt)
;-------------------------------------------------------------------------------
swd32_dbgcheck:		ldi		XL,SWD32_WRITE_TAR
			ldi		r23,0xE0		;DHCSR
			ldi		r22,0x00
			ldi		r21,0xED
			ldi		r20,0xF0
			rcall		swd32_write_dap
			
			rcall		swd32_readback
			jmp		main_loop_ok

;-------------------------------------------------------------------------------
; SWD execute commands 
; par4 = num of chunks
;-------------------------------------------------------------------------------
swd32_swdexec: 		ldi		YL,0			;reset buffer pointer
			ldi		YH,1
			mov		r24,r19

swd32_swdexec_1:	ld		XL,Y+			;command
			ld		r23,Y+
			ld		r22,Y+
			ld		r21,Y+
			ld		r20,Y+
			rcall		swd32_write_dap
			sbrc		r18,1
			rcall		swd32_wait_1ms		
			dec		r24
			brne		swd32_swdexec_1
			
			sbrc		r18,1
			rjmp		swd32_swdexec_2

			ldi		ZL,10			;wait 1ms
			ldi		ZH,0
			call		api_wait_ms
			rcall		swd32_read_drwx		;dummy read SR

swd32_swdexec_2:	jmp		main_loop_ok

;-------------------------------------------------------------------------------
; SWD execute commands 
; par1+2 = timeout
; par3 = activate delay (bit 0)
; par4 = num of chunks
;-------------------------------------------------------------------------------
swd32_swdexec2: 	ldi		YL,0			;reset buffer pointer
			ldi		YH,1
			mov		r24,r19
			movw		r10,r16

swd32_swdexec2_1:	ld		XL,Y+			;command
			sbrs		XL,7
			rjmp		swd32_swdexec2_wh
			ld		r23,Y+
			ld		r22,Y+
			ld		r21,Y+
			ld		r20,Y+
			
swd32_swdexec2_c:	rcall		swd32_write_dap
			sbrc		r18,1
			rcall		swd32_wait_1ms		
swd32_swdexec2_nxt:	dec		r24
			brne		swd32_swdexec2_1
			jmp		main_loop_ok
			
swd32_swdexec2_wh:	push		r24
			movw		r24,r10			;get timeout
			cpi		XL,0x55			;wait for high
			brne		swd32_swdexec2_wl
		
swd32_swdexec2_wh1:	rcall		swd32_read_drwx		;dummy read SR
			ld		XL,Y			;byte 3
			and		XL,r23
			brne		swd32_swdexec2_wh3
			ldd		XL,Y+1			;byte 2
			and		XL,r22
			brne		swd32_swdexec2_wh3
			ldd		XL,Y+2			;byte 1
			and		XL,r21
			brne		swd32_swdexec2_wh3
			ldd		XL,Y+3			;byte 0
			and		XL,r20
			brne		swd32_swdexec2_wh3
			
			rcall		swd32_wait_1ms		;wait 1ms	
			sbiw		r24,1			;timeout?
			brne		swd32_swdexec2_wh1
			
swd32_swdexec2_wh2:	pop		r24
			ldi		r16,0x42		;timeout
			jmp		main_loop
			
swd32_swdexec2_wh3:	adiw		YL,4
			pop		r24
			rjmp		swd32_swdexec2_nxt				

swd32_swdexec2_wl:	cpi		XL,0x44			;wait for low
			brne		swd32_swdexec2_ro

swd32_swdexec2_wl1:	rcall		swd32_read_drwx		;dummy read SR
			ld		XL,Y			;byte 3
			com		XL
			and		XL,r23
			brne		swd32_swdexec2_wh3
			ldd		XL,Y+1			;byte 2
			com		XL
			and		XL,r22
			brne		swd32_swdexec2_wh3
			ldd		XL,Y+2			;byte 1
			com		XL
			and		XL,r21
			brne		swd32_swdexec2_wh3
			ldd		XL,Y+3			;byte 0
			com		XL
			and		XL,r20
			brne		swd32_swdexec2_wh3
			
			rcall		swd32_wait_1ms		
			sbiw		r24,1
			brne		swd32_swdexec2_wl1		

			rjmp		swd32_swdexec2_wh2

swd32_swdexec2_ro:	rcall		swd32_wait_1ms		
			rcall		swd32_read_drwx		;read current address
			jmp		main_loop_ok



;-------------------------------------------------------------------------------
; SWD execute commands 
; par4 = bit 0=reset
;-------------------------------------------------------------------------------
swd32_swdexec3_endok:	jmp		main_loop_ok


swd32_swdexec3: 	ldi		YL,0			;reset buffer pointer
			ldi		YH,1
			movw		ZL,YL			;copy
			sbrc		r19,0
			rcall		swd32_reginit
			sbrs		r19,0
			rcall		swd32_reginit_reset
			sbrc		r19,0				
			sbi		CTRLPORT,SWD32_RST
			sbrs		r19,0
			cbi		CTRLPORT,SWD32_RST

			
swd32_swdexec3_loop:	ld		XL,Z+			;command
			cpi		XL,0xFF			;end?
			breq		swd32_swdexec3_endok

			cpi		XL,0xFE			;pause
			brne		swd32_swdexec3_1

			ld		r20,Z+			;time low
			ld		r21,Z+			;time high
			adiw		ZL,2			;skip next bytes
			movw		r22,ZL
			movw		ZL,r20
			call		api_wait_ms
			movw		ZL,r22
			rjmp		swd32_swdexec3_loop
			
swd32_swdexec3_1:	sbrc		XL,5
			rjmp		swd32_swdexec3_rdap

			ld		r23,Z+
			ld		r22,Z+
			ld		r21,Z+
			ld		r20,Z+
			rcall		swd32_write_dap
			cpi		XL,0x04
			breq		swd32_swdexec3_loop
			ldi		r16,0x50
			add		r16,XL
			jmp		main_loop		;errcode = 0x50 + code			


swd32_swdexec3_rdap:	adiw		ZL,4			;ignore data
			rcall		swd32_read_dap
			call		gen_w32
			rjmp		swd32_swdexec3_loop
			
			

;-------------------------------------------------------------------------------
; erase STM32 F0/F1/F2/F3/F4
;-------------------------------------------------------------------------------
swd32_erase3:		ldi		ZL,LOW(swd32_data_erase3*2)
			ldi		ZH,HIGH(swd32_data_erase3*2)
			ldi		r24,14
			rjmp		swd32_erase2_1
			
swd32_erase4:		ldi		ZL,LOW(swd32_data_erase4*2)
			ldi		ZH,HIGH(swd32_data_erase4*2)
			ldi		r24,14
			rjmp		swd32_erase2_1

swd32_erase5:		ldi		ZL,LOW(swd32_data_erase5*2)
			ldi		ZH,HIGH(swd32_data_erase5*2)
			ldi		r24,12
			rjmp		swd32_erase1_1

swd32_erase6:		ldi		ZL,LOW(swd32_data_erase6*2)
			ldi		ZH,HIGH(swd32_data_erase6*2)
			ldi		r24,14
			rjmp		swd32_erase1_1

swd32_erase1:		cpi		r16,1
			breq		swd32_erase2
			cpi		r16,2
			breq		swd32_erase3
			cpi		r16,4
			breq		swd32_erase4
			cpi		r16,5
			breq		swd32_erase5
			cpi		r16,6
			breq		swd32_erase6
						
			
			ldi		ZL,LOW(swd32_data_erase1*2)
			ldi		ZH,HIGH(swd32_data_erase1*2)

			ldi		r24,12
swd32_erase1_1:		rcall		swd32_write_dap_table
			dec		r24
			brne		swd32_erase1_1
			
			ldi		ZL,0
			ldi		ZH,4
			call		api_wait_ms
			
swd32_erase1_2:		rcall		swd32_read_drwx		;read SR
			andi		r20,0x01
			brne		swd32_erase1_2
			ldi		XL,0x55			
			sts		0x100,XL
			sts		0x101,XL
			sts		0x102,XL
			sts		0x103,XL
			jmp		main_loop_ok

			
swd32_erase2:		ldi		ZL,LOW(swd32_data_erase2*2)
			ldi		ZH,HIGH(swd32_data_erase2*2)

			ldi		r24,14
swd32_erase2_1:		rcall		swd32_write_dap_table
			dec		r24
			brne		swd32_erase2_1
			mov		r24,r19			;num of loops
			
swd32_erase2_2:		dec		r24
			brne		swd32_erase2_3
			ldi		ZL,100			;wait 100ms
			ldi		ZH,0
			call		api_wait_ms
			rcall		swd32_read_drwx		;read SR
			mov		r16,r22
			andi		r22,0x01
			brne		swd32_erase2_2		
			sts		0x100,r20
			sts		0x101,r21
			sts		0x102,r16
			sts		0x103,r23
			jmp		main_loop_ok
			
swd32_erase2_3:		ldi		r16,0x88
			jmp		main_loop			


swd32_erase_chk:	ldi		ZL,0
			ldi		ZH,2
			call		api_wait_ms
			rcall		swd32_read_drwx2		;read SR
			sts		0x100,r20
			sts		0x101,r21
			sts		0x102,r22
			sts		0x103,r23
			cpi		XL,4
			brne		swd32_erase_chke
			jmp		main_loop_ok
swd32_erase_chke:	ldi		r16,0xa0
			jmp		main_loop			

			
swd32_data_erase1:	.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x00,0x00	;clear all
		
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x04	;KEYR
			.db SWD32_WRITE_DRW,	0x00,	0x45,0x67,0x01,0x23	;key 1
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x04	;KEYR
			.db SWD32_WRITE_DRW,	0x00,	0xCD,0xEF,0x89,0xAB	;key 2
			
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x00,0x04	;MER
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x00,0x44	;MER+STRT
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x00,0x44	;MER+STRT
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x0C	;SR
			

swd32_data_erase2:	.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x00,0x00	;clear all
		
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x04	;KEYR
			.db SWD32_WRITE_DRW,	0x00,	0x45,0x67,0x01,0x23	;key 1
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x04	;KEYR
			.db SWD32_WRITE_DRW,	0x00,	0xCD,0xEF,0x89,0xAB	;key 2
			
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x02,0x00	;x32
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x02,0x04	;MER, x32
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x01,0x02,0x04	;MER+STRT
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x01,0x02,0x04	;MER+STRT
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x0C	;SR


swd32_data_erase3:	.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x00,0x00	;clear all
		
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x04	;KEYR
			.db SWD32_WRITE_DRW,	0x00,	0x45,0x67,0x01,0x23	;key 1
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x04	;KEYR
			.db SWD32_WRITE_DRW,	0x00,	0xCD,0xEF,0x89,0xAB	;key 2
			
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x02,0x00	;x32
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x82,0x04	;MER/MER1, x32
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x01,0x82,0x04	;MER/MER1+STRT
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x01,0x82,0x04	;MER/MER1+STRT
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x0C	;SR


swd32_data_erase4:	.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x14	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x00,0x00	;clear all
		
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x08	;KEYR
			.db SWD32_WRITE_DRW,	0x00,	0x45,0x67,0x01,0x23	;key 1
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x08	;KEYR
			.db SWD32_WRITE_DRW,	0x00,	0xCD,0xEF,0x89,0xAB	;key 2
			
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x14	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x02,0x00	;x32
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x14	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x80,0x04	;MER/MER1
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x14	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x01,0x80,0x04	;MER/MER1+STRT
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x01,0x80,0x04	;MER/MER1+STRT
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x10	;SR

swd32_data_erase5:	.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x00,0x00	;clear all
		
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x08	;OPTKEYR
			.db SWD32_WRITE_DRW,	0x00,	0x45,0x67,0x01,0x23	;key 1
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x08	;OPTKEYR
			.db SWD32_WRITE_DRW,	0x00,	0xCD,0xEF,0x89,0xAB	;key 2
			
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x00,0x20	;OPTER
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x00,0x60	;OPTER+STRT
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x00,0x60	;OPTER+STRT
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x20,0x0C	;SR


swd32_data_erase6:	.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x00,0x00	;clear all
		
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x04	;KEYR
			.db SWD32_WRITE_DRW,	0x00,	0x45,0x67,0x01,0x23	;key 1
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x04	;KEYR
			.db SWD32_WRITE_DRW,	0x00,	0xCD,0xEF,0x89,0xAB	;key 2
			
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x02,0x00	;x32
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x00,0x02,0x04	;MER/MER1
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x10	;CR
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x01,0x02,0x04	;MER/MER1+STRT
			.db SWD32_WRITE_DRW,	0x00,	0x00,0x01,0x02,0x04	;MER/MER1+STRT
			.db SWD32_WRITE_TAR,	0x00,	0x40,0x02,0x3C,0x0C	;SR

;-------------------------------------------------------------------------------
; header
; XL=config in
; XL=ack out
;-------------------------------------------------------------------------------
swd32_head:		ldi		XH,8			;bits to do
swd32_head_1:	;	sbrc		XL,7
			mov		r12,r14			;one
			sbrs		XL,7
			mov		r12,r13			;zero
			
			out		CTRLPORT,r12
			lsl		XL
			dec		XH
			out		CTRLPIN,r15
			brne		swd32_head_1

			cbi		CTRLDDR,SWD32_DATA
			cbi		CTRLPORT,SWD32_CLOCK	;TRN
			nop
			nop
			sbi		CTRLPORT,SWD32_CLOCK

			ldi		XL,5
shd2:			dec		XL
			brne		shd2	

		
			;get ack
			clr		XL
			ldi		XH,3
swd32_head_2:		out		CTRLPORT,r14		;ONE (is pull-up)
			lsl		XL
			nop
			sbic		CTRLPIN,SWD32_DATA
			inc		XL
			out		CTRLPIN,r15		;CLOCK
			dec		XH
			brne		swd32_head_2
			ret


;-------------------------------------------------------------------------------
; write fixed 32 bit data word (LSB first)
; (Z+0)=config in
; XL=ack out
; (Z+2...Z+3) data in
;-------------------------------------------------------------------------------
swd32_write_dap_table:	lpm		XL,Z+		;CMD
			adiw		ZL,1		;skip additional byte
			lpm		r23,Z+		;data
			lpm		r22,Z+
			lpm		r21,Z+
			lpm		r20,Z+

;-------------------------------------------------------------------------------
; write 32 bit data word (LSB first)
; XL=config in
; XL=ack out
; r20-r23 data in
;-------------------------------------------------------------------------------
swd32_write_dap:	push		r5
			push		r9
			clr		r5
			mov		r9,XL
swd32_write_dap_0:	rcall		swd32_head		;send header
	
			;TrN switch to output
			out		CTRLPORT,r14		;ONE (is pull-up)
			sbi		CTRLDDR,SWD32_DATA
			out		CTRLPIN,r15		;CLOCK

			cpi		XL,0x02			;WAIT
			brne		swd32_wd_0
	
			mov		XL,r9
			dec		r5
			brne		swd32_write_dap_0	; again


swd32_wd_0:		clr		r4			;parity
			ldi		XH,32			;bits to do
swd32_wd_1:		mov		r12,r14			;one
			sbrs		r20,0
			mov		r12,r13			;zero
			out		CTRLPORT,r12
			lsr		r23
			ror		r22
			ror		r21
			ror		r20
			out		CTRLPIN,r15	
			adc		r4,const_0
			dec		XH
			brne		swd32_wd_1
			;now send parity bit
			mov		r12,r14			;one
			sbrs		r4,0
			mov		r12,r13			;zero
			out		CTRLPORT,r12
			nop
			nop
			nop
			nop
			pop		r5
			pop		r9
			out		CTRLPIN,r15	
			ret

;-------------------------------------------------------------------------------
; read
; XL=config in
; XL=ack out
; r20-r23 data out
;-------------------------------------------------------------------------------
swd32_read_drwx2:	ldi		XL,SWD32_READ_DRW
			clr		r5			;256 tries
			rcall		swd32_read_dap2		;first dummy read
			cpi		XL,2
			breq		swd32_read_dap2x
			ldi		XL,SWD32_READ_RDBUFF
			clr		r5			;256 tries
			
swd32_read_dap2:	rcall		swd32_wait_1ms
			mov		r9,XL
			rcall		swd32_head		;send header

			cpi		XL,0x02			;WAIT
			brne		swd32_rd_0
	
			;TrN switch to output
			out		CTRLPORT,r14		;ONE (is pull-up)
			sbi		CTRLDDR,SWD32_DATA
			out		CTRLPIN,r15		;CLOCK
	
			mov		XL,r9
			dec		r5
			brne		swd32_read_dap2		;read again
swd32_read_dap2x:	ldi		XL,0x02			;WAIT
			ret 			


swd32_read_drwx:	ldi		XL,SWD32_READ_DRW
			rcall		swd32_read_dap		;first dummy read
			ldi		XL,SWD32_READ_RDBUFF
			
swd32_read_dap:		mov		r9,XL
			rcall		swd32_head		;send header

			cpi		XL,0x02			;WAIT
			brne		swd32_rd_0
	
			;TrN switch to output
			out		CTRLPORT,r14		;ONE (is pull-up)
			sbi		CTRLDDR,SWD32_DATA
			out		CTRLPIN,r15		;CLOCK
	
			mov		XL,r9
			rjmp		swd32_read_dap		;read again
			ret 			

swd32_rd_0:		ldi		XH,32
swd32_rd_1:		out		CTRLPORT,r14		;ONE (is pull-up)
			lsr		r23
			ror		r22
			ror		r21
			ror		r20
			sbic		CTRLPIN,SWD32_DATA
			ori		r23,0x80
			out		CTRLPIN,r15		;CLOCK
			dec		XH
			brne		swd32_rd_1

			;ignore parity
			out		CTRLPORT,r14		;ONE (is pull-up)
			nop
			out		CTRLPIN,r15		;CLOCK

			;TrN switch to output
			out		CTRLPORT,r14		;ONE (is pull-up)
			sbi		CTRLDDR,SWD32_DATA
			out		CTRLPIN,r15		;CLOCK
			ret


;-------------------------------------------------------------------------------
; SWD read AP IDR
;-------------------------------------------------------------------------------
swd32_read_apid:	ldi		YL,0
			ldi		YH,1
			ldi		XL,SWD32_READ_DRW
			rcall		swd32_read_dap
			ldi		XL,0x77
			rjmp		swd32_store32

;-------------------------------------------------------------------------------
; SWD read TAR
;-------------------------------------------------------------------------------
swd32_rread_tar:	ldi		YL,0
			ldi		YH,1
			ldi		XL,SWD32_READ_TAR
			rcall		swd32_read_dap
			ldi		XL,SWD32_READ_TAR
			rcall		swd32_read_dap
			ldi		XL,0x77
			rjmp		swd32_store32

;-------------------------------------------------------------------------------
; SWD generic read
;-------------------------------------------------------------------------------
swd32_rlong:		ldi		YL,0
			ldi		YH,1
			movw		r20,r16
			movw		r22,r18			
			
			ldi		XL,S32K_WRITE_TAR
			rcall		swd32_write_dap

			rcall		swd32_read_drwx		;dummy value
swd32_store32:		st		Y+,r20
			st		Y+,r21
			st		Y+,r22
			st		Y+,r23
			st		Y+,XL			;store axk status
			
			jmp		main_loop_ok


;-------------------------------------------------------------------------------
; some wait routines
;-------------------------------------------------------------------------------
swd32_wait_1ms:		push	ZH
			push	ZL
			ldi	ZL,1
			clr	ZH
			call	api_wait_ms
			pop	ZL
			pop	ZH
			ret

swd32_w0:		ldi	ZL,33
swd32_w0_1:		dec	ZL
			brne	swd32_w0_1
swd32_w0_2:		ret


swd32_wait_5ms:		ldi	ZL,5
			ldi	ZH,0
			jmp	api_wait_ms


;-------------------------------------------------------------------------------
; define registers for faster output
;-------------------------------------------------------------------------------
swd32_reginit:		ldi		XL,SWD32_ZERO
			mov		r13,XL
			ldi		XL,SWD32_ONE
			mov		r14,XL
			ldi		XL,SWD32_CLK
			mov		r15,XL
			ret

swd32_reginit_reset:	ldi		XL,SWD32_ZERO_R
			mov		r13,XL
			ldi		XL,SWD32_ONE_R
			mov		r14,XL
			ldi		XL,SWD32_CLK
			mov		r15,XL
			ret

swd32_get_idcode:	ldi		XL,SWD32_READ_CTRL	;read CTRLSTAT
			rcall		swd32_read_dap
			


swd32_read_ctrlstat:	ldi		XL,SWD32_READ_CTRL	;read CTRLSTAT
			rjmp		swd32_read_dap


swd32_wait_ctrlstat:	movw		r0,XL			;mask ad target
			ldi		r24,0
			ldi		r25,0
swd32_wait_ctrlstat_1:	sbiw		r24,1
			breq		swd32_wait_ctrlstat_e
			ldi		XL,SWD32_READ_CTRL	;read CTRLSTAT
			rcall		swd32_read_dap		
			and		r23,r0
			cp		r23,r1
			brne		swd32_wait_ctrlstat_1		
swd32_wait_ctrlstat_e:	ret


swd32_get_register:	mov		XL,r19
			rcall		swd32_read_dap
			call		gen_wres
			jmp		main_loop_ok

