;################################################################################
;#										#
;# avr-chipbasic2 - single chip basic computer with ATmega644			#
;# EEPROM/Flash memory interface						#
;# copyright (c) 2006-2015 Joerg Wolfram (joerg@jcwolfram.de)			#
;#										#
;#										#
;# This program is free software; you can redistribute it and/or		#
;# modify it under the terms of the GNU General Public License			#
;# as published by the Free Software Foundation; either version 3		#
;# of the License, or (at your option) any later version.			#
;#										#
;# This program is distributed in the hope that it will be useful,		#
;# but WITHOUT ANY WARRANTY; without even the implied warranty of		#
;# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the GNU		#
;# General Public License for more details.					#
;#										#
;# You should have received a copy of the GNU General Public			#
;# License along with this library; if not, write to the			#
;# Free Software Foundation, Inc., 59 Temple Place - Suite 330,			#
;# Boston, MA 02111-1307, USA.							#
;#										#
;################################################################################

mem_getconf:	ldi	YL,LOW(2039)		;default -> config 
		ldi	YH,HIGH(2039)
		rcall	mem_eeread		;get value
		cpi	tempreg1,0xa5
		breq	mem_getconf1
mem_initconf_1:	sbiw	YL,1
		ldi	tempreg1,0xa5
		rcall	mem_eewrite
		ldi	tempreg1,0xe1
		rcall	mem_eewrite
		ldi	tempreg1,0x00
		rcall	mem_eewrite
		ldi	tempreg1,0x00
		rcall	mem_eewrite
		rjmp	mem_getconf1

;------------------------------------------------------------------------------
; write sysconf to eeprom
;------------------------------------------------------------------------------
mem_setconf:	ldi	YL,LOW(2040)		;default -> config 
		ldi	YH,HIGH(2040)
		mov	tempreg1,r18		;config register
		rcall	mem_eewrite
		mov	tempreg1,r19
		rcall	mem_eewrite
		mov	tempreg1,tempreg3
		rcall	mem_eewrite		
		ret

;------------------------------------------------------------------------------
; get sysconf
;------------------------------------------------------------------------------
mem_getconf1:	ldi	YL,LOW(2040)		;default -> config 
		ldi	YH,HIGH(2040)
		rcall	mem_eeread		;get value
		andi	tempreg1,0xfb		;set to 644 mode
		out	GPIOR0,tempreg1
		rcall	mem_eeread
		out	GPIOR1,tempreg1
		rcall	mem_eeread
		sts	PCMSK0,tempreg1		
		cbi	GPIOR0,2		;clear "P"-Flag
		ldi	ZL,0xcc			;UBRR1L
		clr	ZH
		ld	tempreg1,Z
		ldi	tempreg2,0xa5
		st	Z,tempreg2
		ld	tempreg2,Z
		st	Z,tempreg1
		cpi	tempreg2,0xa5
		brne	type_m644
		sbis	GPIOR0,5		;skip if PD3 is serial IN
		sbi	GPIOR0,2		;set "P"-Flag
type_m644:	ret

;------------------------------------------------------------------------------
; set memory pointer to prog tempreg1
;------------------------------------------------------------------------------
mem_setprog:	push	ZH
		push	ZL
		rcall	mem_progadr_0		;get ptr in Z
		sts	libmio_ram+44,ZL	;program base L
		sts	libmio_ram+45,ZH	;program base H
		pop	ZL
		pop	ZH
		ret

;------------------------------------------------------------------------------
; read line (ctrl) from program memory to line buffer
;------------------------------------------------------------------------------
mem_readline:	lds	ZL,libmio_ram+44	;base address
		lds	ZH,libmio_ram+45
		ldi	XL,32			;bytes/line
		mul	XL,ctrl
		add	ZL,r0
		adc	ZH,r1
		adiw	ZL,32			;skip name
mem_readline0:	ldi	YL,LOW(bas_linebuf)	;buffer address
		ldi	YH,HIGH(bas_linebuf)
		ldi	XL,32			;no of bytes to copy
mem_readline1:	lpm	XH,Z+			;get from flash
		st	Y+,XH			;put to linebuffer
		dec	XL
		brne	mem_readline1
		ldi	XH,0xff			;append an extra EOL
		st	Y+,XH
		ret

mem_spm:	in	tempreg1,SPMCSR
		sbrc	tempreg1,0
		rjmp	mem_spm
mem_spm1:	out	SPMCSR,XH
		spm
mem_spm2:	in	tempreg1,SPMCSR
		sbrc	tempreg1,0
		rjmp	mem_spm2
		ret

mem_waitrww:	in	tempreg1,SPMCSR
		sbrs	tempreg1,6
		ret
		ldi	XH,0x11
		rcall	mem_spm
		rjmp	mem_waitrww

;------------------------------------------------------------------------------
; check for timer overflow
;------------------------------------------------------------------------------
mem_checktim:	in	tempreg1,TIFR1
		sbrs	tempreg1,2
		ret
		ldi	tempreg1,0x04
		out	TIFR1,tempreg1
		add	vline_l,const_1
		adc	vline_h,const_0
		ret

;------------------------------------------------------------------------------
; write line from Y to program memory
;------------------------------------------------------------------------------
mem_wrline:	mov	tempreg1,ZL		;pointer
		andi	tempreg1,0xe0		;mask out low bits
		brne	mem_wrline_1		;no first line
;erase page
		ldi	XH,0x03			;page erase
		rcall	mem_spm
		rcall	mem_waitrww

mem_wrline_1:	ldi	XL,16			;16 words to copy
mem_wrline_2:	ld	r0,Y+			;low byte
		ld	r1,Y+			;high byte
		ldi	XH,0x01			;write to buffer
		rcall	mem_spm
		adiw	ZL,2			;ptr+2
		dec	XL			;loop counter
		brne	mem_wrline_2		;loop
;check for end of page
		cpi	ZL,0x00			;end of page
		breq	mem_wrline_3		;yes
		ret

;write page
mem_wrline_3:	sbiw	ZL,2
		ldi	XH,0x05			;page write
		rcall	mem_spm
		rcall	mem_waitrww
		adiw	ZL,2
		ret

;------------------------------------------------------------------------------
; write array to program memory
;------------------------------------------------------------------------------
mem_writepage:	cli
mem_wpage0:	push	tempreg1
		push	XH
		push	XL
		rcall	sys_extdisable
		ldi	XH,0x03			;page erase
		rcall	mem_spm
		rcall	mem_waitrww

		sbiw	ZL,2
		ldi	YL,LOW(bas_array+512)	;buffer address
		ldi	YH,HIGH(bas_array+512)
		ldi	XL,0x80			;no. of words to copy
mem_wpage1:	adiw	ZL,2
		ld	r0,Y+			;low byte
		ld	r1,Y+			;high byte
		ldi	XH,0x01			;write to buffer
		rcall	mem_spm
		dec	XL
		brne	mem_wpage1

		ldi	XH,0x05			;page write
		rcall	mem_spm
mem_wpage2:	rcall	mem_waitrww
		pop	XL
		pop	XH
		pop	tempreg1
		sei
		ret

;------------------------------------------------------------------------------
; read name from program memory to Y+
;------------------------------------------------------------------------------
mem_readname:	lds	ZL,libmio_ram+44	;base address of program
		lds	ZH,libmio_ram+45
		sbiw	ZL,32			;point to fname
mem_readname0:	ldi	XL,12			;bytes/name
mem_readname1:	lpm	XH,Z+
		st	Y+,XH			;get bytes
		dec	XL
		brne	mem_readname1
		in	XL,GPIOR2
		sbrc	XL,7
		ret
		ldi	XL,LOW(libmio_cols*libmio_rows-12)	;offset
		ldi	XH,HIGH(libmio_cols*libmio_rows-12)
		add	YL,XL
		adc	YH,XH
		ldi	XL,12
		lds	XH,libmio_color
mem_readname2:	st	Y+,XH
		dec	XL
		brne	mem_readname2
		ret

;------------------------------------------------------------------------------
; read name from program ctrl to Y+
;------------------------------------------------------------------------------
mem_readnamec:	ldi	ZL,LOW(bas_programs*2)	;base address
		ldi	ZH,HIGH(bas_programs*2)
		ldi	XL,LOW(3072)		;bytes/program
		ldi	XH,HIGH(3072)
		mov	tempreg1,ctrl
mem_readnamec1:	cpi	tempreg1,0
		breq	mem_readname0
		add	ZL,XL
		adc	ZH,XH
		dec	tempreg1
		rjmp	mem_readnamec1

;------------------------------------------------------------------------------
; read first byte of program to tempreg1
;------------------------------------------------------------------------------
mem_readfirst:	lds	tempreg1,libmio_lastmenu
mem_readf1:	andi	tempreg1,0x07
		push	XL
		ldi	ZL,LOW(bas_programs*2+12)	;base address
		ldi	ZH,HIGH(bas_programs*2+12)
		ldi	XL,12
		mul	XL,tempreg1
		add	ZH,r0
		lpm	tempreg1,Z+		;type byte
		lpm	tempreg2,Z+		;flag byte
		subi	ZL,14			;set Z to start of prg
		pop	XL
		ret

;------------------------------------------------------------------------------
; save program from editor to flash
;------------------------------------------------------------------------------
mem_save:	cli
		rcall	mem_progadr
		ldi	YL,LOW(bas_linebuf)	;base address
		ldi	YH,HIGH(bas_linebuf)
		ldi	XL,LOW(libmio_vram+11)
		ldi	XH,HIGH(libmio_vram+11)
		ldi	tempreg2,12		;bytes/name
mem_save_01:	ld	tempreg1,X+		;get from filename
		st	Y+,tempreg1		;write to linebuf
		dec	tempreg2
		brne	mem_save_01
		ldi	tempreg1,'B'		;BASIC
		st	Y+,tempreg1
		ldi	tempreg1,0x0c		;Pages
		st	Y+,tempreg1
		ldi	tempreg2,18		;add free space
		ldi	tempreg1,0x20		;space char
mem_save_02:	st	Y+,XH			;write to linebuf
		dec	tempreg2
		brne	mem_save_02
		sbiw	YL,32
		rcall	mem_wrline		;write first line
		clr	ctrl			;line number
;save lines loop
mem_save_03:	ldi	XH,libmio_ecols		;bytes/vram-line
		mul	XH,ctrl
		ldi	XL,LOW(libmio_vram+3*libmio_ecols+3)	;lines start
		ldi	XH,HIGH(libmio_vram+3*libmio_ecols+3)
		add	XL,r0			;calc line address in vram
		adc	XH,r1
;copy line to bas_inbuf
		ldi	YL,LOW(bas_inbuf)	;text buffer address
		ldi	YH,HIGH(bas_inbuf)
		ldi	tempreg2,libmio_ecols-3	;number of chars to copy
mem_save_04:	ld	tempreg1,X+		;get from vram
		st	Y+,tempreg1
		dec	tempreg2
		brne	mem_save_04
		st	Y+,const_0		;add an extra EOL
;shrink line
		ldi	tempreg2,libmio_ecols-3	;max chars
mem_save_05:	ld	tempreg1,-Y		;
		cpi	tempreg1,0x20		;space
		brne	mem_save_06
		st	Y,const_0		;write EOL
		dec	tempreg2
		brne	mem_save_05
mem_save_06:	lds	YL,libmio_vram+11
		cpi	YL,0x5f			;'_'
		brne	mem_save_07
		ldi	YL,LOW(bas_inbuf)	;text buffer address
		ldi	YH,HIGH(bas_inbuf)
		rjmp	mem_save_08
mem_save_07:	call	clin_ext		;compress line
		ldi	YL,LOW(bas_linebuf)	;buffer address
		ldi	YH,HIGH(bas_linebuf)
mem_save_08:	rcall	mem_wrline		;write compressed line
;next line
		inc	ctrl			;line +1
		cpi	ctrl,bas_lines		;last
		brne	mem_save_03		;no
		sei				;enable interrupts
		ret				;OK, end here

;------------------------------------------------------------------------------
; load program from flash to editor
;------------------------------------------------------------------------------
mem_load:	rcall	mem_progadr		;set Z to address
;read name
		ldi	XL,LOW(libmio_vram+11)
		ldi	XH,HIGH(libmio_vram+11)
		ldi	tempreg2,12		;bytes/name
mem_load_01:	lpm	tempreg1,Z+		;get from filename
		st	X+,tempreg1		;write to linebuf
		dec	tempreg2
		brne	mem_load_01
		adiw	ZL,20			;set ptr to first line
		clr	ctrl			;line number

;load lines loop
mem_load_02:	ldi	tempreg2,32		;bytes per line
		ldi	XL,LOW(bas_linebuf)
		ldi	XH,HIGH(bas_linebuf)
mem_load_03:	lpm	tempreg1,Z+		;get from flash
		st	X+,tempreg1		;write to linebuf
		dec	tempreg2
		brne	mem_load_03
;clear text buffer
		ldi	XL,LOW(bas_inbuf)	;text buffer address
		ldi	XH,HIGH(bas_inbuf)
		ldi	tempreg1,0x20		;space
		ldi	tempreg2,libmio_ecols
lce_load_04:	st	X+,tempreg1
		dec	tempreg2
		brne	lce_load_04		;loop
;expand line
		push	ZH			;save registers
		push	ZL
		push	ctrl
		call	xlin_ext		;expand line
		pop	ctrl			;restore registers
		pop	ZL
		pop	ZH
;copy line to vram
		ldi	YL,LOW(bas_inbuf)	;text buffer address
		ldi	YH,HIGH(bas_inbuf)
		ldi	XH,libmio_ecols		;bytes/vram-line
		mul	XH,ctrl
		ldi	XL,LOW(libmio_vram+3*libmio_ecols)	;lines start
		ldi	XH,HIGH(libmio_vram+3*libmio_ecols)
		add	XL,r0			;calc line address
		adc	XH,r1
		ldi	tempreg2,libmio_ecols
lce_load_05:	ld	tempreg1,Y+		;get from inbuf
		st	X+,tempreg1		;store to vram
		dec	tempreg2
		brne	lce_load_05

;next line
		inc	ctrl			;line +1
		cpi	ctrl,bas_lines
		brne	mem_load_02
		ret

;------------------------------------------------------------------------------
; calculate memory program address
;------------------------------------------------------------------------------
mem_progadr:	lds	tempreg1,libmio_lastmenu	;program to save
mem_progadr_0:	andi	tempreg1,0x07		;mask bits
		swap	tempreg1		;*16
		ldi	ZL,0xc0			;*192
		mul	ZL,tempreg1
		ldi	ZL,LOW(bas_programs*2)	;base
		ldi	ZH,HIGH(bas_programs*2)
		add	ZL,r0			;set ptr to start
		adc	ZH,r1
		ret

;------------------------------------------------------------------------------
; clear program
;------------------------------------------------------------------------------
mem_clear:	ldi	YL,LOW(bas_array+512)	;buffer address
		ldi	YH,HIGH(bas_array+512)
		ldi	tempreg1,12		;name head len
		ldi	tempreg2,0x20		;space
mem_clear_1:	st	Y+,tempreg2
		dec	tempreg1
		brne	mem_clear_1
		ldi	tempreg2,'B'
		st	Y+,tempreg2
		ldi	tempreg2,0xff		;mark as empty
		ldi	tempreg1,256-13		;rest of head
mem_clear_1a:	st	Y+,tempreg2
		dec	tempreg1
		brne	mem_clear_1a
		rcall	mem_progadr		;set program adr to Z
		rcall	mem_writepage		;write first page
		adiw	ZL,2
		ldi	YL,LOW(bas_array+512)	;buffer address
		ldi	YH,HIGH(bas_array+512)
		ldi	tempreg1,0x20		;name len
		ldi	tempreg2,0xff		;fill byte
mem_clear_3:	st	Y+,tempreg2		;clear name in buffer
		dec	tempreg1
		brne	mem_clear_3
		ldi	tempreg1,0x0b		;pages to do
mem_clear_4:	push	tempreg1
		rcall	mem_writepage
		adiw	ZL,2
		pop	tempreg1
		dec	tempreg1
		brne	mem_clear_4
		sei
		jmp	sys_extsearch

;------------------------------------------------------------------------------
; copy program to tempreg4
;------------------------------------------------------------------------------
mem_copy:	mov	tempreg1,tempreg4	;copy dest
		andi	tempreg1,0x07		;only 8 programs
		rcall	mem_progadr_0		;set dest pointer
		push	ZH
		push	ZL
		rcall	mem_progadr		;set src pointer
		pop	XL			;dest pointer
		pop	XH
		ldi	tempreg1,0x0c		;number of pages
mem_copy_1:	push	tempreg1		;save counter
		ldi	YL,LOW(bas_array+512)	;buffer address
		ldi	YH,HIGH(bas_array+512)
		ldi	tempreg1,0x00		;no of bytes to copy
mem_copy_2:	lpm	tempreg2,Z+		;get byte from flash
		st	Y+,tempreg2		;store to buffer
		dec	tempreg1		;byte counter
		brne	mem_copy_2		;loop
		push	ZH			;save src pointer
		push	ZL
		movw	ZL,XL			;dst pointer
		rcall	mem_writepage		;write page to flash
		adiw	ZL,2
		movw	XL,ZL			;dst pointer
		pop	ZL
		pop	ZH
		pop	tempreg1
		dec	tempreg1
		brne	mem_copy_1
		sei
		jmp	sys_extsearch

;------------------------------------------------------------------------------
; read one byte (tempreg1) from EEPROM memory (Y+)
;------------------------------------------------------------------------------
mem_eeread:	jmp	libeep_read

;------------------------------------------------------------------------------
; write one byte (tempreg1) to EEPROM memory (Y+)
;------------------------------------------------------------------------------
mem_eewrite:	jmp	libeep_write
