;################################################################################
;#										#
;# libmio - multi i/o for ATMega 16/32 	version 0.20				#
;# copyright (c) 2005/2006 Joerg Wolfram (joerg@jcwolfram.de)			#
;#										#
;# * PAL/NTSC RGB videodisplay with up to 32x24 chars and 8 colors		#
;# * 1 channel sound can play notes and noise					#
;# * ps2-keyboard interface							#
;# * 1200 Bps bidirectional serial port						#
;# * pseudorandom generator							#
;# * 0,1s timer									#
;#										#
;# This library is free software; you can redistribute it and/or		#
;# modify it under the terms of the GNU Lesser General Public			#
;# License as published by the Free Software Foundation; either			#
;# version 2 of the License, or (at your option) any later version.		#
;#										#
;# This library is distributed in the hope that it will be useful,		#
;# but WITHOUT ANY WARRANTY; without even the implied warranty of		#
;# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU		#
;# Lesser General Public License for more details.				#
;#										#
;# You should have received a copy of the GNU Lesser General Public		#
;# License along with this library; if not, write to the			#
;# Free Software Foundation, Inc., 59 Temple Place - Suite 330,			#
;# Boston, MA 02111-1307, USA.							#
;#										#
;################################################################################
;
;Interface:
;
;	libmio_isr	: video interrupt (compare match OCR1B)
;	libmio_init	: initialize timer&port 
; 	libmio_outd	: output dezimal value (XL,XH) with format (tempreg2)
; 	libmio_outh	: output hexadezimal value (XL) with format (tempreg2)
; 	libmio_clr	: clear screen
; 	libmio_fill	: fill screen with char tempreg1
; 	libmio_orom	: output null-teminated string from flash (Z)
; 	libmio_oram	: output null-teminated string from RAM (Z)
; 	libmio_mtext	: output multiple chars (tempreg1, ZL=n)
; 	libmio_char	: output character (tempreg1)
; 	libmio_newline	: output newline (goto start of next line)
; 	libmio_space	: output space character
; 	libmio_comma	: output comma character
;	libmio_plot	: plot/unplot pixel
;	libmio_box	: draw rectangle
;	libmio_fbox	: draw filled rectangle
;	libmio_note	: play a note or noise
;	libmio_waitkey	: wait until a key is pressed
;	libmio_cursor	: wait until a key is pressed and blink Cursor
;	libmio_pser	: put tempreg1 to serial output
;	libmio_pser	: get tempreg1 from serial input
;	libmio_ppar	: put tempreg1 to parallel output
;	libmio_screenshot send screenshot to serial output
;	libmio_chp	: enable/disable charge pump (XL)
;
;###############################################################################
.ifdef use_libmio_all
    .set	use_libmio_outd = 1
    .set	use_libmio_outh = 1
    .set	use_libmio_orom = 1
    .set	use_libmio_oram = 1
    .set	use_libmio_othis = 1
    .set	use_libmio_mtext = 1
    .set	use_libmio_newline = 1
    .set	use_libmio_gotoxy = 1
    .set	use_libmio_plot = 1
    .set	use_libmio_box = 1
    .set	use_libmio_fbox = 1
    .set	use_libmio_bigchar = 1
    .set	use_libmio_sync = 1
    .set	use_libmio_note = 1
    .set	use_libmio_waitkey = 1
    .set	use_libmio_cursor = 1
    .set	use_libmio_pser = 1
    .set	use_libmio_gser = 1
    .set	use_libmio_ppar = 1
    .set	use_libmio_screenshot = 1
    .set	use_libmio_chp = 1
.endif

.macro libmio_oc16
		lpm	XH,Z+			;3 get P0 and P3
		out	PORTC,XH		;1 out P0
		
		lpm	XL,Z+			;3 get P1 and P4
		out	PORTC,XL		;1 out P1

		lpm	tempreg1,Z+		;3 get P2 and P5
		out	PORTC,tempreg1		;1 out P2

		swap	XH			;1 prepare P3
		ld	ZL,Y+			;2 get next char
		out	PORTC,XH		;1 out P3
	
		swap	XL			;1 prepare P4
		mov	ZH,tempreg2		;1 startadr
		lsl	ZL			;1 *2
		out	PORTC,XL		;1 out P4

		swap	tempreg1		;1 prepare P5
		lsl	ZL			;1 *2
		rol	ZH			;1 complete address
		out	PORTC,tempreg1		;1 out P5
.endm

;-------------------------------------------------------------------------------		
;video-interrupt (no registers destroyed, stack=13)
;save registers
;-------------------------------------------------------------------------------		
libmio_isr:	push	tempreg1		;2
		push	tempreg2		;2
		push	ZH			;2 save Z-register
		push	ZL			;2 save Z-register
		push	YH			;2 save Y-register
		push	YL			;2 save Y-register
		push	XH			;2 save Y-register
		push	XL			;2 save Y-register
		push	r0			;2 save r0
		push	r1			;2 save r1		
		in	ZH,SREG			;1 save status
		push	ZH			;2
		
;-------------------------------------------------------------------------------		
;count line up
;-------------------------------------------------------------------------------
libmio_is_06:	lds	XL,libmio_line_l	;2 get line_l
		lds	XH,libmio_line_h	;2 get line_h
		add	XL,byte1		;1 next line
		adc	XH,byte0		;1 next line
		sts	libmio_line_l,XL	;2 writeback line_l
		sts	libmio_line_h,XH	;2 writeback line_l
		
;-------------------------------------------------------------------------------		
;check, if line is image, empty or vsync
;-------------------------------------------------------------------------------
		lds	ZL,libmio_config	;configuration
		cpi	XL,libmio_pixel_rows	;1 max lines
		sbrs	XH,0			;1 skip if no image
		brcs	libmio_is_09		;2 goto image-line (short jump)
		brne	libmio_is_08		;2 no maxline
		lds	YL,libmio_vflags	;odd/even
		sbrc	YL,7
		add	XL,byte1		;1 add 1 line
		subi	YL,0x80			;1 swap odd/even flag
		sts	libmio_line_l,XL	;2 writeback line_l
		ldi	XL,5			;1 fps/10 for pal
		sbrs	ZL,2			;1 pal/ntsc bit
		inc	XL			;1 fps/10 for ntsc
		ldi	tempreg1,50		;1 fps for pal
		sbrs	ZL,2			;1 pal/ntsc bit
		ldi	tempreg1,60		;1 fps for ntsc
		
		lds	XH,libmio_time0		;2 timer
		inc	XH
		cp	XH,XL			;1 frames (5/6)
		brne	libmio_is_06a
		clr	XH
		lds	ZL,libmio_time_l
		lds	ZH,libmio_time_h
		adiw	ZL,1
		sts	libmio_time_l,ZL
		sts	libmio_time_h,ZH		
libmio_is_06a:	sts	libmio_time0,XH		
		lds	YH,libmio_frames	;2 get frame counter
		inc	YH			;1 +1
		cp	YH,tempreg1		;1 frames (50/60)
		brne	libmio_is_07		;2 frame oveflow
		clr	YH			;1 set frames=0
		ori	YL,0x40			;1 set seconds flag
libmio_is_07:	sts	libmio_frames,YH	;2 set frames=0
		sts	libmio_vflags,YL		
		rjmp	libmio_is_15		;2
		
libmio_is_09:	sbrc	ZL,2
		rjmp	libmio_issp		;2 goto image-line (pal)
		rjmp	libmio_issn		;2 goto image-line (ntsc)		

libmio_is_08:	ldi	ZH,libmio_vstart_p	;1 vstart pal
		sbrs	ZL,2			;1 pal/ntsc bit
		ldi	ZH,libmio_vstart_n	;1 vstart ntsc
		cp	XL,ZH
		brne	libmio_is_8a
		cbi	PORTD,6			;VSYNC
		sbrs	ZL,3			;csync bit
		rjmp	libmio_is_15		;2  
		ldi	XH,0x80
		out	OCR1AH,XH		;1 OC1B is low (VSYNC)
		out	OCR1AL,byte0		;1
		rjmp	libmio_is_15		;2  


libmio_is_8a:	ldi	ZH,libmio_vstop_p	;1 vstop pal
		sbrs	ZL,2			;1 pal/ntsc bit
		ldi	ZH,libmio_vstop_n	;1 vstop ntsc
		cp	XL,ZH
		brne	libmio_is_10
		sbi	PORTD,6
		sbrs	ZL,3			;csync bit
		rjmp	libmio_is_15		;2  
		out	OCR1AH,byte0		;1 OC1B is HSYNC
		ldi	ZH,libmio_hslen_p	;1 vstop pal
		sbrs	ZL,2			;1 pal/ntsc bit
		ldi	ZH,libmio_hslen_n	;1 vstop ntsc
		out	OCR1AL,ZH		;1
		rjmp	libmio_is_15		;2 vsync line is empty 

		
libmio_is_10:	ldi	ZH,57			;1 maxlines pal
		sbrs	ZL,2			;1 pal/ntsc bit
		ldi	ZH,7			;1 maxlines ntsc
		cp	XL,ZH
		breq	libmio_is_11		;2 end of frame
		rjmp	libmio_is_15		
		
libmio_is_11:	ldi	XL,LOW(libmio_vram)	;1 set vram start
		sts	libmio_rampos_l,XL	;2
		ldi	XL,HIGH(libmio_vram)	;1
		sts	libmio_rampos_h,XL	;2
		sts	libmio_cline,byte0	;2 cline=0
		sts	libmio_line_l,byte0	;1 line_l=0
		sts	libmio_line_h,byte0	;1 line_h=0

;-------------------------------------------------------------------------------
; calculate envelope position
;-------------------------------------------------------------------------------		
		ldi	ZH,HIGH(libmio_table*2+4099)	;1 start envtable high
		ldi	ZL,LOW(libmio_table*2+4099)	;1 start envtable low
		lds	XL,libmio_env		;2 envelope pointer
		add	ZL,XL			;1 envelope-position
		adc	ZH,byte0		;1
		add	ZL,XL			;1 envelope-position
		adc	ZH,byte0		;1
		add	ZL,XL			;1 envelope-position
		adc	ZH,byte0		;1
		add	ZL,XL			;1 envelope-position
		adc	ZH,byte0		;1
		lpm	XH,Z			;get envelope value
		cp	XH,byte0		;must be zero to stop
		breq	libmio_is_11a		;the end is reached
		inc	XL			;+1
		sts	libmio_env,XL		;write back
libmio_is_11a:	rjmp	libmio_is_15

;-------------------------------------------------------------------------------
;sync to counter (ntsc)
;-------------------------------------------------------------------------------		
libmio_issn:	in	XL,TCNT1L		;1 get timer
		cpi	XL,libmio_hstart_n+68 	;1
		brcs	libmio_issn		;1 loop		
libmio_issn_01:	cpi	XL,libmio_hstart_n+69	
		brcs	libmio_issn_02
libmio_issn_02:	cpi	XL,libmio_hstart_n+70	
		brcs	libmio_issn_03
libmio_issn_03:	cpi	XL,libmio_hstart_n+71	
		brcs	libmio_issn_04
libmio_issn_04:	rjmp	libmio_is_12

;-------------------------------------------------------------------------------
;sync to counter (pal)
;-------------------------------------------------------------------------------		
libmio_issp:	in	XL,TCNT1L		;1 get timer
		cpi	XL,libmio_hstart_p+68 	;1
		brcs	libmio_issp		;1 loop		
libmio_issp_01:	cpi	XL,libmio_hstart_p+69	
		brcs	libmio_issp_02
libmio_issp_02:	cpi	XL,libmio_hstart_p+70	
		brcs	libmio_issp_03
libmio_issp_03:	cpi	XL,libmio_hstart_p+71	
		brcs	libmio_is_12

;-------------------------------------------------------------------------------
; -output character line 4 clocks per pixel
; -128 characters are in table 6x10px each
;-------------------------------------------------------------------------------
libmio_is_12:	lds	YL,libmio_rampos_l	;2 BS low
		lds	YH,libmio_rampos_h	;1 BS high
		lds	tempreg2,libmio_cline	;2 cline
		ldi	ZH,HIGH(libmio_table)	;add table
		add	ZH,tempreg2
		mov	tempreg2,ZH
		ld	ZL,Y+			;get first char of line
		lsl	ZL			;1 char*2
		mov	ZH,tempreg2		;1 startadr
		lsl	ZL			;1 *2
		rol	ZH			;1 complete address
		libmio_oc16			;char 1
		libmio_oc16			;char 2
		libmio_oc16			;char 3
		libmio_oc16			;char 4
		libmio_oc16			;char 5
		libmio_oc16			;char 6
		libmio_oc16			;char 7
		libmio_oc16			;char 8
		libmio_oc16			;char 9
		libmio_oc16			;char 10
		libmio_oc16			;char 11
		libmio_oc16			;char 12
		libmio_oc16			;char 13
		libmio_oc16			;char 14
		libmio_oc16			;char 15
		libmio_oc16			;char 16
		libmio_oc16			;char 17
		libmio_oc16			;char 18
		libmio_oc16			;char 19
		libmio_oc16			;char 20
		libmio_oc16			;char 21
		libmio_oc16			;char 22
		libmio_oc16			;char 23
		libmio_oc16			;char 24
		libmio_oc16			;char 25
		libmio_oc16			;char 26
		libmio_oc16			;char 27
		libmio_oc16			;char 28
		libmio_oc16			;char 29
		libmio_oc16			;char 30
.if libmio_cols > 30
		libmio_oc16			;char 31
.endif
.if libmio_cols > 31
		libmio_oc16			;char 32
.endif
		nop				;1 filling cycle		
		ld	ZL,-Y			;2 correct Y-register
		out	PORTC,byte0		;1 set to black
		
		lds	ZH,libmio_cline		;2 character line
		inc	ZH			;1 next line
		cpi	ZH,0x0a			;1 check
		brne	libmio_is_14		;2 jump if not next line
		clr	ZH			;1 character line = 0
		sts	libmio_rampos_l,YL	;2 new memory position
		sts	libmio_rampos_h,YH	;2 new memory position
libmio_is_14:	sts	libmio_cline,ZH		;2 new character line

libmio_is_15:


;-------------------------------------------------------------------------------
; calculate sound
;-------------------------------------------------------------------------------
		lds	tempreg1,libmio_rand2	;2 noise signal
		lds	XL,libmio_note_val	;2
		cpi	XL,0xff
		breq	libmio_wa_01
		lds	XL,libmio_wpos_l	;2 wave pointer low
		lds	XH,libmio_wpos_h	;2 wave pointer high
		lds	r0,libmio_woff_l	;2 wave offset low
		lds	r1,libmio_woff_h	;2 wave offset high
		add	XL,r0			;1 calculate new pointer
		adc	XH,r1			;1 
		sts	libmio_wpos_l,XL	;2 writeback new pointer
		sts	libmio_wpos_h,XH	;2 
		ldi	ZH,HIGH(libmio_table*2+3075)	;1 start wavetable 1 high
		ldi	ZL,LOW(libmio_table*2+3075)	;1 start wavetable 1 low
		ldi	XL,4			;1 factor
		mul	XL,XH			;2
		add	ZL,r0			;1 wave-position
		adc	ZH,r1			;1 
		lpm	tempreg1,Z		;3 get wave value
libmio_wa_01:	ldi	ZH,HIGH(libmio_table*2+4099)	;1 start envtable high
		ldi	ZL,LOW(libmio_table*2+4099)	;1 start envtable low
		lds	XH,libmio_env		;2 envelope pointer
		ldi	XL,4			;1 factor
		mul	XL,XH			;2
		add	ZL,r0			;1 envelope-position
		adc	ZH,r1			;1
		lpm	tempreg2,Z		;3 get env value
		mulsu	tempreg1,tempreg2	;2 shaped value
		ldi	XH,0x80			;1 neutral signal
		add	XH,r1			;1 add result
		out	OCR2,XH			;2 set level
		
;-------------------------------------------------------------------------------
; pseudo-random generator
;-------------------------------------------------------------------------------
		lds	ZL,libmio_rand1
		lds	ZH,libmio_rand2
		lds	YL,libmio_rand3
		lds	YH,libmio_rand4
		clr	XL
		sbrc	ZL,2
		com	XL
		sbrc	ZL,5
		com	XL
		sbrc	ZH,4
		com	XL
		sbrc	YH,7
		com	XL
		rol	XL
		rol	ZL
		rol	ZH
		rol	YL
		rol	YH
		sts	libmio_rand1,ZL
		sts	libmio_rand2,ZH
		sts	libmio_rand3,YL
		sts	libmio_rand4,YH

;------------------------------------------------------------------------------
;the keyboard routine
;------------------------------------------------------------------------------
libmio_kir:	in	ZL,UCSRA		;status
		sbrs	ZL,7			;RXC?
		rjmp	libmio_kbd_12		;no byte available
		in	ZL,UDR		

		lds	YL,libmio_kbdstate	;get key state
		cpi	ZL,0xf0			;key release
		brne	libmio_kbd_01		;jump if not
		ori	YL,0x80			;set release flag
		rjmp	libmio_kbd_11		;go to the end

libmio_kbd_01:	cpi	ZL,0xe0			;extended code
		brne	libmio_kbd_01a		;jump if not
		ori	YL,0x40			;set extended flag
		rjmp	libmio_kbd_11		;thats all

libmio_kbd_01a:	cpi	ZL,0x83			;extended code
		brne	libmio_kbd_02		;jump if not
		ldi	ZL,0x10

libmio_kbd_02:	sbrc	ZL,7			;skip if no bit 7
		rjmp	libmio_kbd_12		;thats all

		cpi	ZL,0x12			;left shift
		brne	libmio_kbd_03		;no
		ori	YL,0x01			;set shift bit
		sbrc	YL,7			;skip if not release
		andi	YL,0xfe			;clear shift bit
		rjmp	libmio_kbd_10		;thats all
		
libmio_kbd_03:	cpi	ZL,0x59			;right shift
		brne	libmio_kbd_04		;jump if not
		ori	YL,0x02			;set shift bit
		sbrc	YL,7			;skip if not release
		andi	YL,0xfd			;clear shift bit
		rjmp	libmio_kbd_10		;thats all

libmio_kbd_04:	cpi	ZL,0x14			;control
		brne	libmio_kbd_05		;jump if not
		sbrc	YL,6			;extended? (right ctrl key)
		rjmp	libmio_kbd_04a
		ori	YL,0x04			;set control bit
		sbrc	YL,7			;skip if not release
		andi	YL,0xfb			;clear ctrl bit
		rjmp	libmio_kbd_10		;thats all
libmio_kbd_04a:	ori	YL,0x08			;set control bit
		sbrc	YL,7			;skip if not release
		andi	YL,0xf7			;clear ctrl bit
		rjmp	libmio_kbd_10		;thats all
		
libmio_kbd_05:	cpi	ZL,0x11			;alternate
		brne	libmio_kbd_06		;jump if not
		ori	YL,0x10			;set alt bit
		sbrc	YL,7			;skip if not release
		andi	YL,0xef			;clear alt bit
		rjmp	libmio_kbd_10		;thats all
		
libmio_kbd_06:	sbrs	YL,7			;check release flag
		rjmp	libmio_kbd_06a		
		sts	libmio_keycode,byte0	;clear keycode
		rjmp	libmio_kbd_10		

libmio_kbd_06a:	sts	libmio_scancode,ZL	;var for scancode
		sbrc	YL,6			;skip if not extended
		ori	ZL,0x80			;scancode+128
		ldi	ZH,4			;factor for interleaved table				
		mul	ZL,ZH
		ldi	ZH,HIGH(libmio_table*2+3)	;start keytable
		ldi	ZL,LOW(libmio_table*2+3)	;start keytable
		clr	YH			;-> normal table
		sbrc	YL,0			;left shift
		ldi	YH,4
		sbrc	YL,1			;right shift
		ldi	YH,4
		sbrc	YL,4			;ALT
		ldi	YH,8
		add	ZL,r0
		adc	ZH,r1
		add	ZH,YH
		lpm	ZL,Z			;get value
		sts	libmio_keycode,ZL
		sts	libmio_lastkey,ZL
		lds	YL,libmio_kbdstate	;keyboard status
		cpi	ZL,0xe9			;del?
		brne	libmio_kbd_09
		sbrs	YL,2			;control?
		rjmp	libmio_kbd_09
		sbrs	YL,4			;alternate?
		rjmp	libmio_kbd_09
		jmp	0x0000			;restart
libmio_kbd_09:	sbrs	YL,2			;control?
		rjmp	libmio_kbd_10
		cpi	ZL,'C'			;Break?
		brne	libmio_kbd_09a
		sts	libmio_kflags,byte1		;Break Code
libmio_kbd_09a:	cpi	ZL,'P'			;Screenshot?
		brne	libmio_kbd_10
		ldi	tempreg1,0x02
		sts	libmio_kflags,tempreg1	;print Code
libmio_kbd_10: 	andi	YL,0x3f			;clear ext and release bit
libmio_kbd_11:	sts	libmio_kbdstate,YL	;store keyboard status
libmio_kbd_12:
;-------------------------------------------------------------------------------
;serial port (receive)
;-------------------------------------------------------------------------------		
libmio_is_16:
.ifdef use_libmio_serial
		in	XH,PIND			;get signal byte
		com	XH
		lds	XL,libmio_rcnt		;clock counter
		cpi	XL,0x00			;end
		breq	libmio_is_18
		cpi	XL,0xff			;wait for sig
		brne	libmio_ser_01
		sbrc	XH,3			;skip if line is 0
		rjmp	libmio_is_18
		ldi	XL,124			;start value
		sts	libmio_rcnt,XL
		rjmp	libmio_is_18
libmio_ser_01:	cpi	XL,105
		breq	libmio_ser_02		
		cpi	XL,92
		breq	libmio_ser_02		
		cpi	XL,79
		breq	libmio_ser_02		
		cpi	XL,66
		breq	libmio_ser_02		
		cpi	XL,53
		breq	libmio_ser_02		
		cpi	XL,40
		breq	libmio_ser_02		
		cpi	XL,27
		breq	libmio_ser_02		
		cpi	XL,14
		breq	libmio_ser_02		
		dec	XL
		sts	libmio_rcnt,XL
		rjmp	libmio_is_18
		
libmio_ser_02:	dec	XL
		sts	libmio_rcnt,XL
		lds	XL,libmio_rdat
		lsr	XL
		bst	XH,3			;RXD PIN	
		bld	XL,7
		sts	libmio_rdat,XL
		
;-------------------------------------------------------------------------------
;serial port (send)
;-------------------------------------------------------------------------------		
libmio_is_18:	lds	XL,libmio_tcnt		;clock counter
		cpi	XL,0x00	
		breq	libmio_is_20		;nothing to do
		cpi	XL,0xff			;wait for start
		brne	libmio_set_01
		ldi	XL,144
		cbi	PORTD,4			;start bit
		sts	libmio_tcnt,XL
		rjmp	libmio_is_20
		
libmio_set_01:	cpi	XL,131
		breq	libmio_set_03
		cpi	XL,118
		breq	libmio_set_03
		cpi	XL,105
		breq	libmio_set_03
		cpi	XL,92
		breq	libmio_set_03
		cpi	XL,79
		breq	libmio_set_03
		cpi	XL,66
		breq	libmio_set_03
		cpi	XL,53
		breq	libmio_set_03
		cpi	XL,40
		breq	libmio_set_03
		cpi	XL,27
		brne	libmio_set_02
		sbi	PORTD,4
libmio_set_02:	dec	XL
		sts	libmio_tcnt,XL
		rjmp	libmio_is_20

libmio_set_03:	lds	XH,libmio_tdat
		sbrs	XH,0
		cbi	PORTD,4
		sbrc	XH,0
		sbi	PORTD,4
		lsr	XH
		sts	libmio_tdat,XH
		dec	XL
		sts	libmio_tcnt,XL
.endif

;-------------------------------------------------------------------------------
;restore registers and exit
;-------------------------------------------------------------------------------		
libmio_is_20:	pop	ZH			;2 restore sreg
		out	SREG,ZH			;1 put back    
		pop	r1			;2 restore r1
		pop	r0			;2 restore r0
		pop	XL			;2 restore Y-register
		pop	XH			;2 restore Y-register
libmio_is_30:	pop	YL			;2 restore Y-register
		pop	YH			;2 restore Y-register
		pop	ZL			;2 restore Z-register
		pop	ZH			;2 restore Z-register
		pop	tempreg2		;2
		pop	tempreg1		;2 restore register
		reti				;4 thats all
	
;-------------------------------------------------------------------------------
;port konfiguration for port B ... D
;-------------------------------------------------------------------------------
libmio_init:	lds	ZL,libmio_config	;configuration
		ldi	XL,0x0a			;port direction
		out	DDRB,XL
		ldi	XL,0xff
		out	PORTB,XL	

		ldi	XL,0xe0			;port direction
		out	DDRC,XL
		ldi	XL,0x00
		out	PORTC,XL	

		ldi	XL,0xf0			;set pin 4-7 as output
		out	DDRD,XL
		ldi	XL,0xff
		out	PORTD,XL	

		in	XL,PINC			;get portc (bit 2-4 = config)
		sts	libmio_config,XL	;store configuration
		
;-------------------------------------------------------------------------------
;timer configuration
;-------------------------------------------------------------------------------
		ldi	XL,0xc2			;FAST-PWM-Mode 14 (PWM channel B)
		out	TCCR1A,XL
		ldi	XL,0x19			;FAST-PWM-Mode (no prescaler)
		out	TCCR1B,XL
		ldi	ZH,HIGH(libmio_counter_p-1)
		sbrs	ZL,2			;1 pal/ntsc bit
		ldi	ZH,HIGH(libmio_counter_n-1)
		out	ICR1H,ZH
		ldi	ZH,LOW(libmio_counter_p-1)
		sbrs	ZL,2			;1 pal/ntsc bit
		ldi	ZH,LOW(libmio_counter_n-1)
		out	ICR1L,ZH

		out	OCR1AH,byte0		;we use only 8 bits
		ldi	ZH,libmio_hslen_p	;1 hsync length pal
		sbrs	ZL,2			;1 pal/ntsc bit
		ldi	ZH,libmio_hslen_n	;1 hsync length ntsc
		out	OCR1AL,ZH		

		out	OCR1BH,byte0		;we use only 8 bits
		ldi	ZH,libmio_hstart_p	;1 hor position pal
		sbrs	ZL,2			;1 pal/ntsc bit
		ldi	ZH,libmio_hstart_n	;1 hor position ntsc
		out	OCR1BL,ZH		
				
		ldi	XL,0x08
		out	TIMSK,XL		;enable compare OC1B int

		ldi	XL,0x69			;timer 2 config
		out	TCCR2,XL
		ldi	XL,0x80			;zero signal
		out	OCR2,XL
		
		ldi	XL,0x09			;ctc but disable output
		out	TCCR0,XL
		ldi	XL,49			;160KHz
		out	OCR0,XL			

;-------------------------------------------------------------------------------
;usart configuration (synchron for keyboard)
;-------------------------------------------------------------------------------
		out	UCSRA,byte0
		ldi	XL,0x10
		out	UCSRB,XL
		ldi	XL,0xe6
		out	UCSRC,XL

;-------------------------------------------------------------------------------
;clear video-RAM and set pointers
;-------------------------------------------------------------------------------
		ldi	YL,LOW(libmio_ram)
		ldi	YH,HIGH(libmio_ram)
		ldi	XL,LOW(libmio_vram)	;2
		st	Y+,XL			;2 start vram
		ldi	XL,HIGH(libmio_vram)	;2
		st	Y+,XL			;2 start vram
		ldi	XL,0x1e
libmio_init_c:	st	Y+,byte0
	    	dec	XL
		brne	libmio_init_c	
		ldi	XL,0xaa	    
		sts	libmio_rand1,XL
		sts	libmio_rand3,XL
		sts	libmio_channel,byte0	;set output to screen
		ret
		
;-------------------------------------------------------------------------------
; Bin16 -> ASCII (dez)
; XL,XH 	= number
; ctrl		= format
; 	Bit 0,1	 = format	00 = compact
;				01 = leading spaces
;				10 = leading zeroes 0...65536
;				11 = leading zeroes with +- sign  
;			
;	Bit 2,3	 = number of visible digits (2,3,4,5)
;	Bit 4,5	 = comma position (no  xxxx,x   xxx,xx  xx,xxx) 
;	Bit 6	1= big size (only in monochrome mode) 
;-------------------------------------------------------------------------------
.ifdef use_libmio_outd
libmio_outd:	push	tempreg1		;save registers
		push	tempreg2
		push	tempreg3
		push	XL
		push	XH
		ldi	tempreg1,0x20		;space
		mov	tempreg2,ctrl
		andi	tempreg2,0x03
		cpi	tempreg2,0x02		;leading zeroes without sign
		breq	libmio_outd_03
		cpi	tempreg2,0x03		;leading zeroes with +- sign
		brne	libmio_outd_01
		ldi	tempreg1,'+'		;set positive sign
libmio_outd_01:	sbrs	XH,7
		rjmp	libmio_outd_02
		com	XL			;X=-X
		com	XH			;
		add	XL,byte1		;
		adc	XH,byte0		;
		ldi	tempreg1,'-'		;sign is negative
libmio_outd_02:	call	libmio_char		;out sign


libmio_outd_03:	mov	tempreg2,ctrl		;format byte
		andi	tempreg2,0x0c		;mask number of digits
		mov	tempreg3,ctrl		;format byte
		andi	tempreg3,0x30		;mask comma position
		clt				;clear nonzero-flag
		ldi	tempreg1,47		;"0"-1 	
libmio_outd_04:	inc	tempreg1			;
		subi	XL,16			;-10000
		sbci	XH,39
		brcc	libmio_outd_04		;no overflow
		subi	XL,240			;+10000
		sbci	XH,216			;
		cpi	tempreg2,0x0c		;5 digits?
		brne	libmio_outd_05		;no		
		rcall	libmio_nc		;out digit 10^4

libmio_outd_05:	ldi	tempreg1,47		;"0"-1 	
libmio_outd_06:	inc	tempreg1			;
		subi	XL,232			;-1000
		sbci	XH,3			;
		brcc	libmio_outd_06		;no overflow
		subi	XL,24			;+1000
		sbci	XH,252			;
		sbrc	tempreg2,3		;skip if < 4 digits
		rcall	libmio_nc		;out digit 10^3
		cpi	tempreg3,0x30		;comma?
		brne	libmio_outd_07		;no
		rcall	libmio_comma		;

libmio_outd_07:	ldi	tempreg1,47		;"0"-1 	
libmio_outd_08:	inc	tempreg1		;+1
		subi	XL,100			;-100
		sbc	XH,byte0		;
		brcc	libmio_outd_08		;no overflow
		subi	XL,156			;+100
		sbci	XH,255			;
		cpi	tempreg2,0x00		;3 digits
		breq	libmio_outd_09
		rcall	libmio_nc		;out digit 10^2
libmio_outd_09:	cpi	tempreg3,0x20		;comma?
		brne	libmio_outd_10		;no
		rcall	libmio_comma		;

libmio_outd_10:	ldi	tempreg1,47		;"0"-1 	
libmio_outd_11:	inc	tempreg1		;+1
		subi	XL,10			;-10
		brcc	libmio_outd_11		;no overflow
		rcall	libmio_nc		;out digit 10^1
		subi	XL,246			;+10
		cpi	tempreg3,0x10		;comma?
		brne	libmio_outd_12		;no
		rcall	libmio_comma		;
		
libmio_outd_12:	ldi	tempreg1,48		;"0"
		add	tempreg1,XL		;LSB digit
		rcall	libmio_char
		pop	XH
		pop	XL
		pop	tempreg3
		pop	tempreg2
		pop	tempreg1
		ret

;-------------------------------------------------------------------------------
; out numeric char (tempreg1)
;-------------------------------------------------------------------------------
libmio_nc:	cpi	tempreg1,48		;is zero
		breq	libmio_nc_1		;yes
		set				;set flag
		rjmp	libmio_char		;send char
		
libmio_nc_1:	sbrc	ctrl,1			;skip if no leading zeroes
libmio_nc_2:	rjmp	libmio_char		;send zero
		brts	libmio_nc_2		;flag is set
		
		sbrs	ctrl,0			;skip if no leading spaces
		ret 
		rjmp	libmio_space		;send space
		ret
.endif

;-------------------------------------------------------------------------------
; Bin -> ASCII (hex) 
; (XL,XH) = number
; ctrl 		= format
; 	Bit 0	1= 16Bit (4 chars)
;	Bit 6	1= Big size (only in monochrome mode)
;-------------------------------------------------------------------------------
.ifdef use_libmio_outh
libmio_outh:	push	tempreg1		;save temp register
		sbrs	ctrl,0			;format 8/16
		rjmp	libmio_outh1		;only 8
		mov	tempreg1,XH		;
		swap	tempreg1		;high-nibble
		rcall	libmio_oh
		mov	tempreg1,XH		;
		rcall	libmio_oh
libmio_outh1:	mov	tempreg1,XL		;
		swap	tempreg1		;high-nibble
		rcall	libmio_oh
		mov	tempreg1,XL		;
		rcall	libmio_oh
		pop	tempreg1		;restore temp register
		ret

;-------------------------------------------------------------------------------
; out hexadecimal char (tempreg1)
;-------------------------------------------------------------------------------
libmio_oh:	andi	tempreg1,0x0f		;mask nibble
		subi	tempreg1,0xd0		;+30
		cpi	tempreg1,0x3a		;>9
		brcs	libmio_oh_1		;jump, if not
		subi	tempreg1,0xf9		;+6
libmio_oh_1:	rjmp	libmio_char		
.endif
		
;-------------------------------------------------------------------------------
; clear video-RAM
;-------------------------------------------------------------------------------
libmio_clr:	ldi	tempreg1,32		;space
libmio_fill:	push	XL
		push	XH
		push	YL
		push	YH
		ldi	XL,3			;plot-color = white
		sts	libmio_color,XL
		ldi	YH,HIGH(libmio_vram)	;home position
		ldi	YL,LOW(libmio_vram)	;home position
		ldi	XL,libmio_rows
libmio_cls_01:	ldi	XH,libmio_cols
libmio_cls_02:	st	Y+,tempreg1		;write to RAM
		dec	XH			;decrement counter
		brne	libmio_cls_02		;loop
		dec	XL			;decrement counter
		brne	libmio_cls_01		;loop
		pop	YH
		pop	YL
libmio_cls_03:	sts	libmio_cur_x,XL		;save X-position
		sts	libmio_cur_y,XH		;save Y-position
		sts	libmio_lpos,byte0
		pop	XH
		pop	XL
		ret				;the end

;-------------------------------------------------------------------------------
; newline and calculate new position
;-------------------------------------------------------------------------------
.ifdef use_libmio_newline
libmio_newline:	push	XL
		push	XH
		sts	libmio_lpos,byte0
		lds	XL,libmio_channel
		cpi	XL,1			;serial
		brne	libmio_newl1
		pop	XH
		pop	XL
		ldi	tempreg1,0x0d		;carriage return
		rcall	libmio_pser		;write serial
		ldi	tempreg1,0x0a		;newline
		rjmp	libmio_pser		;write serial
libmio_newl1:	cpi	XL,2			;parallel
		brne	libmio_newl2
		pop	XH
		pop	XL
		ldi	tempreg1,0x0d		;carriage return
		rcall	libmio_ppar		;write parallel
		ldi	tempreg1,0x0a		;newline
		rjmp	libmio_ppar		;write parallel
		
libmio_newl2:	lds	XL,libmio_cur_x		;X
		lds	XH,libmio_cur_y		;Y
		clr	XL			;begin of line
		inc	XH			;of next line
		cpi	XH,libmio_rows		;beyond screen
		brne	libmio_cls_03
		ldi	XH,(libmio_rows-1)	;last line
		sts	libmio_cur_x,XL		;save X-position
		sts	libmio_cur_y,XH		;save Y-position
		push	YL
		push	YH
		rjmp	libmio_char_04		;scroll screen
					
.endif

;-------------------------------------------------------------------------------
; output character and calculate new position (tempreg1=char)
;-------------------------------------------------------------------------------
libmio_comma:	ldi	tempreg1,0x2c		;set char to comma
		rjmp	libmio_char		;write
libmio_space:	ldi	tempreg1,32		;set char to space
libmio_char:	push	XL
		push	XH
		lds	XL,libmio_lpos
		inc	XL
		sts	libmio_lpos,XL
		lds	XL,libmio_channel
		cpi	XL,1			;serial
		brne	libmio_char1
		pop	XH
		pop	XL
		rjmp	libmio_pser		;write serial
libmio_char1:	cpi	XL,2			;parallel
		brne	libmio_char2
		pop	XH
		pop	XL
		rjmp	libmio_ppar		;write serial
libmio_char2:	push	YL
		push	YH
		rcall	libmio_calc		;calculate address		
		sbrc	ctrl,6			;big chars
		rjmp	libmio_bchar

		st	Y,tempreg1		;write char
		inc	XL			;next position
		cpi	XL,libmio_cols		;beyond the end of line
		brne	libmio_char_01
		clr	XL			;begin of line
		inc	XH			;of next line
		cpi	XH,libmio_rows		;beyond screen
		brne	libmio_char_01
		ldi	XH,libmio_rows-1	;last line
		rjmp	libmio_char_03		;we must scroll			
libmio_char_01:	sts	libmio_cur_x,XL		;X
		sts	libmio_cur_y,XH		;Y
libmio_char_02:	pop	YH
		pop	YL
		pop	XH
		pop	XL
		ret

; scroll screen
libmio_char_03:	sts	libmio_cur_x,XL		;save X-position
		sts	libmio_cur_y,XH		;save Y-position
libmio_char_04:	ldi	YH,HIGH(libmio_vram)	;home position
		ldi	YL,LOW(libmio_vram)	;home position
		push	tempreg1
		ldi	XL,libmio_rows-1	;lines -1
libmio_char_05:	ldi	XH,libmio_cols		
libmio_char_06:	ldd	tempreg1,Y+libmio_cols
		st	Y+,tempreg1
		dec	XH
		brne	libmio_char_06
		dec	XL
		brne	libmio_char_05		
		ldi	XL,libmio_cols
		ldi	tempreg1,32		;space
libmio_char_07:	st	Y+,tempreg1
		dec	XL
		brne	libmio_char_07
		pop	tempreg1
		rjmp	libmio_char_02

libmio_bchar:	cpi	XL,libmio_cols-3
		brcc	libmio_char_02
		cpi	XH,libmio_rows-5
		brcc	libmio_char_02
		subi	XL,0xfd			;X+3
		sts	libmio_cur_x,XL		;store new X
		push	ZL
		push	ZH
		push	tempreg5		;byte 1
		push	tempreg6		;byte 2
		push 	tempreg7		;byte 3
		push	tempreg8		;used for storing t-flag
		bld	tempreg8,0		;save t-flag
		push	tempreg2		;color sel
		clr	tempreg2		;base=0
		lds	ZL,libmio_color		;get color	
		andi	ZL,0x03			;only bit 0,1
		dec	ZL			;-1
		sbrc	ZL,0
		subi	tempreg2,0xa0		;+0x60
		sbrc	ZL,1
		subi	tempreg2,0x90		;+0x70
		sbrc	tempreg2,7
		ldi	tempreg2,0x70		;white
libmio_bchar_0:	mov	ZL,tempreg1		;char
		ldi	ZH,HIGH(libmio_table)	;start char ROM
		lsl	ZL
		lsl	ZL
		rol	ZH
		ldi	XH,5			;10 char lines
libmio_bchar_1:	clr	tempreg5
		clr	tempreg6
		clr	tempreg7
		lpm	XL,Z+			;get pixel line
		bst	XL,7			;get pixel 0
		bld	tempreg5,0
		bst	XL,3			;get pixel 3
		bld	tempreg6,1
		lpm	XL,Z+			;get pixel line
		bst	XL,7			;get pixel 1
		bld	tempreg5,1
		bst	XL,3			;get pixel 4
		bld	tempreg7,0
		lpm	XL,Z+			;get pixel line
		bst	XL,7			;get pixel 2
		bld	tempreg6,0
		bst	XL,3			;get pixel 5
		bld	tempreg7,1
		sbiw	ZL,3			;-3
		subi	ZH,0xfe			;+512
		lpm	XL,Z+			;get pixel line
		bst	XL,7			;get pixel 0
		bld	tempreg5,2
		bst	XL,3			;get pixel 3
		bld	tempreg6,3
		lpm	XL,Z+			;get pixel line
		bst	XL,7			;get pixel 1
		bld	tempreg5,3
		bst	XL,3			;get pixel 4
		bld	tempreg7,2
		lpm	XL,Z+			;get pixel line
		bst	XL,7			;get pixel 2
		bld	tempreg6,2
		bst	XL,3			;get pixel 5
		bld	tempreg7,3
		sbiw	ZL,3			;-3
		subi	ZH,0xfe			;+512
		add	tempreg5,tempreg2	;add color offset
		add	tempreg6,tempreg2	;add color offset
		add	tempreg7,tempreg2	;add color offset
		st	Y+,tempreg5
		st	Y+,tempreg6
		st	Y+,tempreg7
		adiw	YL,libmio_cols-3	;next line
		dec	XH
		brne	libmio_bchar_1
		pop	tempreg2
		bst	tempreg8,0
		pop	tempreg8
		pop	tempreg7
		pop	tempreg6
		pop	tempreg5
		pop	ZH
		pop	ZL
		rjmp	libmio_char_02


;-------------------------------------------------------------------------------
; output text from ROM (Z=text,nullterminated)
;-------------------------------------------------------------------------------
.ifdef use_libmio_orom
libmio_orom:	push	tempreg1	
libmio_orom_1:	lpm	tempreg1,Z+		;get char
		cpi	tempreg1,0		;end?
		brne	libmio_orom_2		;no
		pop	tempreg1
		ret				;the end
libmio_orom_2:	rcall	libmio_char		;write char to screen
		rjmp	libmio_orom_1		;next char	
.endif

;-------------------------------------------------------------------------------
; output text from ROM (following y,x,nullterminated text)
;-------------------------------------------------------------------------------
.ifdef use_libmio_othis
libmio_othis:	pop	ZH			;get address
		pop	ZL
		clc
		rol	ZL
		rol	ZH
		push	XL
		push	XH
		lpm	XH,Z+
		cpi	XH,0xff			;no position
		breq	libmio_othis_0
		lpm	XL,Z+
		rcall	libmio_gxy
libmio_othis_0:	pop	XH
		pop	XL
		push	tempreg1
libmio_othis_1:	lpm	tempreg1,Z+		;get char
		cpi	tempreg1,0		;end?
		brne	libmio_othis_2		;no
		pop	tempreg1
		clc
		ror	ZH
		ror	ZL
		ijmp
libmio_othis_2:	rcall	libmio_char		;write char to screen
		rjmp	libmio_othis_1		;next char	
.endif

;-------------------------------------------------------------------------------
; output text from RAM (Z=text,nullterminated)
;-------------------------------------------------------------------------------
.ifdef use_libmio_oram
libmio_oram:	push	tempreg1
		ld	tempreg1,Z+		;get char
		cpi	tempreg1,0		;end?
		brne	libmio_oram_1		;no
		pop	tempreg1
		ret				;the end
libmio_oram_1:	call	libmio_char		;write char to screen
		rjmp	libmio_oram		;next char	
.endif

;-------------------------------------------------------------------------------
; output text (multiple char) ZL=number
;-------------------------------------------------------------------------------
.ifdef use_libmio_mtext
libmio_mtext:	call	libmio_char		;write char to screen
		dec	ZL
		brne	libmio_mtext	
		ret
.endif

;-------------------------------------------------------------------------------
; set cursor to position XL:XH
;-------------------------------------------------------------------------------
.ifdef use_libmio_gotoxy
    libmio_gxy:	cpi	XL,libmio_cols
		brcs	libmio_gxy_01
		ldi	XL,libmio_cols-1
libmio_gxy_01:	cpi	XH,libmio_rows
		brcs	libmio_gxy_02
		ldi	XH,libmio_rows-1
libmio_gxy_02:	sts	libmio_cur_x,XL		;save X-position
		sts	libmio_cur_y,XH		;save Y-position
		ret
.endif

;-------------------------------------------------------------------------------
; plot  XL=X XH=Y  
;-------------------------------------------------------------------------------
.ifdef use_libmio_plot
libmio_plot:	push	XL			;save Y-reg
		push	XH
		push	YL			;save coords
		push	YH
		push	tempreg1
		push	tempreg2
		ldi	tempreg2,1		;bit left top
		cpi	XL,libmio_cols*2
		brcs	libmio_plot_1
		ldi	XL,libmio_cols*2-1
libmio_plot_1:		
		cpi	XH,libmio_rows*2
		brcs	libmio_plot_2
		ldi	XH,libmio_rows*2-1
libmio_plot_2:	sbrc	XH,0
		lsl	tempreg2
		sbrc	XH,0
		lsl	tempreg2
		sbrc	XL,0
		lsl	tempreg2
		lsr	XH			;Y/2
		lsr	XL			;X/2
		ldi	YL,libmio_cols
		mul	XH,YL
		ldi	YH,HIGH(libmio_vram)	;home position
		ldi	YL,LOW(libmio_vram)	;home position
		add	YL,XL
		adc	YH,byte0
		adc	YL,r0
		adc	YH,r1
		ld	XL,Y			;char on coord
		lds	XH,libmio_color		;color
		dec	XH			;-1		
		cpi	XH,0x80			;clear pixel?
		brcs	libmio_plot_4
		com	tempreg2
		ori	tempreg2,0xf0
		and	XL,tempreg2
		rjmp	libmio_plot_6
libmio_plot_4:	or	XL,tempreg2
		andi	XL,0x0f			;only 0x00-0x0f
		lds	XH,libmio_color
		dec	XH
		sbrc	XH,0
		ori	XL,0x60
		sbrc	XH,1
		ori	XL,0x70
libmio_plot_6:	cpi	XL,0x20			;space?
		brne	libmio_plot_7
		ldi	XL,0x70			;plot color=7
libmio_plot_7:	st	Y,XL
		pop	tempreg2		
		pop	tempreg1
		pop	YH
		pop	YL
		pop	XH
		pop	XL
		ret
.endif

;-------------------------------------------------------------------------------
; box YL,YH to ZL,ZH
;-------------------------------------------------------------------------------
.ifdef use_libmio_box
libmio_box:	push	XL
		push	XH
		push	YL
		push	YH
		push	ZL
		push	ZH
		push	tempreg2
		cp	ZL,YL
		brcc	libmio_box_1
		mov	tempreg2,ZL
		mov	ZL,YL
		mov	YL,tempreg2
libmio_box_1:	cp	ZH,YH
		brcc	libmio_box_2
		mov	tempreg2,ZH
		mov	ZH,YH
		mov	YH,tempreg2
libmio_box_2:	mov	XL,YL
		mov	XH,YH
libmio_box_3:	rcall	libmio_plot
		inc	XL
		cp	XL,ZL
		brne	libmio_box_3
libmio_box_4:	rcall	libmio_plot
		inc	XH
		cp	ZH,XH
		brne	libmio_box_4
libmio_box_5:	rcall	libmio_plot
		dec	XL
		cp	XL,YL
		brne	libmio_box_5
libmio_box_6:	rcall	libmio_plot
		dec	XH
		cp	XH,YH
		brne	libmio_box_6
		rjmp	libmio_fbox_6
.endif				

;-------------------------------------------------------------------------------
; fbox YL,YH to ZL,ZH
;-------------------------------------------------------------------------------
.ifdef use_libmio_fbox
libmio_fbox:	push	XL
		push	XH
		push	YL
		push	YH
		push	ZL
		push	ZH
		push	tempreg2
		cp	ZL,YL
		brcc	libmio_fbox_1
		mov	tempreg2,ZL
		mov	ZL,YL
		mov	YL,tempreg2
libmio_fbox_1:	cp	ZH,YH
		brcc	libmio_fbox_2
		mov	tempreg2,ZH
		mov	ZH,YH
		mov	YH,tempreg2
libmio_fbox_2:	mov	XH,YH
libmio_fbox_3:	mov	XL,YL
libmio_fbox_4:	rcall	libmio_plot
		cp	XL,ZL
		breq	libmio_fbox_5
		inc	XL
		rjmp	libmio_fbox_4
libmio_fbox_5:	rcall	libmio_plot
		cp	ZH,XH
		breq	libmio_fbox_6
		inc	XH
		rjmp	libmio_fbox_3
.endif
		
libmio_fbox_6:	pop	tempreg2
libmio_fbox_7:	pop	ZH
		pop	ZL
		pop	YH
		pop	YL
		pop	XH
		pop	XL
		ret				

;-------------------------------------------------------------------------------
; scroll XL lines 1 character left
;-------------------------------------------------------------------------------
.ifdef use_libmio_lscroll
libmio_lsc:	push	XL
		push	XH
		push	YL
		push	YH
		push	ZL
		push	ZH
		ldi	YL,LOW(libmio_vram)
		ldi	YH,HIGH(libmio_vram)	
		clr	ZL
libmio_lsc_01:	ldi	ZH,libmio_cols-1
libmio_lsc_02:	ldd	XH,Y+1
		st	Y+,XH
		dec	ZH
		brne	libmio_lsc_02
		st	Y+,byte0
		inc	ZL
		cp	ZL,XL
		breq	libmio_lsc_03
		cpi	ZL,libmio_rows
		brne	libmio_lsc_01
libmio_lsc_03:	rjmp	libmio_fbox_7
.endif

;-------------------------------------------------------------------------------
; scroll XL lines 1/2 character left (1 plot pixel)
;-------------------------------------------------------------------------------
.ifdef use_libmio_lpscroll
libmio_lps:	push	XL
		push	XH
		push	YL
		push	YH
		push	ZL
		push	ZH
		push	r16
		push	r17
		ldi	YL,LOW(libmio_vram)
		ldi	YH,HIGH(libmio_vram)	
		clr	ZL
libmio_lps_01:	ld	r16,Y
		ldi	ZH,libmio_cols-1
libmio_lps_02:	ldd	r17,Y+1
		bst	r16,1
		bld	r16,0
		bst	r17,0
		bld	r16,1
		bst	r16,3
		bld	r16,2
		bst	r17,2
		bld	r16,3
		mov	XH,r17
		andi	XH,0xf0
		andi	r16,0x0f
		or	r16,XH
		st	Y+,r16
		mov	r16,r17
		dec	ZH
		brne	libmio_lps_02
		bst	r17,1
		bld	r17,0
		bst	r17,3
		bld	r17,2
		andi	r17,0xf5
		st	Y+,r17
		inc	ZL
		cp	ZL,XL
		breq	libmio_lps_03
		cpi	ZL,libmio_rows
		brne	libmio_lps_01
libmio_lps_03:	pop	r17
		pop	r16
		rjmp	libmio_fbox_7
.endif
				
;-------------------------------------------------------------------------------
; wait for the end of visible frame
;-------------------------------------------------------------------------------
.ifdef use_libmio_sync
libmio_sync:	push	XL
libmio_sync_1:	lds	XL,libmio_line_l
		cpi	XL,libmio_pixel_rows+1
		breq	libmio_sync_1
libmio_sync_2:	lds	XL,libmio_line_l
		cpi	XL,libmio_pixel_rows+1
		brne	libmio_sync_2
		pop	XL
		ret		    
.endif

;-------------------------------------------------------------------------------
; play a note (XL=note 0..63 0xff=>noise)
;-------------------------------------------------------------------------------
.ifdef use_libmio_note
libmio_note:	push	ZL
		push	ZH
		push	XL
		sts	libmio_note_val,XL	;write note value
		andi	XL,0x3f			;64 notes
		lsl	XL
		lsl	XL
		ldi	ZH,HIGH(libmio_table*2+4611)	;1 start envtable high
		ldi	ZL,LOW(libmio_table*2+4611)	;1 start envtable low
		add	ZL,XL			;offsettable
		adc	ZH,byte0		;
		add	ZL,XL			;offsettable
		adc	ZH,byte0		;
		lpm	XL,Z+			;get low offset
		sts	libmio_woff_l,XL	;write
		adiw	ZL,3
		lpm	XL,Z+			;get high offset
		sts	libmio_woff_h,XL	;write
		sts	libmio_env,byte0	;start
		sts	libmio_wpos_l,byte0
		sts	libmio_wpos_h,byte0
		pop	XL 
		pop	ZH
		pop	ZL
		ret		    
.endif

;------------------------------------------------------------------------------
;wait for a key
;------------------------------------------------------------------------------
.ifdef use_libmio_waitkey
libmio_waitkey:	sts	libmio_lastkey,byte0
libmio_wkey1:	lds	tempreg1,libmio_lastkey
		cpi	tempreg1,0
		breq	libmio_wkey1
.ifdef use_libmio_screenshot		
		rjmp	libmio_scrs
.else
		ret
.endif				
.endif
		
;------------------------------------------------------------------------------
;blink cursor and wait for key (return x,y in X ; adr in Y ; key in tempreg1)
;------------------------------------------------------------------------------
.ifdef use_libmio_cursor
libmio_cursor:	rcall	libmio_calc		;calculate address
		ld	XL,Y			;get original char
		ldi	XH,libmio_curchar
		sts	libmio_lastkey,byte0	;clear key
libmio_cur_1:	lds	tempreg1,libmio_lastkey
		cpi	tempreg1,0
		brne	libmio_cur_e

		lds	tempreg1,libmio_frames
		cpi	tempreg1,27		;ca. half of fps
		brcc	libmio_cur_2
		st	Y,XL
		rjmp	libmio_cur_1
		
libmio_cur_2:	st	Y,XH
		rjmp	libmio_cur_1						
		
libmio_cur_e:	st	Y,XL			;put back original char
		
.ifdef use_libmio_screenshot		
		rcall	libmio_scrs
.endif				
		rjmp	libmio_calc
.endif			

;------------------------------------------------------------------------------
;calculate cursor address to Y
;------------------------------------------------------------------------------
libmio_calc:	lds	XL,libmio_cur_x		;X
		lds	XH,libmio_cur_y		;Y
libmio_calc1:	ldi	YH,libmio_cols		;chars per line
		mov	r1,YH			;store
		ldi	YH,HIGH(libmio_vram)	;home position
		ldi	YL,LOW(libmio_vram)	;home position
		add	YL,XL			;position+x
		adc	YH,byte0		
		mul	XH,r1			;offset from y
		add	YL,r0			;position+(cols*y)
		adc	YH,r1
		ret
		
;------------------------------------------------------------------------------
;put tempreg1 to serial output
;------------------------------------------------------------------------------
libmio_pser:
.ifdef use_libmio_pser
		push	XH
libmio_pser_1:	lds	XH,libmio_tcnt		;counter
		cpi	XH,0x00			;ready?
		brne	libmio_pser_1		;loop
		sts	libmio_tdat,tempreg1	;store data
		dec	XH			;prepare 0xff
		sts	libmio_tcnt,XH		;store start
		pop	XH
.endif
		ret				;thats all

;------------------------------------------------------------------------------
;get tempreg1 from serial output
;------------------------------------------------------------------------------
.ifdef use_libmio_gser
libmio_gser:	push	XH
libmio_gser_1:	lds	XH,libmio_rcnt		;counter
		cpi	XH,0x00			;ready?
		brne	libmio_gser_1		;loop
		dec	XH			;prepare 0xff
		sts	libmio_rcnt,XH		;store start
libmio_gser_2:	lds	XH,libmio_rcnt		;counter
		cpi	XH,0x00			;ready?
		brne	libmio_gser_2		;loop
		lds	tempreg1,libmio_rdat	;get char
		pop	XH
		ret				;thats all
.endif		

;------------------------------------------------------------------------------
;put tempreg1 to printer output
;------------------------------------------------------------------------------
libmio_ppar:
.ifdef use_libmio_ppar
		push	XH
		ldi	XH,0xff
		out	DDRA,XH			;set port A4-7 as outputs
		nop
libmio_ppar_1:	sbic	PINB,2			;busy
		rjmp	libmio_ppar_1		;yes
		out	PORTA,tempreg1		;data
		ldi	XH,100			;pause
libmio_ppar_2:	dec	XH
		brne	libmio_ppar_2
		cbi	PORTB,1			;clear strobe (active)
		ldi	XH,100			;pause
libmio_ppar_3:	dec	XH
		brne	libmio_ppar_3
		sbi	PORTB,1			;set strobe (inactive)
		pop	XH
.endif		
		ret				;thats all


;------------------------------------------------------------------------------
;output screenshot
;------------------------------------------------------------------------------
.ifdef use_libmio_screenshot
libmio_scrs_1:	pop	tempreg1
		ret
libmio_scrs:	push	tempreg1		;
		lds	tempreg1,libmio_kflags
		cpi	tempreg1,2		;screenshot-flag
		brne	libmio_scrs_1
		push	ZH			;save Z-register
		push	ZL			;save Z-register
		push	YH			;save Y-register
		push	YL			;save Y-register
		sts	libmio_kflags,byte0	;clear libmio_scrs-flag
libmio_scrs_2:	ldi	YL,LOW(libmio_vram)
		ldi	YH,HIGH(libmio_vram)
		ldi	ZH,libmio_rows		;number of charlines
libmio_scrs_3:	ldi	ZL,libmio_cols		;number of chars per line
libmio_scrs_4:	ld	tempreg1,Y+
		subi	tempreg1,0x80
		rcall	libmio_pser
		dec	ZL
		brne	libmio_scrs_4
		dec	ZH
		brne	libmio_scrs_3
		ldi	tempreg1,0x0a
		rcall	libmio_pser
		pop	YL
		pop	YH
		pop	ZL
		pop	ZH
		pop	tempreg1
		clr	tempreg1		;kill character		
		ret		
.endif

;------------------------------------------------------------------------------
;enable/disable chargepump
;------------------------------------------------------------------------------
.ifdef use_libmio_chp
libmio_chp:	push	XH
		ldi	XH,0x09			;disable output
		cpi	XL,0			;is zero
		breq	libmio_chp1
		ori	XH,0x10			;disable output
libmio_chp1:	out	TCCR0,XH
		pop	XH
		ret
.endif
		