;################################################################################
;#										#
;# chipbasic - single chip basic computer with ATMega 16			#
;# external memory interface							#
;# copyright (c) 2006 Joerg Wolfram (joerg@jcwolfram.de)			#
;#										#
;#										#
;# This program is free software; you can redistribute it and/or		#
;# modify it under the terms of the GNU General Public License			#
;# as published by the Free Software Foundation; either version 2		#
;# of the License, or (at your option) any later version.			#
;#										#
;# This program is distributed in the hope that it will be useful,		#
;# but WITHOUT ANY WARRANTY; without even the implied warranty of		#
;# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the GNU		#
;# General Public License for more details.					#
;#										#
;# You should have received a copy of the GNU General Public			#
;# License along with this library; if not, write to the			#
;# Free Software Foundation, Inc., 59 Temple Place - Suite 330,			#
;# Boston, MA 02111-1307, USA.							#
;#										#
;################################################################################

    .set	twibl = 92

;-------------------------------------------------------------------------------
; some macros for different controller types 
;-------------------------------------------------------------------------------
.macro get_twcr
		in	r17,TWCR
.endm

.macro put_twcr
		out	TWCR,r17
.endm

.macro get_twdr
		in	tempreg1,TWDR
.endm

.macro put_twdr
		out	TWDR,tempreg1

.endm

.macro put_twbr
		out	TWBR,r17
.endm

.macro get_twsr
		in	XH,TWSR
		andi	XH,0xf8
.endm

;-------------------------------------------------------------------------------
; initialize twi in low speed mode (<= 100kHz)
;-------------------------------------------------------------------------------
i2c_initl:	ldi	r17,twibl		;set bitrate
		put_twbr
		ldi	r17,(1<<TWINT) | (1<<TWEN)
		put_twcr

;-------------------------------------------------------------------------------
; generate stop condition
;-------------------------------------------------------------------------------
i2c_stop:	ldi	r17,(1<<TWSTO) | (1<<TWINT) | (1<<TWEN)
		put_twcr
		ret

;-------------------------------------------------------------------------------
; generate start condition
;-------------------------------------------------------------------------------
i2c_start:	ldi	r17,(1<<TWSTA) | (1<<TWINT) | (1<<TWEN)
		put_twcr
i2c_sta1:	get_twcr			;get control
		sbrs	r17,TWINT
		rjmp	i2c_sta1
		get_twsr			;get status					
		ret

;-------------------------------------------------------------------------------
; write byte to I2C
;-------------------------------------------------------------------------------
i2c_wbyte:	put_twdr
		ldi	r17,(1<<TWINT) | (1<<TWEN)
		put_twcr
i2c_wbyte1:	get_twcr			;get control
		sbrs	r17,TWINT
		rjmp	i2c_wbyte1
		get_twsr			;get status
		ret

;-------------------------------------------------------------------------------
; read byte from I2C (NACK)
;-------------------------------------------------------------------------------
i2c_rbyten:	ldi	r17,(1<<TWINT) | (1<<TWEN)
		put_twcr
i2c_reebn1:	get_twcr			;get control
		sbrs	r17,TWINT
		rjmp	i2c_reeb1
		get_twdr			;get data
		get_twsr			;get status
		ret

;-------------------------------------------------------------------------------
; read byte from I2C (ACK)
;-------------------------------------------------------------------------------
i2c_rbyte:	ldi	r17,(1<<TWINT) | (1<<TWEN) | (1<<TWEA)
		put_twcr
i2c_reeb1:	get_twcr			;get control
		sbrs	r17,TWINT
		rjmp	i2c_reeb1
		get_twdr			;get data
		get_twsr			;get status
		ret
				
;-------------------------------------------------------------------------------
; write memory address to I2C (program)
;-------------------------------------------------------------------------------
i2c_wad:	clr	ereg			;clear error
		rcall	i2c_start		;start condition
		cpi	XH,0x08
		brne	i2c_wad_e		;error
		mov	tempreg1,r16		;write devsel
		andi	tempreg1,0xfe		;clear rw bit
		rcall	i2c_wbyte		
		cpi	XH,0x18
		brne	i2c_wad_e		;error
		mov	tempreg1,ZH		;write MSB
		rcall	i2c_wbyte		
		cpi	XH,0x28
		brne	i2c_wad_e		;error
		mov	tempreg1,ZL		;write LSB
		rcall	i2c_wbyte		
		cpi	XH,0x28
		brne	i2c_wad_e		;error
		ret
i2c_wad_e:	ldi	ereg,10			;set error
		rjmp	i2c_stop		;stop condition and end

;-------------------------------------------------------------------------------
; set EEPROM to read
;-------------------------------------------------------------------------------
i2c_srd:	clr	ereg			;clear error
		rcall	i2c_start		;start condition
		cpi	XH,0x10
		brne	i2c_wad_e		;error
		mov	tempreg1,r16		;read devsel
		ori	tempreg1,1		;set rw bit
		rcall	i2c_wbyte		
		cpi	XH,0x40
		brne	i2c_wad_e		;error
		ret
					
;-------------------------------------------------------------------------------
; read 10 Bytes name to (Y+), XL is program no.
;-------------------------------------------------------------------------------
i2c_rname:	push 	r16
		push	r17
		push	r18
		andi	XL,0x0f
		lsl	XL
		ldi	ZL,LOW(500)
		ldi	ZH,HIGH(500)
		add	ZH,XL
		ldi	r16,0xa0		;EEPROM 0
		rcall	i2c_wad			;write address	
		cpi	ereg,0			;error?
		brne	i2c_rname_e
		rcall	i2c_srd			;set to read
		cpi	ereg,0			;error?
		brne	i2c_rname_e
		ldi	r18,9			;n-1
i2c_rname_1:	rcall	i2c_rbyte		;read byte with ACK
		st	Y+,tempreg1
		dec	r18
		brne	i2c_rname_1
		rcall	i2c_rbyten		;read byte with NAK
		st	Y+,tempreg1
		rcall	i2c_stop	
		libmio_sync			;wait 20ms
		libmio_sync	
i2c_rname_e:	pop	r18
		pop	r17
		pop	r16
		ret	

;-------------------------------------------------------------------------------
; copy 512 Bytes to internal EEPROM, XL is program no.
;-------------------------------------------------------------------------------
i2c_rprog:	push 	r16
		push	r17
		push	r18
		andi	XL,0x0f
		lsl	XL
		ldi	ZL,0
		mov	ZH,XL
		clr	YL			;internal EEPROM adr
		clr	YH
		ldi	r16,0xa0		;EEPROM 0
		rcall	i2c_wad			;write address	
		cpi	ereg,0			;error?
		brne	i2c_rprog_e
		rcall	i2c_srd			;set to read
		cpi	ereg,0			;error?
		brne	i2c_rprog_e
		ldi	r18,0x00		;256 bytes
i2c_rprog_1:	rcall	i2c_rbyte		;read byte with ACK
		call	ee_write
		dec	r18
		brne	i2c_rprog_1
		ldi	r18,0xff		;255 bytes
i2c_rprog_2:	rcall	i2c_rbyte		;read byte with ACK
		call	ee_write
		dec	r18
		brne	i2c_rprog_2
		rcall	i2c_rbyten		;read byte with NAK
		call	ee_write
		rcall	i2c_stop		
i2c_rprog_e:	pop	r18
		pop	r17
		pop	r16
		ret	

;-------------------------------------------------------------------------------
; write 512 Bytes from internal EEPROM, XL is program no.
;-------------------------------------------------------------------------------
i2c_wprog:	push 	r16
		push	r17
		push	r18
		push	r19
		andi	XL,0x0f
		lsl	XL
		ldi	ZL,0
		mov	ZH,XL			
		clr	YL			;internal EEPROM adr
		clr	YH
		ldi	r16,0xa0		;EEPROM 0
		ldi	tempreg4,16		;16 EE-pages
i2c_wprog_1:	rcall	i2c_wad			;write address	
		cpi	ereg,0			;error?
		brne	i2c_wprog_e
		ldi	r18,0x20		;32 bytes
i2c_wprog_2:	call	ee_read
		rcall	i2c_wbyte		;read byte with ACK
		dec	r18
		brne	i2c_wprog_2
		rcall	i2c_stop
		libmio_sync			;wait 40ms
		libmio_sync
		libmio_sync
		adiw	ZL,32
		dec	tempreg4
		brne	i2c_wprog_1		
		
i2c_wprog_e:	pop	r19
		pop	r18
		pop	r17
		pop	r16
		ret	

;-------------------------------------------------------------------------------
; read data word from external eeprom X = (Y)
;-------------------------------------------------------------------------------
i2c_rword:	push 	r16
		push	r17
		push	ZH			;save Z-register
		push	ZL
		mov	ZL,YL			;copy pointer
		mov	ZH,YH
		ldi	r16,0xa2		;EEPROM 1
		rcall	i2c_wad			;write address	
		cpi	ereg,0			;error?
		brne	i2c_rword_err
		rcall	i2c_srd			;set to read
		cpi	ereg,0			;error?
		brne	i2c_rword_err
		rcall	i2c_rbyte		;read byte with ACK
		mov	XL,tempreg1
		rcall	i2c_rbyten		;read byte with NAK
		mov	XH,tempreg1
		rjmp	i2c_rword_e
i2c_rword_err:	ldi	ereg,16			;I2C error		
i2c_rword_e:	rcall	i2c_stop		
		pop	ZL
		pop	ZH
		pop	r17
		pop	r16
		ret	

;-------------------------------------------------------------------------------
; write data word to external eeprom (Y) = X
;-------------------------------------------------------------------------------
i2c_wword:	push 	r16
		push	r17
		push	ZH			;save Z-register
		push	ZL
		mov	ZL,YL			;copy pointer
		mov	ZH,YH
		ldi	r16,0xa2		;EEPROM 1
		rcall	i2c_wad			;write address	
		cpi	ereg,0			;error?
		brne	i2c_rword_err
		mov	r16,XL
		rcall	i2c_wbyte		;read byte with ACK
		mov	r16,XH
		rcall	i2c_wbyte		;read byte with NAK
		rjmp	i2c_rword_e

;-------------------------------------------------------------------------------
; read temperature from external lm75 sensor no. X to X
;-------------------------------------------------------------------------------
i2c_rtemp:	push 	r16
		push	r17
		push	ZH			;save Z-register
		push	ZL
		andi	XL,0x07			;8 addresses
		lsl	XL			;set correct devsel
		ori	XL,0x91
		clr	ereg			;clear error
		rcall	i2c_start		;start condition
		cpi	XH,0x08
		brne	i2c_rword_err		;error
		mov	tempreg1,XL		;write devsel
		rcall	i2c_wbyte		
		cpi	XH,0x40			;error?
		brne	i2c_rword_err
		rcall	i2c_rbyte		;read byte with ACK
		mov	XL,tempreg1
		rcall	i2c_rbyten		;read byte with NAK
		clr	XH
		lsl	tempreg1
		rol	XL
		brcc	i2c_rtemp_1
		com	XH
i2c_rtemp_1:	rjmp	i2c_rword_e
	
