;################################################################################
;#										#
;# fsys - file system library							#
;# copyright (c) 2011 Joerg Wolfram (joerg@jcwolfram.de)			#
;#										#
;# This library is free software; you can redistribute it and/or		#
;# modify it under the terms of the GNU  General Public				#
;# License as published by the Free Software Foundation; either			#
;# version 3 of the License, or (at your option) any later version.		#
;#										#
;# This library is distributed in the hope that it will be useful,		#
;# but WITHOUT ANY WARRANTY; without even the implied warranty of		#
;# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU		#
;# General Public License for more details.					#
;#										#
;# You should have received a copy of the GNU  General Public			#
;# License along with this library; if not, write to the			#
;# Free Software Foundation, Inc., 59 Temple Place - Suite 330,			#
;# Boston, MA 02111-1307, USA.							#
;#										#
;################################################################################

;------------------------------------------------------------------------------
; init filesystem and write messages
;------------------------------------------------------------------------------
fsys_init:	ldi	XL,0
		ldi	XH,0
		out	SPCR,const_0
		sbi	SPI_SPORT,SPI_SPIN	;set port pins
		sbi	SPI_SDDR,SPI_SPIN
		sbi	MOSI_DDR,MOSI_PIN
		sbi	SCK_DDR,SCK_PIN
		sts	fsys_tapenr,const_0	;start with tape 0
		sts	fsys_status,const_0	;card not available
		sts	fsys_maxtape,const_0	;card not available

fsys_init_0:	ldi	ZL,LOW(fsys_buf2)
		ldi	ZH,HIGH(fsys_buf2)
		ldi	XH,10
		ldi	XL,0x20
fsys_init_0a:	st	Z+,XL
		dec	XH
		brne	fsys_init_0a

fsys_init_1:	nop
		nop
		nop
		sbiw	XL,1
		brne	fsys_init_1

		rcall	sdfs_init		;check for SD card

		ldi	XL,1
		ldi	XH,22
		call	sys_gotoxy

		rcall	fsys_pstat		;view type
		lds	XL,fsys_status		;status
		sbrs	XL,7
		ret				;no format check

		rcall	fsys_nfind		;find image (new version)

		ldi	XL,14
		ldi	XH,22
		sts	vsys_xpos,XL
		sts	vsys_ypos,XH
		rcall	fsys_fstat		;format status
;		lds	r20,fsys_status
;		rcall	fsys_hexout
		ret


;------------------------------------------------------------------------------
; wait for SPI is ready for next data
; write XL to SPI
;------------------------------------------------------------------------------
fsys_write_ff2:	rcall	fsys_write_ff
fsys_write_ff:	ldi	XL,0xff			;dummy data
fsys_wspi:	out	SPDR,XL
fsys_wait:	in	XL,SPSR
		sbrs	XL,SPIF
		rjmp	fsys_wait
		in	XL,SPDR
		ret

fsys_write_00:	clr	XL
		rjmp	fsys_wspi


;------------------------------------------------------------------------------
; print out filesystem status
;------------------------------------------------------------------------------
fsys_pstat:	lds	XL,fsys_status
		ldi	ZL,LOW(fsys_txt1*2)
		ldi	ZH,HIGH(fsys_txt1*2)
		cpi	XL,0x01			;card error
		breq	fsys_ppstat
		adiw	ZL,12
		cpi	XL,0x81			;MMC
		breq	fsys_ppstat
		adiw	ZL,12
		cpi	XL,0x82			;SDV1
		breq	fsys_ppstat
		adiw	ZL,12
		cpi	XL,0x83			;SDV2
		breq	fsys_ppstat
		adiw	ZL,12
		cpi	XL,0x84			;SDHC
		breq	fsys_ppstat
		adiw	ZL,12
		cpi	XL,0x02			;ACMD41 ERR
		breq	fsys_ppstat
		adiw	ZL,12
		cpi	XL,0x03			;CMD1 ERR
		breq	fsys_ppstat
		adiw	ZL,12
		cpi	XL,0x04			;PATTERN ERROR
		breq	fsys_ppstat
		adiw	ZL,12
fsys_ppstat:	call	sys_flashtext
fsys_pstat_e:	ret


fsys_txt1:	.db "CARD ERROR ",0x00		;CARD ERR
		.db "MMC-CARD   ",0x00		;MMC-CARD
		.db "SDV1-CARD  ",0x00		;SDV1-CARD
		.db "SDV2-CARD  ",0x00		;SDV2-CARD
		.db "SDHC-CARD  ",0x00		;SDHC-CARD
		.db "ERR-ACMD41 ",0x00		;E-ACMD41
		.db "ERR-CMD1   ",0x00		;E-CMD1
		.db "ERR-CMD8   ",0x00		;E-CMD8
		.db "NO CARD    ",0x00		;NO CARD


;------------------------------------------------------------------------------
; print out filesystem status2
;------------------------------------------------------------------------------
fsys_fstat:	lds	XL,fsys_status
		ldi	ZL,LOW(fsys_txt2*2)
		ldi	ZH,HIGH(fsys_txt2*2)
		sbrs	XL,7
		adiw	ZL,16
		rjmp	fsys_ppstat

fsys_txt2:	.db "AX82-SPEC IMAGE",0x00	;AX-IMAGE
		.db "NO IMAGE FOUND ",0x00	;NO IMAGE

;------------------------------------------------------------------------------
; write command XH to SPI
; XL=return byte (FF=time out)
;------------------------------------------------------------------------------
sdfs_cmd0:	clr	r16			;clear address field
		clr	r17
		movw	r18,r16

sdfs_cmd:	cbi	SPI_SPORT,SPI_SPIN	;CS active
		rcall	fsys_write_ff2		;16 dummy clocks
		andi	XH,0x3f			;limit
		ori	XH,0x40			;set bit 6
		out	SPDR,XH			;CMD
		andi	XH,0x3f			;clear bit 6
		rcall	fsys_wait
		out	SPDR,r19		;address MSB
		rcall	fsys_wait
		out	SPDR,r18
		rcall	fsys_wait
		out	SPDR,r17
		rcall	fsys_wait
		out	SPDR,r16		;address LSB
		rcall	fsys_wait
		ldi	XL,0x95			;dummy checksum
		cpi	XH,0x08
		brne	sdfs_cmd_01
		ldi	XL,0x87
sdfs_cmd_01:	rcall	fsys_wspi
		clr	r0

sdfs_cmd_resp:	rcall	fsys_write_ff		;write 0xff and read data
		cpi	XL,0xff
		brne	sdfs_cmd_end
		dec	r0
		brne	sdfs_cmd_resp
sdfs_cmd_end:	ret

;------------------------------------------------------------------------------
; wait for start block mtoken
;------------------------------------------------------------------------------
fsys_waitsbt:	push	ZL
		push	ZH
		ldi	ZL,0			;timeout
		ldi	ZH,0

fsys_waitsbt1:	cpi	XL,0xfe			;start block token
		brne	fsys_waitsbt2
		pop	ZH
		pop	ZL
		ret				;SBT found

fsys_waitsbt2:	sbiw	ZL,1			;timeout counter
		breq	fsys_waitsbt3		;branch if timeout is reached
		rcall	fsys_write_ff
		rjmp	fsys_waitsbt1		;next try

fsys_waitsbt3:	pop	ZH
		pop	ZL
		ret

;------------------------------------------------------------------------------
; init SD card and determine type
;------------------------------------------------------------------------------
sdfs_init:	;init SPI to f/128
		ldi	XL,0x53			;f/128
		out	SPCR,XL
		ldi	XL,0x04			;timeout factor

		;80 dummy pulses with CS=HI
sdfs_init_00:	ldi	XH,10
		sbi	SPI_SPORT,SPI_SPIN	;CS inactive
sdfs_init_01:	rcall	fsys_write_ff		;128 clock pulses
		dec	XH
		brne	sdfs_init_01

		ldi	YL,0x08			;CMD0 retries
		ldi	YH,0x00

sdfs_init_02:	ldi	XH,0			;CMD0
		rcall	sdfs_cmd0		;send CMD0
		cpi	XL,0xff
		breq	sdfs_init_02a
		sbrc	XL,0			;skip if idle state
		rjmp	sdfs_init_03		;yes->OK
sdfs_init_02a:	sbiw	YL,1			;retry counter
		brne	sdfs_init_02		;retry CMD 0

sdfs_init_err:	sts	fsys_status,const_0	;card not available
		sbi	SPI_SPORT,SPI_SPIN	;CS inactive
		ret				;exit

		;check for SDv2 with CMD8
sdfs_init_03:	ldi	r16,0xaa		;test pattern
		ldi	r17,0x01
		clr	r18
		clr	r19
		ldi	XH,8			;CMD8
		rcall	sdfs_cmd		;send CMD8
		sbrc	XL,2			;illegal?
		rjmp	sdfs_init_04		;yes
		rcall	fsys_write_ff2
		rcall	fsys_write_ff2
		cpi	XL,0xaa			;test pattern
		brne	sdfs_init_err1		;fail
		ldi	XL,0x90			;SDv2
		sts	fsys_status,XL		;write status
		rjmp	sdfs_init_05		;goto wait idle

		;check for MMC / SD
sdfs_init_04:	ldi	XH,55			;CMD55
		rcall	sdfs_cmd0		;send CMD55
		ldi	XH,1			;ACMD41
		rcall	sdfs_cmd0		;send ACMD41
		sbrc	XL,2			;illegal?
		rjmp	sdfs_init_20		;MMC
		ldi	XL,0x82			;SDv1
		sts	fsys_status,XL		;write status

		;wait for card ready
sdfs_init_05:	ldi	YL,0
		ldi	YH,0x80

sdfs_init_06:	ldi	XH,55			;CMD55
		rcall	sdfs_cmd0		;send CMD55
		lds	XL,fsys_status		;get stat
		sbrc	XL,4			;check V2
		ldi	r19,0x40
		ldi	XH,1			;ACMD41
		rcall	sdfs_cmd		;send ACMD41
		sbrs	XL,0			;skip if idle state
		rjmp	sdfs_init_10		;yes->OK
		sbiw	YL,1			;loops
		brne	sdfs_init_06

sdfs_init_07:	ldi	r16,0x02		;ACMD41 err
		sts	fsys_status,r16
		sbi	SPI_SPORT,SPI_SPIN	;CS inactive
		ret

sdfs_init_err1:	ldi	XL,0x04
		sts	fsys_status,XL		;card error
		sbi	SPI_SPORT,SPI_SPIN	;CS inactive
		ret				;exit

sdfs_init_10:	lds	XL,fsys_status		;get stat
		sbrs	XL,4			;
		rjmp	sdfs_init_end		;SDv1 card
		ldi	XL,0x83			;set SDv2
		sts	fsys_status,XL		;write status

		;now check for SDHC or V2
		ldi	XH,58			;CMD58
		rcall	sdfs_cmd0		;send CMD58
		rcall	fsys_write_ff
		push	XL
		rcall	fsys_write_ff
		rcall	fsys_write_ff
		rcall	fsys_write_ff
		pop	XL
		sbrs	XL,6
		rjmp	sdfs_init_end		;is SDv2 card
		ldi	XL,0x84			;SDHC card
		sts	fsys_status,XL		;write status

sdfs_init_end:	clr	r16			;512 Bytes block size
		ldi	r17,2
		clr	r18
		clr	r19
		ldi	XH,16			;CMD16
		rcall	sdfs_cmd		;send CMD16
		sbi	SPI_SPORT,SPI_SPIN	;CS inactive
		ldi	XL,0x50			;f/4
		out	SPCR,XL
		ret				;exit

sdfs_init_20:	ldi	YL,0
		ldi	YH,0x80

sdfs_init_21:	ldi	XH,1			;CMD1
		rcall	sdfs_cmd0		;send CMD1
		sbrs	XL,0			;skip if idle state
		rjmp	sdfs_init_22		;yes->OK
		sbiw	YL,1			;loops
		brne	sdfs_init_21		
		ldi	r16,0x03		;CMD1 err
		sts	fsys_status,r16
		sbi	SPI_SPORT,SPI_SPIN	;CS inactive
		ret

sdfs_init_22:	ldi	XL,0x81			;MMC card
		sts	fsys_status,XL		;write status
		rjmp	sdfs_init_end


;------------------------------------------------------------------------------
; calc sector address to r16..r19
; fsys_tapenr	= tape
; fsys_filenr	= file
; fsys_sectnr	= sector
;------------------------------------------------------------------------------
fsys_sectaddr0:	sts	fsys_sectnr,const_0	;first sector
fsys_sectaddr:	lds	XL,fsys_sectnr
		lds	XH,fsys_offset_l
		add	XL,XH
		sts	fsys_sectnr0,XL
		lds	XL,fsys_offset_m
		adc	XL,const_0
		sts	fsys_sectnr1,XL
		lds	XL,fsys_offset_h
		adc	XL,const_0
		sts	fsys_sectnr2,XL
		lds	XL,fsys_offset_e
		adc	XL,const_0
		sts	fsys_sectnr3,XL

		lds	YL,fsys_tapenr
		ldi	ZL,LOW(3104)
		mul	YL,ZL
		movw	XL,r0
		ldi	ZL,HIGH(3104)
		mul	YL,ZL
		clr	YL
		clr	YH
		add	XH,r0
		adc	YL,r1
		adc	YH,const_0
		ldi	ZL,0x61
		lds	r0,fsys_filenr
		mul	ZL,r0
		add	XL,r0
		adc	XH,r1
		adc	YL,const_0
		adc	YH,const_1
		rcall	fsys_addsec
		ret


;------------------------------------------------------------------------------
; read name
; fsys_tapenr	= tape
; fsys_filenr	= file
;------------------------------------------------------------------------------
fsys_rname:	sts	fsys_sectnr,const_0	;start with sector 0
		rcall	fsys_rsector
		lds	XL,fsys_buffer+16
		cpi	XL,0xcc
		brne	fsys_dname
		ret

fsys_dname:	ldi	YL,LOW(fsys_buffer)
		ldi	YH,HIGH(fsys_buffer)
		ldi	XH,12
		ldi	XL,'.'
fsys_dname_1:	st	Y+,XL
		dec	XH
		brne	fsys_dname_1
		ret

;------------------------------------------------------------------------------
; file error
;------------------------------------------------------------------------------
fsys_ferror:	ret
; delete file

fsys_delfile:	ldi	YL,LOW(fsys_buffer)
		ldi	YH,HIGH(fsys_buffer)
		ldi	XH,12
		ldi	XL,0x20
fsys_delfile_1:	st	Y+,XL
		dec	XH
		brne	fsys_delfile_1
		ldi	XH,250
		ldi	XL,0xff
fsys_delfile_2:	st	Y+,XL
		st	Y+,XL
		dec	XH
		brne	fsys_delfile_2

		rcall	fsys_sectaddr0		;calc sector addr
		rjmp	fsys_swrite

;------------------------------------------------------------------------------
; read sector
; fsys_tapenr	= tape
; fsys_filenr	= file
; fsys_sectnr	= sector
;------------------------------------------------------------------------------
fsys_rsector:	rcall	fsys_sectaddr		;calc sector addr
		rjmp	fsys_sread

;------------------------------------------------------------------------------
; write sector
; fsys_tapenr	= tape
; fsys_filenr	= file
; fsys_sectnr	= sector
;------------------------------------------------------------------------------
fsys_wsector:	rcall	fsys_sectaddr		;calc sector addr
		rjmp	fsys_swrite

;------------------------------------------------------------------------------
; read file
;------------------------------------------------------------------------------
fsys_readfile:	ldi	XL,0
		ldi	XH,23
		call	sys_gotoxy
		ldi	XL,0x20
		sts	vsys_color,XL

		sts	fsys_sectnr,const_0	;start with sector 0
		rcall	fsys_rsector

		ldi	YL,LOW(fsys_buffer)
		ldi	YH,HIGH(fsys_buffer)
		ldd	XL,Y+16			;valid flag
		cpi	XL,0xcc
		breq	fsys_rfile_0
		ret

fsys_rfile_0:	ldi	ZL,LOW(fsys_buf2)
		ldi	ZH,HIGH(fsys_buf2)
		ldi	XH,12
fsys_rfile_0a:	ld	XL,Y+
		st	Z+,XL
		dec	XH
		brne	fsys_rfile_0a

		ldi	YL,LOW(fsys_buffer+256)
		ldi	YH,HIGH(fsys_buffer+256)
		ld	reg_a,Y			;get A register (offset 0)
		ldd	reg_f,Y+1		;get F register (offset 1)

		ldd	reg_c,Y+2		;get C register (offset 2)
		ldd	reg_b,Y+3		;get B register (offset 3)

		ldd	reg_l,Y+4		;get L register (offset 4)
		ldd	reg_H,Y+5		;get H register (offset 5)

		ldd	PC_L,Y+6		;get PC register (offset 6)
		ldd	PC_H,Y+7

		ldd	SP_L,Y+8		;get SP register (offset 8)
		ldd	SP_H,Y+9

		ldd	XL,Y+10			;get i register (offset 10)
		sts	zreg_i,XL

		ldd	XL,Y+11			;get r register (offset 11)
		andi	XL,0x7f
		ldd	XH,Y+12			;get switch value (offset 12)
		sbrc	XH,0
		ori	XL,0x80
		sts	zreg_r,XL
		lsr	XH
		andi	XH,0x07
		mov	XL,XH
		swap	XL
		lsr	XL
		or	XL,XH
		sts	mon_tborder,XL		;set border

		ldd	reg_e,Y+13		;get E register (offset 13)
		ldd	reg_d,Y+14		;get D register (offset 14)

		ldd	XL,Y+15			;get C' register (offset 15)
		sts	zreg_c1,XL
		ldd	XL,Y+16			;get B' register
		sts	zreg_b1,XL

		ldd	XL,Y+17			;get E' register
		sts	zreg_e1,XL
		ldd	XL,Y+18			;get D' register
		sts	zreg_d1,XL

		ldd	XL,Y+19			;get L' register
		sts	zreg_l1,XL
		ldd	XL,Y+20			;get H' register
		sts	zreg_h1,XL

		ldd	XL,Y+21			;get A' register
		sts	zreg_a1,XL
		ldd	XL,Y+22			;get F' register
		sts	zreg_f1,XL

		ldd	XL,Y+23			;get IY register
		sts	zreg_iyl,XL
		ldd	XL,Y+24
		sts	zreg_iyh,XL

		ldd	XL,Y+25			;get IX register
		sts	zreg_ixl,XL
		ldd	XL,Y+26
		sts	zreg_ixh,XL

		ldd	XL,Y+27			;int alowed?
		sts	z80_iflag,XL

		ldd	XL,Y+29			;int mode?
		andi	XL,0x03			;limit
		sts	z80_imode,XL

		ldi	XL,0x40
		sts	fsys_madrh,XL

fsys_rfile_01:	lds	XL,fsys_sectnr		;next sector
		inc	XL
		sts	fsys_sectnr,XL
		cpi	XL,97			;beyond last
		brne	fsys_rfile_01a
		jmp	fsys_ll_clow
fsys_rfile_01a:	andi	XL,0x03
		brne	fsys_rfile_02
		ldi	r19,'*'
		call	sys_char
fsys_rfile_02:	rcall	fsys_rsector		;read sector
		lds	ZH,fsys_sectnr
		dec	ZH			;start with 0
		lsl	ZH
		subi	ZH,0xc0
		ldi	ZL,0x00
		ldi	YL,LOW(fsys_buffer)
		ldi	YH,HIGH(fsys_buffer)
		ldi	XH,0
fsys_rfile_03:	ld	XL,Y+
		putmembyte XL,ZL,ZH
		adiw	ZL,1
		ld	XL,Y+
		putmembyte XL,ZL,ZH
		adiw	ZL,1
		dec	XH
		brne	fsys_rfile_03
		rjmp	fsys_rfile_01


;------------------------------------------------------------------------------
; write file
;------------------------------------------------------------------------------
fsys_writefile:	ldi	XL,0
		ldi	XH,23
		call	sys_gotoxy
		ldi	XL,0x20
		sts	vsys_color,XL

		ldi	YL,LOW(fsys_buffer)
		ldi	YH,HIGH(fsys_buffer)
		ldi	XL,0xff
		ldi	XH,0x00
fsys_wfile_0a:	st	Y+,XL
		st	Y+,XL
		dec	XH
		brne	fsys_wfile_0a

		subi	YH,2

		ldi	ZL,LOW(fsys_buf2)
		ldi	ZH,HIGH(fsys_buf2)
		ldi	XH,12
fsys_wfile_0b:	ld	XL,Z+
		st	Y+,XL
		dec	XH
		brne	fsys_wfile_0b

		ldi	XL,0xcc
		sts	fsys_buffer+16,XL	;set as valid

		ldi	YL,LOW(fsys_buffer+256)
		ldi	YH,HIGH(fsys_buffer+256)
		st	Y,reg_a			;put A register (offset 0)
		std	Y+1,reg_f		;put F register (offset 1)

		std	Y+2,reg_c		;put C register (offset 2)
		std	Y+3,reg_b		;put B register (offset 3)

		std	Y+4,reg_l		;put L register (offset 4)
		std	Y+5,reg_h		;put H register (offset 5)

		std	Y+6,PC_L		;put PC register (offset 6)
		std	Y+7,PC_H

		std	Y+8,SP_L		;put SP register (offset 8)
		std	Y+9,SP_H

		lds	XL,zreg_i
		std	Y+10,XL			;put i register (offset 10)

		lds	XL,zreg_r
		andi	XL,0x7f
		std	Y+11,XL			;put r register (offset 11)

		lds	XL,zreg_r
		lds	XH,mon_tborder
		lsl	XH
		andi	XH,0x0e
		sbrc	XL,7
		ori	XH,0x01
		std	Y+12,XH

		std	Y+13,reg_e		;put E register (offset 13)
		std	Y+14,reg_d		;put D register (offset 14)

		lds	XL,zreg_c1		;put BC' register
		lds	XH,zreg_b1
		std	Y+15,XL
		std	Y+16,XH

		lds	XL,zreg_e1		;put DE' register
		lds	XH,zreg_d1
		std	Y+17,XL
		std	Y+18,XH

		lds	XL,zreg_l1		;put HL' register
		lds	XH,zreg_h1
		std	Y+19,XL
		std	Y+20,XH

		lds	XL,zreg_a1		;put AF' register
		lds	XH,zreg_f1
		std	Y+21,XL
		std	Y+22,XH

		lds	XL,zreg_iyl		;put IY register
		lds	XH,zreg_iyh
		std	Y+23,XL
		std	Y+24,XH

		lds	XL,zreg_ixl		;put IX register
		lds	XH,zreg_ixh
		std	Y+25,XL
		std	Y+26,XH

		lds	XL,z80_iflag
		andi	XL,0x01
		std	Y+27,XL

		std	Y+28,XL

		lds	XL,z80_imode
		andi	XL,0x03
		std	Y+29,XL

		sts	fsys_sectnr,const_0	;start with sector 0
		rcall	fsys_wsector

fsys_wfile_01:	lds	XL,fsys_sectnr
		inc	XL
		sts	fsys_sectnr,XL
		cpi	XL,97			;beyond last
		brne	fsys_wfile_01a
		jmp	fsys_ll_clow
fsys_wfile_01a:	andi	XL,0x03
		brne	fsys_wfile_02
		ldi	r19,'*'
		call	sys_char
fsys_wfile_02:	lds	ZH,fsys_sectnr
		dec	ZH
		lsl	ZH
		subi	ZH,0xc0
		ldi	ZL,0x00
		ldi	YL,LOW(fsys_buffer)
		ldi	YH,HIGH(fsys_buffer)
		ldi	XH,0
fsys_wfile_03:	getmembyte XL,ZL,ZH
		st	Y+,XL
		adiw	ZL,1
		getmembyte XL,ZL,ZH
		st	Y+,XL
		adiw	ZL,1
		dec	XH
		brne	fsys_wfile_03

		rcall	fsys_wsector		;write sector
		rjmp	fsys_wfile_01

