//###############################################################################
//#										#
//# MXE11 PDP11-Unix Emulator for Mikrocontrollers				#
//#										#
//# copyright (c) 2017 Joerg Wolfram (joerg@jcwolfram.de)			#
//#										#
//#										#
//# This program is free software; you can redistribute it and/or		#
//# modify it under the terms of the GNU General Public License			#
//# as published by the Free Software Foundation; either version 3		#
//# of the License, or (at your option) any later version.			#
//#										#
//# This program is distributed in the hope that it will be useful,		#
//# but WITHOUT ANY WARRANTY; without even the implied warranty of		#
//# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the GNU		#
//# General Public License for more details.					#
//#										#
//# You should have received a copy of the GNU General Public			#
//# License along with this library// if not, write to the			#
//# Free Software Foundation, Inc., 59 Temple Place - Suite 330,		#
//# Boston, MA 02111-1307, USA.							#
//#										#
//###############################################################################
#include "board.h"
#include "cpu.h"
#include "xvars.h"

unsigned short dac_buffer[4];
unsigned char dac_flag[4];
unsigned short adc_buffer[4];
unsigned char adc_flag[4];


//-------------------------------------------------------------------------------
// PC simulates only dummy devices
//-------------------------------------------------------------------------------
#if MCU < 10
void start_adc()
{
	int i;
	for(i=0;i<4;i++)
	{
		adc_buffer[i]=0;
		adc_flag[i]=0;
	}
}

void stop_adc()
{

}

unsigned short read_adc0()
{
	return 111;
}

unsigned short read_adc1()
{
	return 222;
}

unsigned short read_adc2()
{
	return 333;
}

unsigned short read_adc3()
{
	return 444;
}


unsigned char aread_adc0()
{
	char c;
	
	switch(adc_flag[0])
	{
		case 0:		adc_buffer[0]=111;
				c=adc_buffer[0] / 1000;
				adc_buffer[0]-=(c *1000);
				adc_flag[0]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[0] / 100;
				adc_buffer[0]-=(c *100);
				adc_flag[0]++;	
				return(c+48);

		case 2:		c=adc_buffer[0] / 10;
				adc_buffer[0]-=(c *10);
				adc_flag[0]++;	
				return(c+48);

		case 3:		c=adc_buffer[0];
				adc_flag[0]++;	
				return(c+48);
				
		default:	adc_flag[0]=0;	
				return(10);
				
	}					
	return 0;					
}

unsigned char aread_adc1()
{
	char c;
	
	switch(adc_flag[1])
	{
		case 0:		adc_buffer[1]=222;
				c=adc_buffer[1] / 1000;
				adc_buffer[1]-=(c *1000);
				adc_flag[1]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[1] / 100;
				adc_buffer[1]-=(c *100);
				adc_flag[1]++;	
				return(c+48);

		case 2:		c=adc_buffer[1] / 10;
				adc_buffer[1]-=(c *10);
				adc_flag[1]++;	
				return(c+48);

		case 3:		c=adc_buffer[1];
				adc_flag[1]++;	
				return(c+48);
				
		default:	adc_flag[1]=0;	
				return(10);
				
	}					
	return 0;					
}


unsigned char aread_adc2()
{
	char c;
	
	switch(adc_flag[2])
	{
		case 0:		adc_buffer[2]=333;
				c=adc_buffer[2] / 1000;
				adc_buffer[2]-=(c *1000);
				adc_flag[2]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[2] / 100;
				adc_buffer[2]-=(c *100);
				adc_flag[2]++;	
				return(c+48);

		case 2:		c=adc_buffer[2] / 10;
				adc_buffer[2]-=(c *10);
				adc_flag[2]++;	
				return(c+48);

		case 3:		c=adc_buffer[2];
				adc_flag[2]++;	
				return(c+48);
				
		default:	adc_flag[2]=0;	
				return(10);
				
	}
	return 0;					
}

unsigned char aread_adc3()
{
	char c;
	
	switch(adc_flag[3])
	{
		case 0:		adc_buffer[3]=444;
				c=adc_buffer[3] / 1000;
				adc_buffer[3]-=(c *1000);
				adc_flag[3]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[3] / 100;
				adc_buffer[3]-=(c *100);
				adc_flag[3]++;	
				return(c+48);

		case 2:		c=adc_buffer[3] / 10;
				adc_buffer[3]-=(c *10);
				adc_flag[3]++;	
				return(c+48);

		case 3:		c=adc_buffer[3];
				adc_flag[3]++;	
				return(c+48);
				
		default:	adc_flag[3]=0;	
				return(10);
				
	}					
	return 0;					
}

void start_dac()
{
	int i;
	for(i=0;i<4;i++)
	{
		dac_buffer[i]=0;
		dac_flag[i]=0;
	}
}

void stop_dac()
{
}

void write_dac0(unsigned short data)
{
	printf("PWM0: %d\n",data);
}

void write_dac1(unsigned short data)
{
	printf("PWM1: %d\n",data);
}

void write_dac2(unsigned short data)
{
	printf("PWM2: %d\n",data);
}

void write_dac3(unsigned short data)
{
	printf("PWM3: %d\n",data);
}

void awrite_dac0(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[0]=dac_buffer[0]*10+data-0x30;
		dac_flag[0]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[0]==1) 
		{
			if(dac_buffer[0] > 1024) dac_buffer[0]=1024;
			printf("PWM0: %d\n",dac_buffer[0]);
		}
		dac_flag[0]=0;
		dac_buffer[0]=0;
	}
}

void awrite_dac1(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[1]=dac_buffer[1]*10+data-0x30;
		dac_flag[1]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[1]==1)
		{
			if(dac_buffer[1] > 1024) dac_buffer[1]=1024;
			printf("PWM1: %d\n",dac_buffer[1]);
		}
		dac_flag[1]=0;
		dac_buffer[1]=0;
	}
}


void awrite_dac2(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[2]=dac_buffer[2]*10+data-0x30;
		dac_flag[2]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[2]==1)
		{
			if(dac_buffer[2] > 1024) dac_buffer[2]=1024;
			printf("PWM2: %d\n",dac_buffer[2]);
		}
		dac_flag[2]=0;
		dac_buffer[2]=0;
	}
}

void awrite_dac3(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[3]=dac_buffer[3]*10+data-0x30;
		dac_flag[3]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[3]==1)
		{
			if(dac_buffer[3] > 1024) dac_buffer[3]=1024;
			printf("PWM3: %d\n",dac_buffer[3]);
		}
		dac_flag[3]=0;
		dac_buffer[3]=0;
	}
}


void init_dio()
{
}

void set_dio_out(unsigned char val)
{
	printf("DOUT: %X\n",val);
}


unsigned char get_dio_in(void)
{
	return 1;
}

#endif

//-------------------------------------------------------------------------------
// SPC56EL60
//-------------------------------------------------------------------------------
#if MCU == 10
void start_adc()
{
	int i;
	for(i=0;i<4;i++)
	{
		adc_buffer[i]=0;
		adc_flag[i]=0;
	}
	init_adc(15);	// 4 channels ADC
}

void stop_adc()
{
	disable_adc();
}

unsigned short read_adc0()
{
	return ADC_0.CDR[0].BIT.CDATA >> 2;
}

unsigned short read_adc1()
{
	return ADC_0.CDR[1].BIT.CDATA >> 2;
}

unsigned short read_adc2()
{
	return ADC_0.CDR[11].BIT.CDATA >> 2;
}

unsigned short read_adc3()
{
	return ADC_0.CDR[12].BIT.CDATA >> 2;
}


unsigned char aread_adc0()
{
	char c;
	
	switch(adc_flag[0])
	{
		case 0:		adc_buffer[0]=ADC_0.CDR[0].BIT.CDATA >> 2;
				c=adc_buffer[0] / 1000;
				adc_buffer[0]-=(c *1000);
				adc_flag[0]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[0] / 100;
				adc_buffer[0]-=(c *100);
				adc_flag[0]++;	
				return(c+48);

		case 2:		c=adc_buffer[0] / 10;
				adc_buffer[0]-=(c *10);
				adc_flag[0]++;	
				return(c+48);

		case 3:		c=adc_buffer[0];
				adc_flag[0]++;	
				return(c+48);
				
		default:	adc_flag[0]=0;	
				return(10);
				
	}					
	return 0;					
}


unsigned char aread_adc1()
{
	char c;
	
	switch(adc_flag[1])
	{
		case 0:		adc_buffer[1]=ADC_0.CDR[1].BIT.CDATA >> 2;
				c=adc_buffer[1] / 1000;
				adc_buffer[1]-=(c *1000);
				adc_flag[1]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[1] / 100;
				adc_buffer[1]-=(c *100);
				adc_flag[1]++;	
				return(c+48);

		case 2:		c=adc_buffer[1] / 10;
				adc_buffer[1]-=(c *10);
				adc_flag[1]++;	
				return(c+48);

		case 3:		c=adc_buffer[1];
				adc_flag[1]++;	
				return(c+48);
				
		default:	adc_flag[1]=0;	
				return(10);
				
	}					
	return 0;					
}


unsigned char aread_adc2()
{
	char c;
	
	switch(adc_flag[2])
	{
		case 0:		adc_buffer[2]=ADC_0.CDR[11].BIT.CDATA >> 2;
				c=adc_buffer[2] / 1000;
				adc_buffer[2]-=(c *1000);
				adc_flag[2]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[2] / 100;
				adc_buffer[2]-=(c *100);
				adc_flag[2]++;	
				return(c+48);

		case 2:		c=adc_buffer[2] / 10;
				adc_buffer[2]-=(c *10);
				adc_flag[2]++;	
				return(c+48);

		case 3:		c=adc_buffer[2];
				adc_flag[2]++;	
				return(c+48);
				
		default:	adc_flag[2]=0;	
				return(10);
				
	}					
	return 0;					
}

unsigned char aread_adc3()
{
	char c;
	
	switch(adc_flag[3])
	{
		case 0:		adc_buffer[3]=ADC_0.CDR[12].BIT.CDATA >> 2;
				c=adc_buffer[3] / 1000;
				adc_buffer[3]-=(c *1000);
				adc_flag[3]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[3] / 100;
				adc_buffer[3]-=(c *100);
				adc_flag[3]++;	
				return(c+48);

		case 2:		c=adc_buffer[3] / 10;
				adc_buffer[3]-=(c *10);
				adc_flag[3]++;	
				return(c+48);

		case 3:		c=adc_buffer[3];
				adc_flag[3]++;	
				return(c+48);
				
		default:	adc_flag[3]=0;	
				return(10);
				
	}					
	return 0;					
}


void start_dac()
{
	int i;
	for(i=0;i<4;i++)
	{
		dac_buffer[i]=0;
		dac_flag[i]=0;
	}
	enable_pwm0(1024,0);				// 4 channels DAC (PWM)
	enable_pwm1(1024,0);
	enable_pwm2(1024,0);
	enable_pwm3(1024,0);
}

void stop_dac()
{
	disable_pwm0();
	disable_pwm1();
	disable_pwm2();
	disable_pwm3();
}

void write_dac0(unsigned short data)
{
	set_pwm3(data & 0x07FF);
}

void write_dac1(unsigned short data)
{
	set_pwm2(data & 0x07FF);
}

void write_dac2(unsigned short data)
{
	set_pwm1(data & 0x07FF);
}

void write_dac3(unsigned short data)
{
	set_pwm0(data & 0x07FF);
}

void awrite_dac0(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[0]=dac_buffer[0]*10+data-0x30;
		dac_flag[0]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[0]==1) 
		{
			if(dac_buffer[0] > 1024) dac_buffer[0]=1024;
			set_pwm3(dac_buffer[0]);
		}
		dac_flag[0]=0;
		dac_buffer[0]=0;
	}
}

void awrite_dac1(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[1]=dac_buffer[1]*10+data-0x30;
		dac_flag[1]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[1]==1)
		{
			if(dac_buffer[1] > 1024) dac_buffer[1]=1024;
			set_pwm2(dac_buffer[1]);
		}
		dac_flag[1]=0;
		dac_buffer[1]=0;
	}
}


void awrite_dac2(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[2]=dac_buffer[2]*10+data-0x30;
		dac_flag[2]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[2]==1)
		{
			if(dac_buffer[2] > 1024) dac_buffer[2]=1024;
			set_pwm1(dac_buffer[2]);
		}
		dac_flag[2]=0;
		dac_buffer[2]=0;
	}
}

void awrite_dac3(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[3]=dac_buffer[3]*10+data-0x30;
		dac_flag[3]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[3]==1)
		{
			if(dac_buffer[3] > 1024) dac_buffer[3]=1024;
			set_pwm0(dac_buffer[3]);
		}
		dac_flag[3]=0;
		dac_buffer[3]=0;
	}
}



void init_dio()
{
	set_portpin_input_pullup(PORT_B,11);
	set_portpin_input_pullup(PORT_B,12);
	set_portpin_input_pullup(PORT_B,13);
	set_portpin_input_pullup(PORT_B,14);
	set_portpin_output(PORT_A,0);
	set_portpin_output(PORT_A,1);
	set_portpin_output(PORT_D,10);
	set_portpin_output(PORT_D,11);	
	set_portpin_zero(PORT_A,0);
	set_portpin_zero(PORT_A,1);
	set_portpin_zero(PORT_D,10);
	set_portpin_zero(PORT_D,11);	
}


void set_dio_out(unsigned char val)
{
	set_portpin_level(PORT_A,0,val & 1);
	set_portpin_level(PORT_A,1,val & 2);
	set_portpin_level(PORT_D,10,val & 4);
	set_portpin_level(PORT_D,11,val & 8);	
}


unsigned char get_dio_in(void)
{
	return(	get_portpin_level(PORT_B,11) |
		(get_portpin_level(PORT_B,12) << 1) |
		(get_portpin_level(PORT_B,13) << 2) |
		(get_portpin_level(PORT_B,14) << 3));
}

#endif

#if MCU == 11

unsigned long reg_save[8];

void start_adc()
{
	int i;
	for(i=0;i<4;i++)
	{
		adc_buffer[i]=0;
		adc_flag[i]=0;
	}
	init_adc(15);	// 4 channels ADC
//	KPRINTF("LL_INIT ADC\r\n");
}

void stop_adc()
{
	disable_adc();
//	KPRINTF("LL_EXIT ADC\r\n");
}


unsigned short read_adc0()
{
	return ADC_0.CDR[0].BIT.CDATA >> 2;
}

unsigned short read_adc1()
{
	return ADC_0.CDR[1].BIT.CDATA >> 2;
}

unsigned short read_adc2()
{
	return ADC_0.CDR[11].BIT.CDATA >> 2;
}

unsigned short read_adc3()
{
	return ADC_0.CDR[12].BIT.CDATA >> 2;
}


unsigned char aread_adc0()
{
	char c;
	
	switch(adc_flag[0])
	{
		case 0:		adc_buffer[0]=ADC_0.CDR[0].BIT.CDATA >> 2;
				c=adc_buffer[0] / 1000;
				adc_buffer[0]-=(c *1000);
				adc_flag[0]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[0] / 100;
				adc_buffer[0]-=(c *100);
				adc_flag[0]++;	
				return(c+48);

		case 2:		c=adc_buffer[0] / 10;
				adc_buffer[0]-=(c *10);
				adc_flag[0]++;	
				return(c+48);

		case 3:		c=adc_buffer[0];
				adc_flag[0]++;	
				return(c+48);
				
		default:	adc_flag[0]=0;	
				return(10);
				
	}					
	return 0;					
}


unsigned char aread_adc1()
{
	char c;
	
	switch(adc_flag[1])
	{
		case 0:		adc_buffer[1]=ADC_0.CDR[1].BIT.CDATA >> 2;
				c=adc_buffer[1] / 1000;
				adc_buffer[1]-=(c *1000);
				adc_flag[1]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[1] / 100;
				adc_buffer[1]-=(c *100);
				adc_flag[1]++;	
				return(c+48);

		case 2:		c=adc_buffer[1] / 10;
				adc_buffer[1]-=(c *10);
				adc_flag[1]++;	
				return(c+48);

		case 3:		c=adc_buffer[1];
				adc_flag[1]++;	
				return(c+48);
				
		default:	adc_flag[1]=0;	
				return(10);
				
	}					
	return 0;					
}


unsigned char aread_adc2()
{
	char c;
	
	switch(adc_flag[2])
	{
		case 0:		adc_buffer[2]=ADC_0.CDR[11].BIT.CDATA >> 2;
				c=adc_buffer[2] / 1000;
				adc_buffer[2]-=(c *1000);
				adc_flag[2]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[2] / 100;
				adc_buffer[2]-=(c *100);
				adc_flag[2]++;	
				return(c+48);

		case 2:		c=adc_buffer[2] / 10;
				adc_buffer[2]-=(c *10);
				adc_flag[2]++;	
				return(c+48);

		case 3:		c=adc_buffer[2];
				adc_flag[2]++;	
				return(c+48);
				
		default:	adc_flag[2]=0;	
				return(10);
				
	}					
	return 0;					
}

unsigned char aread_adc3()
{
	char c;
	
	switch(adc_flag[3])
	{
		case 0:		adc_buffer[3]=ADC_0.CDR[12].BIT.CDATA >> 2;
				c=adc_buffer[3] / 1000;
				adc_buffer[3]-=(c *1000);
				adc_flag[3]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[3] / 100;
				adc_buffer[3]-=(c *100);
				adc_flag[3]++;	
				return(c+48);

		case 2:		c=adc_buffer[3] / 10;
				adc_buffer[3]-=(c *10);
				adc_flag[3]++;	
				return(c+48);

		case 3:		c=adc_buffer[3];
				adc_flag[3]++;	
				return(c+48);
				
		default:	adc_flag[3]=0;	
				return(10);
				
	}					
	return 0;					
}

void start_dac()
{
	int i;
	for(i=0;i<4;i++)
	{
		dac_buffer[i]=0;
		dac_flag[i]=0;
	}
	enable_pwm0(1024,0);				// 4 channels DAC (PWM)
	enable_pwm1(1024,0);
	enable_pwm2(1024,0);
	enable_pwm3(1024,0);
}

void stop_dac()
{
	disable_pwm0();
	disable_pwm1();
	disable_pwm2();
	disable_pwm3();
}

void write_dac0(unsigned short data)
{
	set_pwm3(data & 0x07FF);
}

void write_dac1(unsigned short data)
{
	set_pwm2(data & 0x07FF);
}

void write_dac2(unsigned short data)
{
	set_pwm1(data & 0x07FF);
}

void write_dac3(unsigned short data)
{
	set_pwm0(data & 0x07FF);
}

void awrite_dac0(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[0]=dac_buffer[0]*10+data-0x30;
		dac_flag[0]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[0]==1) 
		{
			if(dac_buffer[0] > 1024) dac_buffer[0]=1024;
			set_pwm3(dac_buffer[0]);
		}
		dac_flag[0]=0;
		dac_buffer[0]=0;
	}
}

void awrite_dac1(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[1]=dac_buffer[1]*10+data-0x30;
		dac_flag[1]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[1]==1)
		{
			if(dac_buffer[1] > 1024) dac_buffer[1]=1024;
			set_pwm2(dac_buffer[1]);
		}
		dac_flag[1]=0;
		dac_buffer[1]=0;
	}
}


void awrite_dac2(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[2]=dac_buffer[2]*10+data-0x30;
		dac_flag[2]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[2]==1)
		{
			if(dac_buffer[2] > 1024) dac_buffer[2]=1024;
			set_pwm1(dac_buffer[2]);
		}
		dac_flag[2]=0;
		dac_buffer[2]=0;
	}
}

void awrite_dac3(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[3]=dac_buffer[3]*10+data-0x30;
		dac_flag[3]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[3]==1)
		{
			if(dac_buffer[3] > 1024) dac_buffer[3]=1024;
			set_pwm0(dac_buffer[3]);
		}
		dac_flag[3]=0;
		dac_buffer[3]=0;
	}
}

void init_dio()
{
	set_portpin_input_pullup(PORT_B,11);
	set_portpin_input_pullup(PORT_B,12);
	set_portpin_input_pullup(PORT_B,13);
	set_portpin_input_pullup(PORT_B,14);
	set_portpin_output(PORT_A,0);
	set_portpin_output(PORT_A,1);
	set_portpin_output(PORT_D,10);
	set_portpin_output(PORT_D,11);	
	set_portpin_zero(PORT_A,0);
	set_portpin_zero(PORT_A,1);
	set_portpin_zero(PORT_D,10);
	set_portpin_zero(PORT_D,11);	
}


void set_dio_out(unsigned char val)
{
	set_portpin_level(PORT_A,0,val & 1);
	set_portpin_level(PORT_A,1,val & 2);
	set_portpin_level(PORT_D,10,val & 4);
	set_portpin_level(PORT_D,11,val & 8);	
}

unsigned char get_dio_in(void)
{
	return(	get_portpin_level(PORT_B,11) |
		(get_portpin_level(PORT_B,12) << 1) |
		(get_portpin_level(PORT_B,13) << 2) |
		(get_portpin_level(PORT_B,14) << 3));
}

#endif

//-------------------------------------------------------------------------------
// STM32F103
//-------------------------------------------------------------------------------
#if MCU == 20
void start_adc()
{
	int i;
	for(i=0;i<4;i++)
	{
		adc_buffer[i]=0;
		adc_flag[i]=0;
	}
	init_adc_cont(0x0F);	// 4 channels ADC
}

void stop_adc()
{
	disable_adc();
}

unsigned short read_adc0()
{
	return get_adc(0) >> 6;
}

unsigned short read_adc1()
{
	return get_adc(1) >> 6;
}

unsigned short read_adc2()
{
	return get_adc(2) >> 6;
}

unsigned short read_adc3()
{
	return get_adc(3) >> 6;
}


unsigned char aread_adc0()
{
	char c;
	
	switch(adc_flag[0])
	{
		case 0:		adc_buffer[0]=get_adc(0) >> 6;
				c=adc_buffer[0] / 1000;
				adc_buffer[0]-=(c *1000);
				adc_flag[0]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[0] / 100;
				adc_buffer[0]-=(c *100);
				adc_flag[0]++;	
				return(c+48);

		case 2:		c=adc_buffer[0] / 10;
				adc_buffer[0]-=(c *10);
				adc_flag[0]++;	
				return(c+48);

		case 3:		c=adc_buffer[0];
				adc_flag[0]++;	
				return(c+48);
				
		default:	adc_flag[0]=0;	
				return(10);
				
	}					
	return 0;					
}


unsigned char aread_adc1()
{
	char c;
	
	switch(adc_flag[1])
	{
		case 0:		adc_buffer[1]=get_adc(1) >> 6;
				c=adc_buffer[1] / 1000;
				adc_buffer[1]-=(c *1000);
				adc_flag[1]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[1] / 100;
				adc_buffer[1]-=(c *100);
				adc_flag[1]++;	
				return(c+48);

		case 2:		c=adc_buffer[1] / 10;
				adc_buffer[1]-=(c *10);
				adc_flag[1]++;	
				return(c+48);

		case 3:		c=adc_buffer[1];
				adc_flag[1]++;	
				return(c+48);
				
		default:	adc_flag[1]=0;	
				return(10);
				
	}					
	return 0;					
}


unsigned char aread_adc2()
{
	char c;
	
	switch(adc_flag[2])
	{
		case 0:		adc_buffer[2]=get_adc(2) >> 6;
				c=adc_buffer[2] / 1000;
				adc_buffer[2]-=(c *1000);
				adc_flag[2]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[2] / 100;
				adc_buffer[2]-=(c *100);
				adc_flag[2]++;	
				return(c+48);

		case 2:		c=adc_buffer[2] / 10;
				adc_buffer[2]-=(c *10);
				adc_flag[2]++;	
				return(c+48);

		case 3:		c=adc_buffer[2];
				adc_flag[2]++;	
				return(c+48);
				
		default:	adc_flag[2]=0;	
				return(10);
				
	}					
	return 0;					
}

unsigned char aread_adc3()
{
	char c;
	
	switch(adc_flag[3])
	{
		case 0:		adc_buffer[3]=get_adc(3) >> 6;
				c=adc_buffer[3] / 1000;
				adc_buffer[3]-=(c *1000);
				adc_flag[3]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[3] / 100;
				adc_buffer[3]-=(c *100);
				adc_flag[3]++;	
				return(c+48);

		case 2:		c=adc_buffer[3] / 10;
				adc_buffer[3]-=(c *10);
				adc_flag[3]++;	
				return(c+48);

		case 3:		c=adc_buffer[3];
				adc_flag[3]++;	
				return(c+48);
				
		default:	adc_flag[3]=0;	
				return(10);
				
	}					
	return 0;					
}


void start_dac()
{
	int i;
	for(i=0;i<4;i++)
	{
		dac_buffer[i]=0;
		dac_flag[i]=0;
	}
	enable_pwm4(1024,0);				// 4 channels DAC (PWM)
	enable_pwm5(1024,0);
	enable_pwm6(1024,0);
	enable_pwm7(1024,0);
}

void stop_dac()
{
	disable_pwm4();
	disable_pwm5();
	disable_pwm6();
	disable_pwm7();
}

void write_dac0(unsigned short data)
{
	set_pwm4(data & 0x07FF);
}

void write_dac1(unsigned short data)
{
	set_pwm5(data & 0x07FF);
}

void write_dac2(unsigned short data)
{
	set_pwm6(data & 0x07FF);
}

void write_dac3(unsigned short data)
{
	set_pwm7(data & 0x07FF);
}

void awrite_dac0(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[0]=dac_buffer[0]*10+data-0x30;
		dac_flag[0]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[0]==1) 
		{
			if(dac_buffer[0] > 1024) dac_buffer[0]=1024;
			set_pwm4(dac_buffer[0]);
		}
		dac_flag[0]=0;
		dac_buffer[0]=0;
	}
}

void awrite_dac1(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[1]=dac_buffer[1]*10+data-0x30;
		dac_flag[1]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[1]==1)
		{
			if(dac_buffer[1] > 1024) dac_buffer[1]=1024;
			set_pwm5(dac_buffer[1]);
		}
		dac_flag[1]=0;
		dac_buffer[1]=0;
	}
}


void awrite_dac2(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[2]=dac_buffer[2]*10+data-0x30;
		dac_flag[2]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[2]==1)
		{
			if(dac_buffer[2] > 1024) dac_buffer[2]=1024;
			set_pwm6(dac_buffer[2]);
		}
		dac_flag[2]=0;
		dac_buffer[2]=0;
	}
}

void awrite_dac3(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[3]=dac_buffer[3]*10+data-0x30;
		dac_flag[3]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[3]==1)
		{
			if(dac_buffer[3] > 1024) dac_buffer[3]=1024;
			set_pwm7(dac_buffer[3]);
		}
		dac_flag[3]=0;
		dac_buffer[3]=0;
	}
}

void init_dio()
{
	set_portpin_input_pullup(PORT_C,0);
	set_portpin_input_pullup(PORT_C,1);
	set_portpin_input_pullup(PORT_C,2);
	set_portpin_input_pullup(PORT_C,3);
	set_port_level(PORT_B,0);
	set_portpin_output(PORT_B,12);
	set_portpin_output(PORT_B,13);
	set_portpin_output(PORT_B,14);
	set_portpin_output(PORT_B,15);	
}

void set_dio_out(unsigned char val)
{
	GPIOB.ODR.REG=val << 12;
}


unsigned char get_dio_in(void)
{
	unsigned short q;	
	q=GPIOC.IDR.REG;
	return(q & 0x0F);
}

#endif

//-------------------------------------------------------------------------------
// STM32F107
//-------------------------------------------------------------------------------
#if MCU == 25
void start_adc()
{
	int i;
	for(i=0;i<4;i++)
	{
		adc_buffer[i]=0;
		adc_flag[i]=0;
	}
	init_adc_cont(0xF0);	// 4 channels ADC
}

void stop_adc()
{
	disable_adc();
}

unsigned short read_adc0()
{
	return get_adc(4) >> 6;
}

unsigned short read_adc1()
{
	return get_adc(5) >> 6;
}

unsigned short read_adc2()
{
	return get_adc(6) >> 6;
}

unsigned short read_adc3()
{
	return get_adc(7) >> 6;
}


unsigned char aread_adc0()
{
	char c;
	
	switch(adc_flag[0])
	{
		case 0:		adc_buffer[0]=get_adc(4) >> 6;
				c=adc_buffer[0] / 1000;
				adc_buffer[0]-=(c *1000);
				adc_flag[0]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[0] / 100;
				adc_buffer[0]-=(c *100);
				adc_flag[0]++;	
				return(c+48);

		case 2:		c=adc_buffer[0] / 10;
				adc_buffer[0]-=(c *10);
				adc_flag[0]++;	
				return(c+48);

		case 3:		c=adc_buffer[0];
				adc_flag[0]++;	
				return(c+48);
				
		default:	adc_flag[0]=0;	
				return(10);
				
	}					
	return 0;					
}


unsigned char aread_adc1()
{
	char c;
	
	switch(adc_flag[1])
	{
		case 0:		adc_buffer[1]=get_adc(5) >> 6;
				c=adc_buffer[1] / 1000;
				adc_buffer[1]-=(c *1000);
				adc_flag[1]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[1] / 100;
				adc_buffer[1]-=(c *100);
				adc_flag[1]++;	
				return(c+48);

		case 2:		c=adc_buffer[1] / 10;
				adc_buffer[1]-=(c *10);
				adc_flag[1]++;	
				return(c+48);

		case 3:		c=adc_buffer[1];
				adc_flag[1]++;	
				return(c+48);
				
		default:	adc_flag[1]=0;	
				return(10);
				
	}					
	return 0;					
}


unsigned char aread_adc2()
{
	char c;
	
	switch(adc_flag[2])
	{
		case 0:		adc_buffer[2]=get_adc(6) >> 6;
				c=adc_buffer[2] / 1000;
				adc_buffer[2]-=(c *1000);
				adc_flag[2]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[2] / 100;
				adc_buffer[2]-=(c *100);
				adc_flag[2]++;	
				return(c+48);

		case 2:		c=adc_buffer[2] / 10;
				adc_buffer[2]-=(c *10);
				adc_flag[2]++;	
				return(c+48);

		case 3:		c=adc_buffer[2];
				adc_flag[2]++;	
				return(c+48);
				
		default:	adc_flag[2]=0;	
				return(10);
				
	}					
	return 0;					
}

unsigned char aread_adc3()
{
	char c;
	
	switch(adc_flag[3])
	{
		case 0:		adc_buffer[3]=get_adc(7) >> 6;
				c=adc_buffer[3] / 1000;
				adc_buffer[3]-=(c *1000);
				adc_flag[3]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[3] / 100;
				adc_buffer[3]-=(c *100);
				adc_flag[3]++;	
				return(c+48);

		case 2:		c=adc_buffer[3] / 10;
				adc_buffer[3]-=(c *10);
				adc_flag[3]++;	
				return(c+48);

		case 3:		c=adc_buffer[3];
				adc_flag[3]++;	
				return(c+48);
				
		default:	adc_flag[3]=0;	
				return(10);
				
	}					
	return 0;					
}


void start_dac()
{
	int i;
	for(i=0;i<4;i++)
	{
		dac_buffer[i]=0;
		dac_flag[i]=0;
	}
	enable_pwm0(1024,0);				// 4 channels DAC (PWM)
	enable_pwm1(1024,0);
	enable_pwm6(1024,0);
	enable_pwm7(1024,0);
}

void stop_dac()
{
	disable_pwm0();
	disable_pwm1();
	disable_pwm6();
	disable_pwm7();
}

void write_dac0(unsigned short data)
{
	set_pwm0(data & 0x07FF);
}

void write_dac1(unsigned short data)
{
	set_pwm1(data & 0x07FF);
}

void write_dac2(unsigned short data)
{
	set_pwm6(data & 0x07FF);
}

void write_dac3(unsigned short data)
{
	set_pwm7(data & 0x07FF);
}


void awrite_dac0(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[0]=dac_buffer[0]*10+data-0x30;
		dac_flag[0]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[0]==1) 
		{
			if(dac_buffer[0] > 1024) dac_buffer[0]=1024;
			set_pwm0(dac_buffer[0]);
		}
		dac_flag[0]=0;
		dac_buffer[0]=0;
	}
}

void awrite_dac1(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[1]=dac_buffer[1]*10+data-0x30;
		dac_flag[1]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[1]==1)
		{
			if(dac_buffer[1] > 1024) dac_buffer[1]=1024;
			set_pwm1(dac_buffer[1]);
		}
		dac_flag[1]=0;
		dac_buffer[1]=0;
	}
}


void awrite_dac2(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[2]=dac_buffer[2]*10+data-0x30;
		dac_flag[2]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[2]==1)
		{
			if(dac_buffer[2] > 1024) dac_buffer[2]=1024;
			set_pwm6(dac_buffer[2]);
		}
		dac_flag[2]=0;
		dac_buffer[2]=0;
	}
}

void awrite_dac3(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[3]=dac_buffer[3]*10+data-0x30;
		dac_flag[3]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[3]==1)
		{
			if(dac_buffer[3] > 1024) dac_buffer[3]=1024;
			set_pwm7(dac_buffer[3]);
		}
		dac_flag[3]=0;
		dac_buffer[3]=0;
	}
}



void init_dio()
{
	set_portpin_input_pullup(PORT_B,8);
	set_portpin_input_pullup(PORT_B,9);
	set_portpin_input_pullup(PORT_B,10);
	set_portpin_input_pullup(PORT_B,11);
	set_portpin_output(PORT_B,12);
	set_portpin_output(PORT_B,13);
	set_portpin_output(PORT_B,14);
	set_portpin_output(PORT_B,15);
	set_portpin_zero(PORT_B,12);
	set_portpin_zero(PORT_B,13);
	set_portpin_zero(PORT_B,14);
	set_portpin_zero(PORT_B,15);
}

void set_dio_out(unsigned char val)
{
	GPIOB.ODR.REG=val << 12;
}

unsigned char get_dio_in(void)
{
	unsigned short q;	
	q=GPIOB.IDR.REG >> 8;
	return(q & 0x0F);
}

#endif

//-------------------------------------------------------------------------------
// STM32F411
//-------------------------------------------------------------------------------
#if MCU == 30
void start_adc()
{
	int i;
	for(i=0;i<4;i++)
	{
		adc_buffer[i]=0;
		adc_flag[i]=0;
	}
	init_adc_cont(15);	// 4 channels ADC
}

void stop_adc()
{
	disable_adc();
}

unsigned short read_adc0()
{
	return get_adc(0) >> 3;
}

unsigned short read_adc1()
{
	return get_adc(1) >> 3;
}

unsigned short read_adc2()
{
	return get_adc(2) >> 3;
}

unsigned short read_adc3()
{
	return get_adc(3) >> 3;
}


unsigned char aread_adc0()
{
	char c;
	
	switch(adc_flag[0])
	{
		case 0:		adc_buffer[0]=get_adc(0) >> 6;
				c=adc_buffer[0] / 1000;
				adc_buffer[0]-=(c *1000);
				adc_flag[0]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[0] / 100;
				adc_buffer[0]-=(c *100);
				adc_flag[0]++;	
				return(c+48);

		case 2:		c=adc_buffer[0] / 10;
				adc_buffer[0]-=(c *10);
				adc_flag[0]++;	
				return(c+48);

		case 3:		c=adc_buffer[0];
				adc_flag[0]++;	
				return(c+48);
				
		default:	adc_flag[0]=0;	
				return(10);
				
	}					
	return 0;					
}


unsigned char aread_adc1()
{
	char c;
	
	switch(adc_flag[1])
	{
		case 0:		adc_buffer[1]=get_adc(1) >> 6;
				c=adc_buffer[1] / 1000;
				adc_buffer[1]-=(c *1000);
				adc_flag[1]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[1] / 100;
				adc_buffer[1]-=(c *100);
				adc_flag[1]++;	
				return(c+48);

		case 2:		c=adc_buffer[1] / 10;
				adc_buffer[1]-=(c *10);
				adc_flag[1]++;	
				return(c+48);

		case 3:		c=adc_buffer[1];
				adc_flag[1]++;	
				return(c+48);
				
		default:	adc_flag[1]=0;	
				return(10);
				
	}					
	return 0;					
}


unsigned char aread_adc2()
{
	char c;
	
	switch(adc_flag[2])
	{
		case 0:		adc_buffer[2]=get_adc(2) >> 6;
				c=adc_buffer[2] / 1000;
				adc_buffer[2]-=(c *1000);
				adc_flag[2]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[2] / 100;
				adc_buffer[2]-=(c *100);
				adc_flag[2]++;	
				return(c+48);

		case 2:		c=adc_buffer[2] / 10;
				adc_buffer[2]-=(c *10);
				adc_flag[2]++;	
				return(c+48);

		case 3:		c=adc_buffer[2];
				adc_flag[2]++;	
				return(c+48);
				
		default:	adc_flag[2]=0;	
				return(10);
				
	}					
	return 0;					
}

unsigned char aread_adc3()
{
	char c;
	
	switch(adc_flag[3])
	{
		case 0:		adc_buffer[3]=get_adc(3) >> 6;
				c=adc_buffer[3] / 1000;
				adc_buffer[3]-=(c *1000);
				adc_flag[3]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[3] / 100;
				adc_buffer[3]-=(c *100);
				adc_flag[3]++;	
				return(c+48);

		case 2:		c=adc_buffer[3] / 10;
				adc_buffer[3]-=(c *10);
				adc_flag[3]++;	
				return(c+48);

		case 3:		c=adc_buffer[3];
				adc_flag[3]++;	
				return(c+48);
				
		default:	adc_flag[3]=0;	
				return(10);
				
	}					
	return 0;					
}

void start_dac()
{
	int i;
	for(i=0;i<4;i++)
	{
		dac_buffer[i]=0;
		dac_flag[i]=0;
	}
	enable_pwm4(1024,0);				// 4 channels DAC (PWM)
	enable_pwm5(1024,0);
	enable_pwm6(1024,0);
	enable_pwm7(1024,0);
}

void stop_dac()
{
	disable_pwm4();
	disable_pwm5();
	disable_pwm6();
	disable_pwm7();
}

void write_dac0(unsigned short data)
{
	set_pwm4(data & 0x07FF);
}

void write_dac1(unsigned short data)
{
	set_pwm5(data & 0x07FF);
}

void write_dac2(unsigned short data)
{
	set_pwm6(data & 0x07FF);
}

void write_dac3(unsigned short data)
{
	set_pwm7(data & 0x07FF);
}


void awrite_dac0(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[0]=dac_buffer[0]*10+data-0x30;
		dac_flag[0]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[0]==1) 
		{
			if(dac_buffer[0] > 1024) dac_buffer[0]=1024;
			set_pwm4(dac_buffer[0]);
		}
		dac_flag[0]=0;
		dac_buffer[0]=0;
	}
}

void awrite_dac1(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[1]=dac_buffer[1]*10+data-0x30;
		dac_flag[1]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[1]==1)
		{
			if(dac_buffer[1] > 1024) dac_buffer[1]=1024;
			set_pwm5(dac_buffer[1]);
		}
		dac_flag[1]=0;
		dac_buffer[1]=0;
	}
}


void awrite_dac2(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[2]=dac_buffer[2]*10+data-0x30;
		dac_flag[2]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[2]==1)
		{
			if(dac_buffer[2] > 1024) dac_buffer[2]=1024;
			set_pwm6(dac_buffer[2]);
		}
		dac_flag[2]=0;
		dac_buffer[2]=0;
	}
}

void awrite_dac3(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[3]=dac_buffer[3]*10+data-0x30;
		dac_flag[3]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[3]==1)
		{
			if(dac_buffer[3] > 1024) dac_buffer[3]=1024;
			set_pwm7(dac_buffer[3]);
		}
		dac_flag[3]=0;
		dac_buffer[3]=0;
	}
}



void init_dio()
{
	set_portpin_input_pullup(PORT_D,0);
	set_portpin_input_pullup(PORT_D,1);
	set_portpin_input_pullup(PORT_D,2);
	set_portpin_input_pullup(PORT_D,3);
	set_port_level(PORT_D,0);
	set_portpin_output(PORT_D,4);
	set_portpin_output(PORT_D,5);
	set_portpin_output(PORT_D,6);
	set_portpin_output(PORT_D,7);	
}

void set_dio_out(unsigned char val)
{
	unsigned short x=val;
	x <<= 4;
	set_port_level(PORT_D,x);
}


unsigned char get_dio_in(void)
{
	unsigned short x=get_port_level(PORT_D);
	return(x & 0x0f);
}


#endif


//-------------------------------------------------------------------------------
// STM32F405
//-------------------------------------------------------------------------------
#if MCU == 35
void start_adc()
{
	int i;
	for(i=0;i<4;i++)
	{
		adc_buffer[i]=0;
		adc_flag[i]=0;
	}
	init_adc_cont(0xF0);	// 4 channels ADC
}

void stop_adc()
{
	disable_adc();
}

unsigned short read_adc0()
{
	return get_adc(4) >> 6;
}

unsigned short read_adc1()
{
	return get_adc(5) >> 6;
}

unsigned short read_adc2()
{
	return get_adc(6) >> 6;
}

unsigned short read_adc3()
{
	return get_adc(7) >> 6;
}


unsigned char aread_adc0()
{
	char c;
	
	switch(adc_flag[0])
	{
		case 0:		adc_buffer[0]=get_adc(4) >> 6;
				c=adc_buffer[0] / 1000;
				adc_buffer[0]-=(c *1000);
				adc_flag[0]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[0] / 100;
				adc_buffer[0]-=(c *100);
				adc_flag[0]++;	
				return(c+48);

		case 2:		c=adc_buffer[0] / 10;
				adc_buffer[0]-=(c *10);
				adc_flag[0]++;	
				return(c+48);

		case 3:		c=adc_buffer[0];
				adc_flag[0]++;	
				return(c+48);
				
		default:	adc_flag[0]=0;	
				return(10);
				
	}					
	return 0;					
}


unsigned char aread_adc1()
{
	char c;
	
	switch(adc_flag[1])
	{
		case 0:		adc_buffer[1]=get_adc(5) >> 6;
				c=adc_buffer[1] / 1000;
				adc_buffer[1]-=(c *1000);
				adc_flag[1]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[1] / 100;
				adc_buffer[1]-=(c *100);
				adc_flag[1]++;	
				return(c+48);

		case 2:		c=adc_buffer[1] / 10;
				adc_buffer[1]-=(c *10);
				adc_flag[1]++;	
				return(c+48);

		case 3:		c=adc_buffer[1];
				adc_flag[1]++;	
				return(c+48);
				
		default:	adc_flag[1]=0;	
				return(10);
				
	}					
	return 0;					
}


unsigned char aread_adc2()
{
	char c;
	
	switch(adc_flag[2])
	{
		case 0:		adc_buffer[2]=get_adc(6) >> 6;
				c=adc_buffer[2] / 1000;
				adc_buffer[2]-=(c *1000);
				adc_flag[2]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[2] / 100;
				adc_buffer[2]-=(c *100);
				adc_flag[2]++;	
				return(c+48);

		case 2:		c=adc_buffer[2] / 10;
				adc_buffer[2]-=(c *10);
				adc_flag[2]++;	
				return(c+48);

		case 3:		c=adc_buffer[2];
				adc_flag[2]++;	
				return(c+48);
				
		default:	adc_flag[2]=0;	
				return(10);
				
	}					
	return 0;					
}

unsigned char aread_adc3()
{
	char c;
	
	switch(adc_flag[3])
	{
		case 0:		adc_buffer[3]=get_adc(7) >> 6;
				c=adc_buffer[3] / 1000;
				adc_buffer[3]-=(c *1000);
				adc_flag[3]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[3] / 100;
				adc_buffer[3]-=(c *100);
				adc_flag[3]++;	
				return(c+48);

		case 2:		c=adc_buffer[3] / 10;
				adc_buffer[3]-=(c *10);
				adc_flag[3]++;	
				return(c+48);

		case 3:		c=adc_buffer[3];
				adc_flag[3]++;	
				return(c+48);
				
		default:	adc_flag[3]=0;	
				return(10);
				
	}					
	return 0;					
}

void start_dac()
{
	int i;
	for(i=0;i<4;i++)
	{
		dac_buffer[i]=0;
		dac_flag[i]=0;
	}
	enable_pwm0(1024,0);				// 4 channels DAC (PWM)
	enable_pwm1(1024,0);
	enable_pwm2(1024,0);
	enable_pwm3(1024,0);
}

void stop_dac()
{
	disable_pwm0();
	disable_pwm1();
	disable_pwm2();
	disable_pwm3();
}

void write_dac0(unsigned short data)
{
	set_pwm0(data & 0x07FF);
}

void write_dac1(unsigned short data)
{
	set_pwm1(data & 0x07FF);
}

void write_dac2(unsigned short data)
{
	set_pwm2(data & 0x07FF);
}

void write_dac3(unsigned short data)
{
	set_pwm3(data & 0x07FF);
}


void awrite_dac0(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[0]=dac_buffer[0]*10+data-0x30;
		dac_flag[0]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[0]==1) 
		{
			if(dac_buffer[0] > 1024) dac_buffer[0]=1024;
			set_pwm4(dac_buffer[0]);
		}
		dac_flag[0]=0;
		dac_buffer[0]=0;
	}
}

void awrite_dac1(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[1]=dac_buffer[1]*10+data-0x30;
		dac_flag[1]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[1]==1)
		{
			if(dac_buffer[1] > 1024) dac_buffer[1]=1024;
			set_pwm5(dac_buffer[1]);
		}
		dac_flag[1]=0;
		dac_buffer[1]=0;
	}
}


void awrite_dac2(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[2]=dac_buffer[2]*10+data-0x30;
		dac_flag[2]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[2]==1)
		{
			if(dac_buffer[2] > 1024) dac_buffer[2]=1024;
			set_pwm6(dac_buffer[2]);
		}
		dac_flag[2]=0;
		dac_buffer[2]=0;
	}
}

void awrite_dac3(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[3]=dac_buffer[3]*10+data-0x30;
		dac_flag[3]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[3]==1)
		{
			if(dac_buffer[3] > 1024) dac_buffer[3]=1024;
			set_pwm7(dac_buffer[3]);
		}
		dac_flag[3]=0;
		dac_buffer[3]=0;
	}
}



void init_dio()
{
	set_portpin_input_pullup(PORT_B,4);
	set_portpin_input_pullup(PORT_B,5);
	set_portpin_input_pullup(PORT_B,6);
	set_portpin_input_pullup(PORT_B,7);
	set_port_level(PORT_B,0);
	set_portpin_output(PORT_B,12);
	set_portpin_output(PORT_B,13);
	set_portpin_output(PORT_B,14);
	set_portpin_output(PORT_B,15);	
}

void set_dio_out(unsigned char val)
{
	unsigned short x=val;
	x <<= 12;
	set_port_level(PORT_B,x);
}


unsigned char get_dio_in(void)
{
	unsigned short x=get_port_level(PORT_B);
	return((x >> 4) & 0x0f);
}

#endif


//-------------------------------------------------------------------------------
// STM32L475
//-------------------------------------------------------------------------------
#if MCU == 40
void start_adc()
{
	int i;
	for(i=0;i<4;i++)
	{
		adc_buffer[i]=0;
		adc_flag[i]=0;
	}
//	init_adc(0xF0);	// 4 channels ADC
}

void stop_adc()
{
//	disable_adc();
}

unsigned short read_adc0()
{
//	return get_adc(4) >> 6;
	return 0;
}

unsigned short read_adc1()
{
//	return get_adc(5) >> 6;
	return 0;
}

unsigned short read_adc2()
{
//	return get_adc(6) >> 6;
	return 0;
}

unsigned short read_adc3()
{
//	return get_adc(7) >> 6;
	return 0;
}

unsigned char aread_adc0()
{
	char c;
	
	switch(adc_flag[0])
	{
		case 0:		adc_buffer[0]=1111;
				c=adc_buffer[0] / 1000;
				adc_buffer[0]-=(c *1000);
				adc_flag[0]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[0] / 100;
				adc_buffer[0]-=(c *100);
				adc_flag[0]++;	
				return(c+48);

		case 2:		c=adc_buffer[0] / 10;
				adc_buffer[0]-=(c *10);
				adc_flag[0]++;	
				return(c+48);

		case 3:		c=adc_buffer[0];
				adc_flag[0]++;	
				return(c+48);
				
		default:	adc_flag[0]=0;	
				return(10);
				
	}					
	return 0;					
}


unsigned char aread_adc1()
{
	char c;
	
	switch(adc_flag[1])
	{
		case 0:		adc_buffer[1]=2222;
				c=adc_buffer[1] / 1000;
				adc_buffer[1]-=(c *1000);
				adc_flag[1]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[1] / 100;
				adc_buffer[1]-=(c *100);
				adc_flag[1]++;	
				return(c+48);

		case 2:		c=adc_buffer[1] / 10;
				adc_buffer[1]-=(c *10);
				adc_flag[1]++;	
				return(c+48);

		case 3:		c=adc_buffer[1];
				adc_flag[1]++;	
				return(c+48);
				
		default:	adc_flag[1]=0;	
				return(10);
				
	}					
	return 0;					
}


unsigned char aread_adc2()
{
	char c;
	
	switch(adc_flag[2])
	{
		case 0:		adc_buffer[2]=3333;
				c=adc_buffer[2] / 1000;
				adc_buffer[2]-=(c *1000);
				adc_flag[2]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[2] / 100;
				adc_buffer[2]-=(c *100);
				adc_flag[2]++;	
				return(c+48);

		case 2:		c=adc_buffer[2] / 10;
				adc_buffer[2]-=(c *10);
				adc_flag[2]++;	
				return(c+48);

		case 3:		c=adc_buffer[2];
				adc_flag[2]++;	
				return(c+48);
				
		default:	adc_flag[2]=0;	
				return(10);
				
	}					
	return 0;					
}

unsigned char aread_adc3()
{
	char c;
	
	switch(adc_flag[3])
	{
		case 0:		adc_buffer[3]=444;
				c=adc_buffer[3] / 1000;
				adc_buffer[3]-=(c *1000);
				adc_flag[3]++;	
				return(c+48);
				
		case 1:		c=adc_buffer[3] / 100;
				adc_buffer[3]-=(c *100);
				adc_flag[3]++;	
				return(c+48);

		case 2:		c=adc_buffer[3] / 10;
				adc_buffer[3]-=(c *10);
				adc_flag[3]++;	
				return(c+48);

		case 3:		c=adc_buffer[3];
				adc_flag[3]++;	
				return(c+48);
				
		default:	adc_flag[3]=0;	
				return(10);
				
	}					
	return 0;					
}

void start_dac()
{
	int i;
	for(i=0;i<4;i++)
	{
		dac_buffer[i]=0;
		dac_flag[i]=0;
	}
//	enable_pwm0(1024,0);				// 4 channels DAC (PWM)
//	enable_pwm1(1024,0);
//	enable_pwm2(1024,0);
//	enable_pwm3(1024,0);
}

void stop_dac()
{
//	disable_pwm0();
//	disable_pwm1();
//	disable_pwm2();
//	disable_pwm3();
}

void write_dac0(unsigned short data)
{
//	set_pwm0(data & 0x07FF);
}

void write_dac1(unsigned short data)
{
//	set_pwm1(data & 0x07FF);
}

void write_dac2(unsigned short data)
{
//	set_pwm2(data & 0x07FF);
}

void write_dac3(unsigned short data)
{
//	set_pwm3(data & 0x07FF);
}


void awrite_dac0(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[0]=dac_buffer[0]*10+data-0x30;
		dac_flag[0]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[0]==1) 
		{
			if(dac_buffer[0] > 1024) dac_buffer[0]=1024;
//			set_pwm4(dac_buffer[0]);
		}
		dac_flag[0]=0;
		dac_buffer[0]=0;
	}
}

void awrite_dac1(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[1]=dac_buffer[1]*10+data-0x30;
		dac_flag[1]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[1]==1)
		{
			if(dac_buffer[1] > 1024) dac_buffer[1]=1024;
//			set_pwm5(dac_buffer[1]);
		}
		dac_flag[1]=0;
		dac_buffer[1]=0;
	}
}


void awrite_dac2(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[2]=dac_buffer[2]*10+data-0x30;
		dac_flag[2]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[2]==1)
		{
			if(dac_buffer[2] > 1024) dac_buffer[2]=1024;
//			set_pwm6(dac_buffer[2]);
		}
		dac_flag[2]=0;
		dac_buffer[2]=0;
	}
}

void awrite_dac3(unsigned char data)
{
	if((data >= 0x30) && (data <=0x39))
	{
		dac_buffer[3]=dac_buffer[3]*10+data-0x30;
		dac_flag[3]=1;
	}
	if(data < 0x21)
	{
		if(dac_flag[3]==1)
		{
			if(dac_buffer[3] > 1024) dac_buffer[3]=1024;
//			set_pwm7(dac_buffer[3]);
		}
		dac_flag[3]=0;
		dac_buffer[3]=0;
	}
}



void init_dio()
{
	set_portpin_input_pullup(PORT_B,4);
	set_portpin_input_pullup(PORT_B,5);
	set_portpin_input_pullup(PORT_B,6);
	set_portpin_input_pullup(PORT_B,7);
	set_port_level(PORT_B,0);
	set_portpin_output(PORT_B,12);
	set_portpin_output(PORT_B,13);
	set_portpin_output(PORT_B,14);
	set_portpin_output(PORT_B,15);	
}

void set_dio_out(unsigned char val)
{
	unsigned short x=val;
	x <<= 12;
	set_port_level(PORT_B,x);
}


unsigned char get_dio_in(void)
{
	unsigned short x=get_port_level(PORT_B);
	return((x >> 4) & 0x0f);
}

#endif
