//###############################################################################
//#										#
//# MXE11 PDP11-Unix Emulator for Mikrocontrollers				#
//#										#
//# copyright (c) 2017 Joerg Wolfram (joerg@jcwolfram.de)			#
//#										#
//#										#
//# This program is free software; you can redistribute it and/or		#
//# modify it under the terms of the GNU General Public License			#
//# as published by the Free Software Foundation; either version 3		#
//# of the License, or (at your option) any later version.			#
//#										#
//# This program is distributed in the hope that it will be useful,		#
//# but WITHOUT ANY WARRANTY; without even the implied warranty of		#
//# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the GNU		#
//# General Public License for more details.					#
//#										#
//# You should have received a copy of the GNU General Public			#
//# License along with this library// if not, write to the			#
//# Free Software Foundation, Inc., 59 Temple Place - Suite 330,		#
//# Boston, MA 02111-1307, USA.							#
//#										#
//###############################################################################
#include "board.h"
#include "cpu.h"
#include "vars.h"


#if MCU < 10
#ifdef NOX
	#include "windows.h"
	DCB new_psettings;
	COMMTIMEOUTS wtout;
	HANDLE port_handle;
#else
	#include <termios.h>
	struct termios old_psettings,new_psettings;
	int port_handle;
#endif	//NOX
	extern SDL_Surface* screen;
	extern void set_outchar_routine(void (*outfunc)(unsigned char));
	extern void init_lfsr_default(void);
#else
	extern long get_date();
	extern void set_date(unsigned long);
	extern void init_dio(void);
#endif	//MCU < 10

#if MCU == 11
	extern void masterloop(void);
#endif

extern void cpu_init();
extern void cpu_step();
extern void memory_init();
extern void console_init(char);
extern void serve_console();
extern void terminal_init();
extern void console_exit();
extern void rk11_init();
extern void rk11_flush();
extern void kl11_init();
extern void rk11_bootsetup();
extern void serve_kl11();
extern void console_out(unsigned char);
extern void start_adc(void);
extern void start_dac(void);

extern unsigned short system_state;
extern unsigned short do_wait;

extern volatile unsigned short rx_rptr8,rx_wptr8,rx_cap8;
extern volatile unsigned short tx_rptr8,tx_wptr8,tx_cap8;
extern volatile unsigned short rx_rptr9,rx_wptr9,rx_cap9;
extern volatile unsigned short tx_rptr9,tx_wptr9,tx_cap9;

extern cycle_t do_cycle[32];

int c;



void console_out(unsigned char c)
{
	while(tx_cap8 < 1);
#if MCU > 9
	disable_interrupts();
#endif
	txbuffer8[tx_wptr8++]= c;
	tx_wptr8 &= 0xFF;
	tx_cap8--;
#if MCU > 9
	enable_interrupts();
#endif
}


unsigned char console_in(void)
{
	unsigned char c;
	
	if (rx_cap8 == 0) return 0;
#if MCU > 9
	disable_interrupts();
#endif
	c=rxbuffer8[rx_rptr8++];
	rx_rptr8 &= 0xFF;
	rx_cap8--;
#if MCU > 9
	enable_interrupts();
#endif
	return c;
}

#if MCU > 9
void console2_out(unsigned char c)
{
	while(tx_cap9 < 1);
	disable_interrupts();
	txbuffer9[tx_wptr9++]= c;
	tx_wptr9 &= 0xFF;
	tx_cap9--;
	enable_interrupts();
}

unsigned char console2_in(void)
{
	unsigned char c;
	
	if (rx_cap9 == 0) return 0;
	disable_interrupts();
	c=rxbuffer9[rx_rptr9++];
	rx_rptr9 &= 0xFF;
	rx_cap9--;
	enable_interrupts();
	return c;
}
#endif



//######################################################################################

#if MCU < 10
int main(int argc, char *argv[])
{
	int prev=0;
	char portname[100];
	int perr,n;	
	char tch;
	
	portname[99]=0;	//stopper for safety

	if( SDL_Init( SDL_INIT_EVERYTHING ) < 0 )
	{
		printf( "SDL could not initialize! SDL_Error: %s\n", SDL_GetError() );
	}
	screen = SDL_SetVideoMode(SCREEN_WIDTH, SCREEN_HEIGHT, 16, SDL_SWSURFACE);
	SDL_WM_SetCaption("MXE11-SDL", "MXE11-SDL");

	printf("WINX=%d\n",SCREEN_WIDTH);
	printf("WINY=%d\n",SCREEN_HEIGHT);
	printf("ARGC=%d\n",argc);

	if(argc == 2)
	{
#ifndef NOX
	// unix version
		strcpy(portname, "");
		strncat(portname,argv[1],32);
	
		printf("PORT=%s\n",portname);

		set_outchar_routine(console_out);
		init_lfsr_default();				//for /dev/random
		memory_init();	
		cpu_init();
		console_init(0);
		KPRINTF("CONNECT TO %s ... ",portname);
	
		port_handle=open(portname, O_RDWR | O_NOCTTY | O_NDELAY);
		if(port_handle < 0) KPRINTF("FAILED!\r\n");
		perr=tcgetattr(port_handle,&old_psettings);
		memset(&new_psettings,0,sizeof(new_psettings));
		new_psettings.c_cflag = CS8 | CLOCAL | CREAD;
		new_psettings.c_iflag = IGNPAR;
		new_psettings.c_oflag = 0;
		new_psettings.c_lflag = 0;

		cfsetispeed(&new_psettings,B38400);
		cfsetospeed(&new_psettings,B38400);

		perr=tcsetattr(port_handle,TCSANOW, &new_psettings);

		if(perr < 0) KPRINTF("CONFIG FAILED!\r\n");
		else KPRINTF("CONFIGURED\r\n");

		do
		{
				n=read(port_handle,portname,1);
				if(n==1) console_out(portname[0]);
				serve_console();
				tch=console_in();
				if(tch!=0) write(port_handle,&tch,1);
				usleep(80);
		}
		while(system_state != 0);
		console_exit();
		tcsetattr(port_handle,TCSANOW,&old_psettings);
		if(port_handle >= 0) close(port_handle);					
		return 0;
		//windows variant
#else
		strcpy(portname, "\\\\.\\");		
		strncat(portname,argv[1],32);
	
//		printf("PORT=%s\n",portname);

		set_outchar_routine(console_out);
		init_lfsr_default();				//for /dev/random
		memory_init();	
		cpu_init();
		console_init(0);
		KPRINTF("CONNECT TO %s ... ",argv[1]);
	
		perr=0;
		port_handle = CreateFileA(portname,GENERIC_READ|GENERIC_WRITE,0,NULL,OPEN_EXISTING,0,NULL);
		if(port_handle < 0) KPRINTF("FAILED!\r\n");
		memset(&new_psettings,0,sizeof(new_psettings));
		new_psettings.DCBlength=sizeof(new_psettings);
		if(!BuildCommDCBA("baud=38400 data=8 parity=n stop=1",&new_psettings))
		{
			CloseHandle(port_handle);
			perr=-1;
		}

		if(!SetCommState(port_handle,&new_psettings))
		{
			CloseHandle(port_handle);
			perr=-1;
		}
		
		wtout.ReadIntervalTimeout		= MAXDWORD;
		wtout.ReadTotalTimeoutMultiplier	= 0;
		wtout.ReadTotalTimeoutConstant		= 0;
		wtout.WriteTotalTimeoutMultiplier	= 0;
		wtout.WriteTotalTimeoutConstant		= 0;
		
		if(!SetCommTimeouts(port_handle,&wtout))
		{
			CloseHandle(port_handle);
			perr=-1;
		}		

		if(perr < 0) KPRINTF("CONFIG FAILED!\r\n");
		else KPRINTF("CONFIGURED\r\n");


		do
		{
				ReadFile(port_handle,portname,1,(LPDWORD)((void *)&n), NULL);
				if(n==1) console_out(portname[0]);
				serve_console();
				tch=console_in();
				if(tch!=0) WriteFile(port_handle,&tch,1,(LPDWORD)((void *)&n), NULL);
				usleep(80);
		}
		while(system_state != 0);
		console_exit();
		if(perr >=0) CloseHandle(port_handle);		
		return 0;
#endif
	}



#else
int main(void)
{
#endif


#if MCU > 9
	unilib_init();
	set_clock(CLOCK_SETTING);
	set_portpin_output(STAT_LED);
	set_portpin_output(DRIVE_LED);
	set_portpin_zero(STAT_LED);
	set_portpin_one(DRIVE_LED);

//	while(1);
	
	xrtc_init(RTC_CLOCK,RTC_DATA);

#if MCU == 30
	//Disable U7
	set_portpin_output(PORT_E,3);
	set_portpin_one(PORT_E,3);
#endif	
	
	unilib_pause_ms(250);
	
	enable_tick(TICK);

	TERMINAL_UART_INIT(TERMINAL_BAUD_DIVIDER);	//terminal 1
	TERMINAL_PUT(0x0c);				//clear screen
	TERMINAL_GET();					//flush input
		
	TERM9_UART_INIT(TERM9_BAUD_DIVIDER);		//terminal 2
	TERM9_PUT(0x0c);				//clear screen
	TERM9_GET();					//flush input
		
	unilib_pause_ms(100);
#endif

	set_outchar_routine(console_out);
	init_lfsr_default();				//for /dev/random
	memory_init();	
	cpu_init();
	console_init(1);
	start_adc();
	start_dac();
#if LOAD_TEST == 0
	kl11_init();
	rk11_init();
#endif
	rk11_bootsetup();
/*
	unsigned long *xptr;
	unsigned long xdat;
	xptr=0x40013000;
	xdat=*xptr;	
	KPRINTF("CR1=%lx\r\n",xdat);
	xptr=0x40013004;
	xdat=*xptr;	
	KPRINTF("CR2=%lx\r\n",xdat);
*/

#ifdef F1_OVERCLOCKING
	unilib_pause_ms(200);
	FLASH.ACR.BIT.LATENCY = 3;
	set_clock(CLOCK_SETTING2);
	enable_tick(TICK2);
	TERMINAL_UART_INIT(TERMINAL_BAUD_DIVIDER2);	//terminal 1
	TERM9_UART_INIT(TERM9_BAUD_DIVIDER2);		//terminal 2
#endif


#if MCU > 9

	unilib_pause_ms(10);
	set_portpin_one(STAT_LED);			//set status to run

	init_dio();

#if MCU == 11
	masterloop();
#endif
	while(1)
	{
		do_cycle[cycle_type >> SHIFT_CYCLE]();
	}
#endif

#if MCU < 10
	do
	{
		do_cycle[cycle_type]();

		if(++prev > 200)
		{
			serve_console();
			serve_kl11();
			prev=0;
			tick_counter++;
		}
	}
	while(system_state != 0);
	rk11_flush();
	console_exit();
#endif


	return 0;
}

