//###############################################################################
//#										#
//# universal microcontroller library						#
//#										#
//# copyright (c) 2010-2018 Joerg Wolfram (joerg@jcwolfram.de)			#
//#										#
//#										#
//# This program is free software; you can redistribute it and/or		#
//# modify it under the terms of the GNU General Public License			#
//# as published by the Free Software Foundation; either version 3		#
//# of the License, or (at your option) any later version.			#
//#										#
//# This program is distributed in the hope that it will be useful,		#
//# but WITHOUT ANY WARRANTY; without even the implied warranty of		#
//# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the GNU		#
//# General Public License for more details.					#
//#										#
//# You should have received a copy of the GNU General Public			#
//# License along with this library// if not, write to the			#
//# Free Software Foundation, Inc., 59 Temple Place - Suite 330,		#
//# Boston, MA 02111-1307, USA.							#
//#										#
//###############################################################################
#include <stdint.h>

#define PI 3.1415926536

#define size_t unsigned long

#define TAB_TYPE const
#define FDATA_TYPE const

//nop

typedef struct
{
	int quot;
	int rem;
} div_t;

typedef struct
{
	long quot;
	long rem;
} ldiv_t;

//pri


void unilib_outchar(unsigned char);

#define putc unilib_outchar

#define COLOR16_BLACK	0x0000
#define COLOR16_BLUE	0xf800
#define COLOR16_RED	0x001f
#define COLOR16_MAGENTA	0xf81f
#define COLOR16_GREEN	0x07e0
#define COLOR16_CYAN	0xffe0
#define COLOR16_YELLOW	0x07ff
#define COLOR16_WHITE	0xffff

#ifdef _TRFA
#define _TRFA
typedef enum
{
	FALSE = 0, TRUE = !FALSE
}
bool;
#endif
#ifndef _UNILIB_CAN_FRAME
#define _UNILIB_CAN_FRAME
typedef struct _unilib_can_frame{
	unsigned long id;
	unsigned char mode;
	unsigned char len;
	unsigned char data[8];
	unsigned long timestamp;
}unilib_can_frame;
#endif


#ifndef _UNILIB_DATETIME
#define _UNILIB_DATETIME
typedef struct _unilib_date_time{
	unsigned short year;
	unsigned char month;
	unsigned char day;
	unsigned char hour;
	unsigned char min;
	unsigned char sec;
}unilib_date_time;
#endif

//#define CC2500_PKT_BYTES 	33		// packet size 
//#######################################################################
//# init the CC2500 without communication setup				#
//#######################################################################
void CC2500_init_minimal(void);

//#######################################################################
//# init the CC2500 (250Kbit/s, compatibility mode with CC2541)		#
//#######################################################################
void CC2500_init_250k(PORT_TYPE cs_port, PORT_TYPE cs_pin);

//#######################################################################
//# init the CC2500 (50Kbit/s)						#
//#######################################################################
void CC2500_init_50k(PORT_TYPE cs_port, PORT_TYPE cs_pin);

//#######################################################################
//# init the CC2500 (2400 Bit/s)					#
//#######################################################################
void CC2500_init_2k4(PORT_TYPE cs_port, PORT_TYPE cs_pin);

//#######################################################################
//# send a packet							#
//#######################################################################
void CC2500_start_tx(unsigned char *txBuffer,unsigned char len,unsigned char addr);

//#######################################################################
//# start RX								#
//#######################################################################
void CC2500_start_rx(unsigned char len,unsigned char addr);

//#######################################################################
//# read a received packet						#
//#######################################################################
void CC2500_get_rxdata(unsigned char *rxBuffer,unsigned char len);

//#######################################################################
//#######################################################################
unsigned char CC2500_get_state(void);

//#######################################################################
//# set channel								#
//#######################################################################
void CC2500_set_channel(unsigned char channel);

//#######################################################################
//# get RSSI								#
//#######################################################################
unsigned char CC2500_get_rssi(void);

//#######################################################################
//# get partnumber							#
//#######################################################################
unsigned short CC2500_get_info(void);

//#######################################################################
//# goto IDLE mode							#
//#######################################################################
void CC2500_idle(void);

//#######################################################################
//# calibrate								#
//#######################################################################
void CC2500_calibrate(void);

//#######################################################################
//# detune								#
//#######################################################################
void CC2500_detune(unsigned long freq);

//#######################################################################
//# get calibration data						#
//#######################################################################
unsigned long CC2500_get_calibration(void);

//#######################################################################
//# set calibration data						#
//#######################################################################
void CC2500_set_calibration(unsigned long cal_val);

//#######################################################################
//# st into power down mode						#
//#######################################################################
void CC2500_sleep(void);

//#######################################################################
//#									#
//#	init crc16							#
//#									#
//#######################################################################
void init_crc16(unsigned short startvalue,unsigned short polynom, unsigned char invert);

//#######################################################################
//#									#
//#	update crc16 with byte						#
//#									#
//#######################################################################
void update_crc16(unsigned char val);

//#######################################################################
//#									#
//#	update crc16 with byte array					#
//#									#
//#######################################################################
void calculate_crc16(unsigned char *buffer, unsigned int len);

//#######################################################################
//#									#
//#	get crc16 value							#
//#									#
//#######################################################################
unsigned short get_crc16(void);

//############################################################################
//#                                                                          #
//# encrypt with AES128 (data is little endian)                              #
//#                                                                          #
//############################################################################
void encrypt_aes128(unsigned char* data, unsigned char* key);

//############################################################################
//#                                                                          #
//# decrypt with AES128 (data is little endian)                              #
//#                                                                          #
//############################################################################
void decrypt_aes128(unsigned char* data, unsigned char* key);

//############################################################################
//#                                                                          #
//# encrypt with AES256 (data is little endian)                              #
//#                                                                          #
//############################################################################
void encrypt_aes256(unsigned char* data, unsigned char* key);

//############################################################################
//#                                                                          #
//# decrypt with AES256 (data is little endian)                              #
//#                                                                          #
//############################################################################
void decrypt_aes256(unsigned char* data, unsigned char* key);

//##############################################################################
//#                                                                            #
//# init the hash                                                              #
//#                                                                            #
//##############################################################################
void sha256_init(void);

//##############################################################################
//#                                                                            #
//# update the hash with data                                                  #
//#                                                                            #
//##############################################################################
void sha256_update(unsigned char *data, unsigned long len);

//##############################################################################
//#                                                                            #
//# output the hash                                                            #
//#                                                                            #
//##############################################################################
void sha256_final(unsigned char *hash);

//#######################################################################
//#									#
//#	xtea encrypt							#
//#									#
//#######################################################################
void encrypt_xtea (unsigned int num_cycles, unsigned char *data, unsigned char *key);

//#######################################################################
//#									#
//#	xtea decrypt							#
//#									#
//#######################################################################
void decrypt_xtea (unsigned int num_cycles, unsigned char *data, unsigned char *key);

//#######################################################################
//# power down (signed mode = HIGH-Z)					#
//#######################################################################
void ad53x2_signed_pdown(void);

//#######################################################################
//# write both channels (signed mode)					#
//#######################################################################
void ad53x2_signed_ab(int value_a,int value_b);

//#######################################################################
//# write channel a (signed mode)					#
//#######################################################################
void ad53x2_signed_a(int value);

//#######################################################################
//# write channel b (signed mode)					#
//#######################################################################
void ad53x2_signed_b(int value);

//#######################################################################
//# init the ad53x2 (signed mode)					#
//#######################################################################
void ad53x2_init_signed(PORT_TYPE sync_port, unsigned char sync_pin,PORT_TYPE ldac_port, unsigned char ldac_pin);

//#######################################################################
//# power down (unsigned mode)	PULL-DOWN 1K				#
//#######################################################################
void ad53x2_unsigned_pdown(void);

//#######################################################################
//# write both channels (unsigned mode)					#
//#######################################################################
void ad53x2_unsigned_ab(unsigned int value_a,unsigned int value_b);

//#######################################################################
//# write channel a (unsigned mode)					#
//#######################################################################
void ad53x2_unsigned_a(unsigned int value);

//#######################################################################
//# write channel b (unsigned mode)					#
//#######################################################################
void ad53x2_unsigned_b(unsigned int value);

//#######################################################################
//# init the ad53x2 (unsigned mode)					#
//#######################################################################
void ad53x2_init_unsigned(PORT_TYPE sync_port, unsigned char sync_pin,PORT_TYPE ldac_port, unsigned char ldac_pin);

//#######################################################################
//# check device ID							#
//#######################################################################
unsigned char dataflash_checkid(void);

//#######################################################################
//# init the dataflash							#
//# ret: 	0=no dataflash						#
//# 	 	1=AT45DB041						#
//# 	 	2=AT45DB081						#
//# 	 	3=AT45DB161						#
//# 	 	4=AT45DB321						#
//# 	 	5=AT45DB642						#
//#######################################################################
unsigned char dataflash_init(PORT_TYPE cs_port, unsigned char cs_pin);

//#######################################################################
//# read ststus byte							#
//# 1=ready, 0=busy							#
//#######################################################################
unsigned char dataflash_read_status(void);

//#######################################################################
//# wait for device is ready						#
//# 0=ready, 1=timeout							#
//#######################################################################
unsigned char dataflash_wait_ready(unsigned char timeout);

//#######################################################################
//# read from page (byte-wise)						#
//#######################################################################
unsigned char dataflash_read_page(unsigned char *buffer,unsigned short page, unsigned short start, unsigned short bytes);

//#######################################################################
//# read from page (word_wise)						#
//#######################################################################
unsigned char dataflash_read_page_ww(unsigned short *buffer,unsigned short page, unsigned short start, unsigned short words);

//#######################################################################
//# read from page (longword_wise)					#
//#######################################################################
unsigned char dataflash_read_page_lw(unsigned long *buffer,unsigned short page, unsigned short start, unsigned short longs);

//#######################################################################
//# read data from buffer (byte-wise)					#
//# 0 = buffer 1							#
//# 1 = buffer 2							#
//#######################################################################
unsigned char dataflash_read_buffer(unsigned char bufnum,unsigned char *buffer, unsigned short start, unsigned short bytes);

//#######################################################################
//# read data from buffer (word-wise)					#
//# 0 = buffer 1							#
//# 1 = buffer 2							#
//#######################################################################
unsigned char dataflash_read_buffer_ww(unsigned char bufnum,unsigned short *buffer, unsigned short start, unsigned short words);

//#######################################################################
//# read data from buffer (longword-wise)				#
//# 0 = buffer 1							#
//# 1 = buffer 2							#
//#######################################################################
unsigned char dataflash_read_buffer_lw(unsigned char bufnum,unsigned long *buffer, unsigned short start, unsigned short longs);

//#######################################################################
//# write data to buffer (byte-wise)					#
//# 0 = buffer 1							#
//# 1 = buffer 2							#
//#######################################################################
unsigned char dataflash_write_buffer(unsigned char bufnum,unsigned char *buffer,unsigned short start, unsigned short bytes);

//#######################################################################
//# write data to buffer (word-wise)					#
//# 0 = buffer 1							#
//# 1 = buffer 2							#
//#######################################################################
unsigned char dataflash_write_buffer_ww(unsigned char bufnum,unsigned short *buffer,unsigned short start, unsigned short words);

//#######################################################################
//# write data to buffer (longword-wise)				#
//# 0 = buffer 1							#
//# 1 = buffer 2							#
//#######################################################################
unsigned char dataflash_write_buffer_lw(unsigned char bufnum,unsigned long *buffer,unsigned short start, unsigned short longs);

//#######################################################################
//# load buffer 1/2 from flash						#
//#######################################################################
unsigned char dataflash_load_page(unsigned char bufnum,unsigned short page,unsigned char blocked);

//#######################################################################
//# write buffer 1/2 to flash						#
//#######################################################################
unsigned char dataflash_prog_page(unsigned char bufnum,unsigned short page,unsigned char blocked);

//#######################################################################
//#									#
//#	init debounce							#
//#									#
//#######################################################################
void enable_debounce(unsigned char s);

//#######################################################################
//#									#
//#	add a debounce pin						#
//#									#
//#######################################################################
unsigned char add_debounce(PORT_TYPE debounce_port,int debounce_pin);

//#######################################################################
//#									#
//#	get debounce pin value						#
//#									#
//#######################################################################
unsigned char get_debounced(unsigned char num);

//#######################################################################
//#									#
//#	this must be called periodicly (in interrupt)			#
//#									#
//#######################################################################
void serve_debounce(void);

//#######################################################################
//#									#
//#	init ENC							#
//#									#
//#######################################################################
int enc28j60_init(unsigned char spi_port,int prescaler,PORT_TYPE sel_port,int sel_pin);

//#######################################################################
//#									#
//#	get a packet							#
//#									#
//#######################################################################
int enc28j60_receive_packet(unsigned char buffer[]);

//#######################################################################
//#									#
//#	send a packet							#
//#									#
//#######################################################################
int enc28j60_send_packet(unsigned char buffer[],int frame_length);

//#######################################################################
//#									#
//#	float: break float into normalized fraction and power of 2	#
//#									#
//#######################################################################
float frexpf(float x, int *eptr);

//#######################################################################
//#									#
//# 	float: absolute value						#
//#									#
//#######################################################################
float fabsf(float x);

//#######################################################################
//#									#
//#	float: arcus cosinus						#
//#									#
//#######################################################################
float acosf(float x);

//#######################################################################
//#									#
//#	float: arcus cosinus						#
//#									#
//#######################################################################
float asinf(float x);

//#######################################################################
//#                                                    			#
//#	float: arcus tangens 						#
//#                                                    			#
//#######################################################################
float atanf(float x);

//#######################################################################
//#                                                    			#
//#	float: arcus tangens with 2 arguments				#
//#                                                    			#
//#######################################################################
float atan2f(float y, float x);

//#######################################################################
//#									#
//#	float: round towards +inf					#
//#									#
//#######################################################################
float ceilf(float x);

//#######################################################################
//#                                                    			#
//#	float: calculate cosinus (radians arg)				#
//#                                                    			#
//#######################################################################
float cosf(float x);

//##################################################################
//#	float: e^x
//##################################################################
float expf(float x);

//#######################################################################
//#									#
//#	float: round towards -inf					#
//#									#
//#######################################################################
float floorf(float x);

//##################################################################
//#	float: ln(x)
//##################################################################
float logf(float x);

//#######################################################################
//#									#
//#	float:	log10(x)						#
//#									#
//#######################################################################
float log10f(float x);

//#######################################################################
//#									#
//#	float:	x^y							#
//#									#
//#######################################################################
float powf(float x, float y);

//#######################################################################
//#                                                    			#
//#	float: calculate sinus (radians arg)				#
//#                                                    			#
//#######################################################################
float sinf(float x);

//#######################################################################
//#									#
//#	float: square root						#
//#									#
//#######################################################################
float sqrtf(float x);

//#######################################################################
//#                                                    			#
//#	float: calculate tangens (radians arg)				#
//#                                                    			#
//#######################################################################
float tanf(float x);

//#######################################################################
//#									#
//#	set active window						#
//#									#
//#######################################################################
void glcd128ca_setwin(unsigned char x1,unsigned char y1,unsigned char x2,unsigned char y2);

//#######################################################################
//#									#
//#	fill window						#
//#									#
//#######################################################################
void glcd128ca_fillwin(unsigned char dx,unsigned char dy,unsigned int color);

//#######################################################################
//#									#
//#	set background color						#
//#									#
//#######################################################################
void glcd128ca_set_bgcolor(unsigned int color);

//#######################################################################
//#									#
//#	set foreground color						#
//#									#
//#######################################################################
void glcd128ca_set_fgcolor(unsigned int color);

//#######################################################################
//#									#
//#	set background color						#
//#									#
//#######################################################################
void glcd128ca_set_bgcolor_rgb(unsigned char red,unsigned char green, unsigned char blue);

//#######################################################################
//#									#
//#	set foreground color						#
//#									#
//#######################################################################
void glcd128ca_set_fgcolor_rgb(unsigned char red,unsigned char green, unsigned char blue);

//#######################################################################
//#									#
//#	clear screen							#
//#									#
//#######################################################################
void glcd128ca_clrscr(void);

//#######################################################################
//#									#
//#	enable display							#
//#									#
//#######################################################################
void enable_glcd128ca(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char wrpin,unsigned char rdpin,unsigned char cdpin,unsigned char cspin,unsigned char rspin);

//#######################################################################
//#									#
//#	set display contrast						#
//#									#
//#######################################################################
void glcd128ca_set_contrast(unsigned char contrast);

//#######################################################################
//#									#
//#	set screen position						#
//#									#
//#######################################################################
void glcd128ca_gotoxy(unsigned char x,unsigned char y);

//#######################################################################
//#									#
//#	new line							#
//#									#
//#######################################################################
void glcd128ca_newline(void);

//#######################################################################
//#									#
//#	output char (opaque)						#
//#									#
//#######################################################################
void glcd128ca_char(unsigned char c);

//#######################################################################
//#									#
//#	output big char (opaque)					#
//#									#
//#######################################################################
void glcd128ca_bchar(unsigned char c);

//#######################################################################
//#									#
//#	get current x coordinate					#
//#									#
//#######################################################################
unsigned int glcd128ca_get_drawpos_x(void);

//#######################################################################
//#									#
//#	get current y coordinate					#
//#									#
//#######################################################################
unsigned int glcd128ca_get_drawpos_y(void);

//#######################################################################
//#									#
//#	get current x coordinate					#
//#									#
//#######################################################################
unsigned int glcd128ca_get_textpos_x(void);

//#######################################################################
//#									#
//#	get current y coordinate					#
//#									#
//#######################################################################
unsigned int glcd128ca_get_textpos_y(void);

//#######################################################################
//#									#
//#	draw pixel							#
//#									#
//#######################################################################
void glcd128ca_plot(unsigned int x,unsigned int y);

//#######################################################################
//#									#
//#	draw line							#
//#									#
//#######################################################################
void glcd128ca_draw(unsigned int x1,unsigned int y1,unsigned int x2,unsigned int y2);

//#######################################################################
//#									#
//#	draw line to							#
//#									#
//#######################################################################
void glcd128ca_drawto(unsigned int x2,unsigned int y2);

//#######################################################################
//#									#
//#	draw box							#
//#									#
//#######################################################################
void glcd128ca_box(unsigned int x1,unsigned int y1,unsigned int dx,unsigned int dy);

//#######################################################################
//#									#
//#	draw filled box							#
//#									#
//#######################################################################
void glcd128ca_fbox(unsigned char x1,unsigned char y1,unsigned char dx,unsigned char dy);

//#######################################################################
//#									#
//#	draw sprite							#
//#									#
//#######################################################################
void glcd128ca_draw_sprite(unsigned char x,unsigned char y,unsigned char dx,unsigned char dy,unsigned int *buffer);

//#######################################################################
//#									#
//#	draw 8x8 sprite with bg						#
//#									#
//#######################################################################
void glcd128ca_draw_sprite_bg(unsigned char x,unsigned char y,unsigned int transp,unsigned int *buffer,unsigned int *background);

//#######################################################################
//#									#
//#	undraw 8x8 sprite with bg					#
//#									#
//#######################################################################
void glcd128ca_undraw_sprite_bg(unsigned char x,unsigned char y,unsigned int *background);

//#######################################################################
//#									#
//#	undraw 8x8 sprite with bg					#
//#									#
//#######################################################################
void glcd128ca_set_bg(unsigned char x,unsigned char y,unsigned char dx,unsigned char dy,unsigned int *background);

extern volatile unsigned char *unilib_gs4lcd_sdata;

//#######################################################################
//#									#
//#	select font (0=8x12  1=6x10)					#
//#									#
//#######################################################################
void gs4lcd_set_font(char font);

//#######################################################################
//#									#
//#	set color normal						#
//#									#
//#######################################################################
void gs4lcd_set_textcolor(unsigned char fgcolor,unsigned char bgcolor);

//#######################################################################
//#									#
//#	wait for screeen sync						#
//#									#
//#######################################################################
void gs4lcd_waitsync(void);

//#######################################################################
//#									#
//#	clear screen							#
//#									#
//#######################################################################
void gs4lcd_clrscr(char color);

//#######################################################################
//#									#
//#	set screen position						#
//#									#
//#######################################################################
void gs4lcd_gotoxy(unsigned char x,unsigned char y);

//#######################################################################
//#									#
//#	new line							#
//#									#
//#######################################################################
void gs4lcd_newline(void);

//#######################################################################
//#									#
//#	set wrapmode							#
//#									#
//#######################################################################
void gs4lcd_wrapmode(void);

//#######################################################################
//#									#
//#	set scrollmode							#
//#									#
//#######################################################################
void gs4lcd_scrollmode(void);

//#######################################################################
//#									#
//#	get maximal x coordinate					#
//#									#
//#######################################################################
unsigned int gs4lcd_get_drawsize_x(void);

//#######################################################################
//#									#
//#	get maximal y coordinate					#
//#									#
//#######################################################################
unsigned int gs4lcd_get_drawsize_y(void);

//#######################################################################
//#									#
//#	get maximal x coordinate					#
//#									#
//#######################################################################
unsigned int gs4lcd_get_textsize_x(void);

//#######################################################################
//#									#
//#	get maximal y coordinate					#
//#									#
//#######################################################################
unsigned int gs4lcd_get_textsize_y(void);

//#######################################################################
//#									#
//#	get current x coordinate					#
//#									#
//#######################################################################
unsigned int gs4lcd_get_drawpos_x(void);

//#######################################################################
//#									#
//#	get current y coordinate					#
//#									#
//#######################################################################
unsigned int gs4lcd_get_drawpos_y(void);

//#######################################################################
//#									#
//#	get current x coordinate					#
//#									#
//#######################################################################
unsigned int gs4lcd_get_textpos_x(void);

//#######################################################################
//#									#
//#	get current y coordinate					#
//#									#
//#######################################################################
unsigned int gs4lcd_get_textpos_y(void);

//#######################################################################
//#									#
//#	write char							#
//#									#
//#######################################################################
void gs4lcd_char(unsigned char data);

//#######################################################################
//#									#
//#	write char (raw mode)						#
//#									#
//#######################################################################
void gs4lcd_char_raw(unsigned char data);

//#######################################################################
//#									#
//#	draw pixel with color						#
//#									#
//#######################################################################
void gs4lcd_plot(unsigned int x,unsigned int y,char color);

//#######################################################################
//#									#
//#	draw line							#
//#									#
//#######################################################################
void gs4lcd_draw(unsigned int x1,unsigned int y1,unsigned int x2,unsigned int y2,char color);

//#######################################################################
//#									#
//#	draw line to							#
//#									#
//#######################################################################
void gs4lcd_drawto(unsigned int x2,unsigned int y2,char color);

//#######################################################################
//#									#
//#	draw box							#
//#									#
//#######################################################################
void gs4lcd_box(unsigned int x1,unsigned int y1,unsigned int dx,unsigned int dy,char color);

//#######################################################################
//#									#
//#	draw filled box							#
//#									#
//#######################################################################
void gs4lcd_fbox(unsigned int x1,unsigned int y1,unsigned int dx,unsigned int dy,char color);

//#######################################################################
//#									#
//#	draw 8x8 sprite							#
//#									#
//#######################################################################
void gs4lcd_sprite(unsigned int px,unsigned int py,unsigned char *sdata);

//#######################################################################
//#									#
//# draw XBM image from flash array					#
//#									#
//#######################################################################
void gs4lcd_image(unsigned char *pixbuf,unsigned int px,unsigned int py,unsigned int dx,unsigned int dy, unsigned char fcolor,unsigned char bcolor);

//#######################################################################
//#									#
//#	init 480x64 monochrome GLCD 					#
//#									#
//#	ctrlport	= port with control signals			#
//#	flm_pin		= ctrlport pin for FLM signal			#
//#	lp_pin		= ctrlport pin for LP signal			#
//#	m_pin		= ctrlport pin for M signal			#
//#									#
//#######################################################################
void init_gs4lcd480a(PORT_TYPE ctrlport,unsigned char flm_pin,unsigned char lp_pin,unsigned char m_pin);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#
//#######################################################################
void serve_gs4lcd480a(void);

extern volatile unsigned char *unilib_glcd_sdata;

extern volatile unsigned char *unilib_glcd_pattern;

//#######################################################################
//#									#
//#	set pattern pointer						#
//#									#
//#######################################################################
void glcd_set_pattern(unsigned char *ptr);

//#######################################################################
//#									#
//#	select font (0=8x12  1=6x10)					#
//#									#
//#######################################################################
void glcd_set_font(char font);

//#######################################################################
//#									#
//#	set text color							#
//#									#
//#######################################################################
void glcd_set_textcolor(unsigned char color);

//#######################################################################
//#									#
//#	wait for screeen sync						#
//#									#
//#######################################################################
void glcd_waitsync(void);

//#######################################################################
//#									#
//#	clear screen							#
//#									#
//#######################################################################
void glcd_clrscr(unsigned char color);

//#######################################################################
//#									#
//#	set screen position						#
//#									#
//#######################################################################
void glcd_gotoxy(unsigned char x,unsigned char y);

//#######################################################################
//#									#
//#	new line							#
//#									#
//#######################################################################
void glcd_newline(void);

//#######################################################################
//#									#
//#	set wrapmode							#
//#									#
//#######################################################################
void glcd_wrapmode(void);

//#######################################################################
//#									#
//#	set scrollmode							#
//#									#
//#######################################################################
void glcd_scrollmode(void);

//#######################################################################
//#									#
//#	get maximal x coordinate					#
//#									#
//#######################################################################
unsigned int glcd_get_drawsize_x(void);

//#######################################################################
//#									#
//#	get maximal y coordinate					#
//#									#
//#######################################################################
unsigned int glcd_get_drawsize_y(void);

//#######################################################################
//#									#
//#	get maximal x coordinate					#
//#									#
//#######################################################################
unsigned int glcd_get_textsize_x(void);

//#######################################################################
//#									#
//#	get maximal y coordinate					#
//#									#
//#######################################################################
unsigned int glcd_get_textsize_y(void);

//#######################################################################
//#									#
//#	get current x coordinate					#
//#									#
//#######################################################################
unsigned int glcd_get_drawpos_x(void);

//#######################################################################
//#									#
//#	get current y coordinate					#
//#									#
//#######################################################################
unsigned int glcd_get_drawpos_y(void);

//#######################################################################
//#									#
//#	get current x coordinate					#
//#									#
//#######################################################################
unsigned int glcd_get_textpos_x(void);

//#######################################################################
//#									#
//#	get current y coordinate					#
//#									#
//#######################################################################
unsigned int glcd_get_textpos_y(void);

//#######################################################################
//#									#
//#	write char							#
//#									#
//#######################################################################
void glcd_char(unsigned char data);

//#######################################################################
//#									#
//#	write char (raw mode)						#
//#									#
//#######################################################################
void glcd_char_raw(unsigned char data);

//#######################################################################
//#									#
//#	draw pixel with color						#
//#									#
//#	0	= undraw						#
//#	1	= draw							#
//#	2	= invert						#
//#	8	= pattern						#
//#	9	= inverse pattern					#
//#									#
//#######################################################################
void glcd_plot(unsigned int x,unsigned int y,char color);

//#######################################################################
//#									#
//#	draw line							#
//#									#
//#	0	= undraw						#
//#	1	= draw							#
//#	2	= invert						#
//#	8	= pattern						#
//#	9	= inverse pattern					#
//#									#
//#######################################################################
void glcd_draw(unsigned int x1,unsigned int y1,unsigned int x2,unsigned int y2,char color);

//#######################################################################
//#									#
//#	draw line to							#
//#									#
//#	0	= undraw						#
//#	1	= draw							#
//#	2	= invert						#
//#	8	= pattern						#
//#	9	= inverse pattern					#
//#									#
//#######################################################################
void glcd_drawto(unsigned int x2,unsigned int y2,char color);

//#######################################################################
//#									#
//#	draw box							#
//#									#
//#	0	= undraw						#
//#	1	= draw							#
//#	2	= invert						#
//#	8	= pattern						#
//#	9	= inverse pattern					#
//#									#
//#######################################################################
void glcd_box(unsigned int x1,unsigned int y1,unsigned int dx,unsigned int dy,char color);

//#######################################################################
//#									#
//#	draw filled box							#
//#									#
//#	0	= undraw						#
//#	1	= draw							#
//#	2	= invert						#
//#	8	= pattern						#
//#	9	= inverse pattern					#
//#									#
//#######################################################################
void glcd_fbox(unsigned int x1,unsigned int y1,unsigned int dx,unsigned int dy,char color);

//#######################################################################
//#									#
//#	draw 8x8 sprite							#
//#									#
//#######################################################################
void glcd_sprite(unsigned int px,unsigned int py,unsigned char *sdata);

//#######################################################################
//#									#
//# draw XBM image from flash array					#
//#									#
//# pcolor 0:	inverse							#
//#	   1:	normal							#
//#	   2:	toogle							#
//#	   3:	transparent						#
//#									#
//#######################################################################
void glcd_image(const unsigned char *pixbuf,unsigned int px,unsigned int py,unsigned int dx,unsigned int dy, unsigned char pcolor);

//#######################################################################
//#									#
//#	enable GLCD (unipolar mode)					#
//#									#
//#######################################################################
void init_glcd128a(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char cs1pin,unsigned char cs2pin,unsigned char dipin,unsigned char epin);

//#######################################################################
//#									#
//#	set display contrast						#
//#									#
//#######################################################################
void glcd128a_set_contrast(unsigned char contrast);

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_glcd128a(void);

//#######################################################################
//#									#
//#	init display					#
//#									#
//#######################################################################
void init_glcd128b(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char resetpin,unsigned char cspin,unsigned char rspin,unsigned char wrpin,unsigned char rdpin);

//#######################################################################
//#									#
//#	init display (without cs/rd)					#
//#									#
//#######################################################################
void init_glcd128b_minimal(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char resetpin,unsigned char rspin,unsigned char wrpin);

//#######################################################################
//#									#
//#	set display contrast						#
//#									#
//#######################################################################
void glcd128b_set_contrast(unsigned char contrast);

//#######################################################################
//#									#
//#	switch display on						#
//#									#
//#######################################################################
void glcd128b_on(void);

//#######################################################################
//#									#
//#	switch display off						#
//#									#
//#######################################################################
void glcd128b_off(void);

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_glcd128b(void);

//#######################################################################
//#									#
//#	init LCD (positive CS)						#
//#									#
//#######################################################################
void init_glcd128c(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char cs1pin,unsigned char cs2pin,unsigned char dipin,unsigned char epin);

//#######################################################################
//#									#
//#	set display contrast						#
//#									#
//#######################################################################
void glcd128c_set_contrast(unsigned char contrast);

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_glcd128c(void);

//#######################################################################
//#									#
//#	init 320x64 monochrome GLCD 					#
//#									#
//#	ctrlport	= port with control signals			#
//#	flm_pin		= ctrlport pin for FLM signal			#
//#	lp_pin		= ctrlport pin for LP signal			#
//#	m_pin		= ctrlport pin for M signal			#
//#									#
//#######################################################################
void init_glcd320a(PORT_TYPE dataport,unsigned char doffset,PORT_TYPE ctrlport,unsigned char cp_pin,unsigned char lp_pin,unsigned char flm_pin);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#
//#######################################################################
void serve_glcd320a(void);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#
//#######################################################################
void serve_glcd320a(void);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#
//#######################################################################
void serve_glcd320a_inv(void);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#
//#######################################################################
void serve_glcd320a_inv(void);

//#######################################################################
//#									#
//#	init 320x240 monochrome GLCD 					#
//#									#
//#	dataport	= port with data signals in D0-D3  		#
//#	ctrlport	= port with control signals			#
//#	flm_pin		= ctrlport pin for FLM signal			#
//#	lp_pin		= ctrlport pin for LP signal			#
//#	m_pin		= ctrlport pin for M signal			#
//#									#
//#######################################################################
void init_glcd320a_n0(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char cp_pin,unsigned char lp_pin,unsigned char flm_pin);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#	
//#######################################################################
void serve_glcd320a_n0(void);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#
//#######################################################################
void serve_glcd320a_n0_inv(void);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#	
//#######################################################################
void serve_glcd320a_n0(void);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#
//#######################################################################
void serve_glcd320a_n0_inv(void);

//#######################################################################
//#									#
//#	init 320x240 monochrome GLCD 					#
//#									#
//#	dataport	= port with data signals in D4-D7  		#
//#	ctrlport	= port with control signals			#
//#	flm_pin		= ctrlport pin for FLM signal			#
//#	lp_pin		= ctrlport pin for LP signal			#
//#	m_pin		= ctrlport pin for M signal			#
//#									#
//#######################################################################
void init_glcd320a_n1(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char cp_pin,unsigned char lp_pin,unsigned char flm_pin);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#	
//#######################################################################
void serve_glcd320a_n1(void);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#
//#######################################################################
void serve_glcd320a_n1_inv(void);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#	
//#######################################################################
void serve_glcd320a_n1(void);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#
//#######################################################################
void serve_glcd320a_n1_inv(void);

//#######################################################################
//#									#
//#	init 320x240 monochrome GLCD 					#
//#									#
//#	dataport	= port with data signals in D8-D11 		#
//#	ctrlport	= port with control signals			#
//#	flm_pin		= ctrlport pin for FLM signal			#
//#	lp_pin		= ctrlport pin for LP signal			#
//#	m_pin		= ctrlport pin for M signal			#
//#									#
//#######################################################################
void init_glcd320a_n2(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char cp_pin,unsigned char lp_pin,unsigned char flm_pin);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#	
//#######################################################################
void serve_glcd320a_n2(void);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#
//#######################################################################
void serve_glcd320a_n2_inv(void);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#	
//#######################################################################
void serve_glcd320a_n2(void);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#
//#######################################################################
void serve_glcd320a_n2_inv(void);

//#######################################################################
//#									#
//#	init 320x240 monochrome GLCD 					#
//#									#
//#	dataport	= port with data signals in D12-D15		#
//#	ctrlport	= port with control signals			#
//#	flm_pin		= ctrlport pin for FLM signal			#
//#	lp_pin		= ctrlport pin for LP signal			#
//#	m_pin		= ctrlport pin for M signal			#
//#									#
//#######################################################################
void init_glcd320a_n3(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char cp_pin,unsigned char lp_pin,unsigned char flm_pin);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#	
//#######################################################################
void serve_glcd320a_n3(void);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#
//#######################################################################
void serve_glcd320a_n3_inv(void);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#	
//#######################################################################
void serve_glcd320a_n3(void);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#
//#######################################################################
void serve_glcd320a_n3_inv(void);

//#######################################################################
//#									#
//#	init 480x64 monochrome GLCD 					#
//#									#
//#	ctrlport	= port with control signals			#
//#	flm_pin		= ctrlport pin for FLM signal			#
//#	lp_pin		= ctrlport pin for LP signal			#
//#	m_pin		= ctrlport pin for M signal			#
//#									#
//#######################################################################
void init_glcd480a(PORT_TYPE ctrlport,unsigned char flm_pin,unsigned char lp_pin,unsigned char m_pin);

//#######################################################################
//#									#
//#	service	routine (per pixel line, should be > 4KHz)		#
//#									#
//#######################################################################
void serve_glcd480a(void);

//#######################################################################
//#									#
//#	select font (0=8x12  1=6x10)					#
//#									#
//#######################################################################
void glcd_ellipse(unsigned int mx,unsigned int my,unsigned char rx,unsigned char ry,char color);

//#######################################################################
//#									#
//#	hamming encode							#
//#									#
//#######################################################################
unsigned int encode_hamm4 (unsigned char value);

//#######################################################################
//#									#
//#	hamming decode	(results error count in upper byte)		#
//#									#
//#######################################################################
unsigned int decode_hamm4 (unsigned int value);

extern unsigned char _iclcd_font;

extern unsigned char _iclcd_lwidth;

//###############################################################################
//#										#
//#	test if display is available (0/1)					#
//#										#
//###############################################################################
int iclcd_test(void);

//###############################################################################
//#										#
//#	clear screen with actual background color				#
//#										#
//###############################################################################
void iclcd_clear(void);

//###############################################################################
//#										#
//#	set color (foregroundg, background)					#
//#										#
//###############################################################################
void iclcd_set_color(unsigned char fg, unsigned char bg);

//###############################################################################
//#										#
//#	set text position							#
//#										#
//###############################################################################
void iclcd_set_textpos(unsigned char row, unsigned char col);

//###############################################################################
//#										#
//#	set last point								#
//#										#
//###############################################################################
void iclcd_set_graphpos(unsigned char y, unsigned char x);

//###############################################################################
//#										#
//#	plot a point								#
//#										#
//###############################################################################
void iclcd_plot(unsigned char y, unsigned char x);

//###############################################################################
//#										#
//#	draw line								#
//#										#
//###############################################################################
void iclcd_draw(unsigned char y1, unsigned char x1,unsigned char y2, unsigned char x2);

//###############################################################################
//#										#
//#	draw line from last point						#
//#										#
//###############################################################################
void iclcd_drawto(unsigned char y, unsigned char x);

//###############################################################################
//#										#
//#	draw filled box								#
//#										#
//###############################################################################
void iclcd_fbox(unsigned char y1, unsigned char x1,unsigned char y2, unsigned char x2);

//###############################################################################
//#										#
//#	draw ellipse								#
//#										#
//###############################################################################
void iclcd_ellipse(unsigned char y, unsigned char x,unsigned char ry, unsigned char rx);

//###############################################################################
//#										#
//#	set lcd font size (0..3)						#
//#										#
//###############################################################################
void iclcd_set_font(unsigned char font);

//###############################################################################
//#										#
//#	set line width for drawing (1..3)					#
//#										#
//###############################################################################
void iclcd_set_linewidth(unsigned char width);

//###############################################################################
//#										#
//#	lcd wakeup (display and backlight on)					#
//#										#
//###############################################################################
void iclcd_wake(void);

//###############################################################################
//#										#
//#	lcd sleep (display and backlight off)					#
//#										#
//###############################################################################
void iclcd_sleep(void);

//###############################################################################
//#										#
//#	my little printf							#
//#										#
//###############################################################################
void iclcd_printf (const char *format, ...);

//###############################################################################
//#										#
//#	output packed greyscale data line from buffer				#
//#										#
//###############################################################################
void iclcd_greyline(unsigned char line_number,unsigned char *buffer);

//###############################################################################
//#										#
//#	output packed color data line from buffer				#
//#										#
//###############################################################################
void iclcd_colorline(unsigned char line_number,unsigned char *buffer);

//###############################################################################
//#										#
//#	output rgb data line							#
//#										#
//###############################################################################
void iclcd_rgbline(unsigned char line_number,unsigned char *buffer);

//###############################################################################
//#										#
//#	set tile pattern							#
//#										#
//###############################################################################
void iclcd_tile_set(unsigned char tile_number,unsigned char *buffer);

//###############################################################################
//#										#
//#	fill tile solid								#
//#										#
//###############################################################################
void iclcd_tile_set_solid(unsigned char tile_number,unsigned char color);

//###############################################################################
//#										#
//#	add pre-defined pattern to tile						#
//#										#
//###############################################################################
void iclcd_tile_add_pattern(unsigned char tile_number,unsigned char pattern,unsigned char color);

//###############################################################################
//#										#
//#	add character from charset to tile					#
//#										#
//###############################################################################
void iclcd_tile_add_char(unsigned char tile_number,unsigned char tchar,unsigned char color);

//###############################################################################
//#										#
//#	copy a tile								#
//#										#
//###############################################################################
void iclcd_tile_copy(unsigned char dst_tile,unsigned char src_tile);

//###############################################################################
//#										#
//#	X-mirror a tile								#
//#										#
//###############################################################################
void iclcd_tile_mirror_x(unsigned char tile);

//###############################################################################
//#										#
//#	Y-mirror a tile								#
//#										#
//###############################################################################
void iclcd_tile_mirror_y(unsigned char tile);

//###############################################################################
//#										#
//#	rotate a tile CW							#
//#										#
//###############################################################################
void iclcd_tile_rotate_cw(unsigned char tile);

//###############################################################################
//#										#
//#	rotate a tile CCW							#
//#										#
//###############################################################################
void iclcd_tile_rotate_ccw(unsigned char tile);

//###############################################################################
//#										#
//#	invert a tile (negative color)						#
//#										#
//###############################################################################
void iclcd_tile_invert(unsigned char tile);

//###############################################################################
//#										#
//#	draw a sprite								#
//#										#
//###############################################################################
void iclcd_sprite_draw(	unsigned char tile,unsigned char backup_tile,unsigned char posy,unsigned char posx,unsigned char transp_color);

//###############################################################################
//#										#
//#	undraw a sprite (restore background)					#
//#										#
//###############################################################################
void iclcd_sprite_restore(unsigned char backup_tile,unsigned char posy,unsigned char posx);

//###############################################################################
//#										#
//#										#
//#										#
//###############################################################################
void iclcd_bigsprite_draw(unsigned char tile,unsigned char backup_tile,unsigned char posy,unsigned char posx,unsigned char transp_color);

//###############################################################################
//#										#
//#										#
//#										#
//###############################################################################
void iclcd_bigsprite_restore(unsigned char backup_tile,unsigned char posy,unsigned char posx);

//#######################################################################
//# ckeck presence							#
//#######################################################################
unsigned char gen_24c01_check(unsigned char devaddr);

//#######################################################################
//# read bytes								#
//#######################################################################
unsigned char gen_24c01_read(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# write bytes								#
//#######################################################################
unsigned char gen_24c01_write(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# ckeck presence							#
//#######################################################################
unsigned char gen_24c02_check(unsigned char devaddr);

//#######################################################################
//# read bytes								#
//#######################################################################
unsigned char gen_24c02_read(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# write bytes								#
//#######################################################################
unsigned char gen_24c02_write(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# ckeck presence							#
//#######################################################################
unsigned char gen_24c04_check(unsigned char devaddr);

//#######################################################################
//# read bytes								#
//#######################################################################
unsigned char gen_24c04_read(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# write bytes								#
//#######################################################################
unsigned char gen_24c04_write(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# ckeck presence							#
//#######################################################################
unsigned char gen_24c08_check(unsigned char devaddr);

//#######################################################################
//# read bytes								#
//#######################################################################
unsigned char gen_24c08_read(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# write bytes								#
//#######################################################################
unsigned char gen_24c08_write(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# ckeck presence							#
//#######################################################################
unsigned char gen_24c128_check(unsigned char devaddr);

//#######################################################################
//# read bytes								#
//#######################################################################
unsigned char gen_24c128_read(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# write bytes								#
//#######################################################################
unsigned char gen_24c128_write(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# ckeck presence							#
//#######################################################################
unsigned char gen_24c16_check(unsigned char devaddr);

//#######################################################################
//# read bytes								#
//#######################################################################
unsigned char gen_24c16_read(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# write bytes								#
//#######################################################################
unsigned char gen_24c16_write(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# ckeck presence							#
//#######################################################################
unsigned char gen_24c256_check(unsigned char devaddr);

//#######################################################################
//# read bytes								#
//#######################################################################
unsigned char gen_24c256_read(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# write bytes								#
//#######################################################################
unsigned char gen_24c256_write(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# ckeck presence							#
//#######################################################################
unsigned char gen_24c32_check(unsigned char devaddr);

//#######################################################################
//# read bytes								#
//#######################################################################
unsigned char gen_24c32_read(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# write bytes								#
//#######################################################################
unsigned char gen_24c32_write(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# ckeck presence							#
//#######################################################################
unsigned char gen_24c512_check(unsigned char devaddr);

//#######################################################################
//# read bytes								#
//#######################################################################
unsigned char gen_24c512_read(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# write bytes								#
//#######################################################################
unsigned char gen_24c512_write(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# ckeck presence							#
//#######################################################################
unsigned char gen_24c32_check(unsigned char devaddr);

//#######################################################################
//# read bytes								#
//#######################################################################
unsigned char gen_24c32_read(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# write bytes								#
//#######################################################################
unsigned char gen_24c32_write(unsigned char *buffer,unsigned int addr, int len, unsigned char devaddr);

//#######################################################################
//# ckeck presence of device with addr devaddr				#
//#######################################################################
unsigned char lm73_check(unsigned char devaddr);

//#######################################################################
//# read temperature as integer (grad * 100)				#
//#######################################################################
int lm73_read_int(unsigned char devaddr);

//#######################################################################
//# read temperature as float with current resolution			#
//#######################################################################
float lm73_read_float(unsigned char devaddr);

//#######################################################################
//# set sensor resolution						#
//#									#
//# 0 = 11 Bits (0,25°C resolution)					#
//# 1 = 12 Bits (0,125°C resolution)					#
//# 2 = 13 Bits (0,0625°C resolution)					#
//# 3 = 14 Bits (0,031625°C resolution)					#
//#									#
//#######################################################################
unsigned char lm73_set_resolution(unsigned char devaddr,unsigned char resolution);

//#######################################################################
//# set sensor to power-down						#
//#######################################################################
unsigned char lm73_pwr_down(unsigned char devaddr);

//#######################################################################
//# power-up sensor							#
//#######################################################################
unsigned char lm73_pwr_up(unsigned char devaddr);

//#######################################################################
//# ckeck presence							#
//#######################################################################
unsigned char lm75_check(unsigned char devaddr);

//#######################################################################
//# read temperature (grad * 10)					#
//#######################################################################
int lm75_read_int(unsigned char devaddr);

//#######################################################################
//# read temperature (grad)						#
//#######################################################################
float lm75_read_float(unsigned char devaddr);

//#######################################################################
//# set sensor to power-down						#
//#######################################################################
unsigned char lm75_pwr_down(unsigned char devaddr);

//#######################################################################
//# power-up sensor							#
//#######################################################################
unsigned char lm75_pwr_up(unsigned char devaddr);

//#######################################################################
//# ckeck presence of device with addr devaddr				#
//#######################################################################
unsigned char lp25h_check(unsigned char devaddr);

//#######################################################################
//# ckeck presence of device with addr devaddr				#
//#######################################################################
unsigned char lp25h_enable(unsigned char devaddr);

//#######################################################################
//# ckeck presence of device with addr devaddr				#
//#######################################################################
unsigned char lp25h_disable(unsigned char devaddr);

//#######################################################################
//# read temperature as integer (grad * 100)				#
//#######################################################################
float lp25h_read_pressure(unsigned char devaddr);

//#######################################################################
//# read temperature as integer (grad * 100)				#
//#######################################################################
float lp25h_read_temperature(unsigned char devaddr);

//#######################################################################
//# ckeck presence of device with addr devaddr				#
//#######################################################################
unsigned char ov7431_check(void);

//#######################################################################
//# set gain (0-127)							#
//#######################################################################
unsigned char ov7431_set_gain(unsigned char gain);

//#######################################################################
//# set brightness (0-255)						#
//#######################################################################
unsigned char ov7431_set_brightness(unsigned char brightness);

//#######################################################################
//# set edge enhancement (0/1)						#
//#######################################################################
unsigned char ov7431_set_edge_enhancement(unsigned char eh);

//#######################################################################
//# set analog boost (0/1)						#
//#######################################################################
unsigned char ov7431_set_boost(unsigned char boost);

//#######################################################################
//# set exposure time (0-82)						#
//#######################################################################
unsigned char ov7431_set_exposure(unsigned char time);

//#######################################################################
//# set AGC/AEC (0=manual / 1=auto)					#
//#######################################################################
unsigned char ov7431_set_mode(unsigned char mode);

//#######################################################################
//# default init for OV7431						#
//#######################################################################
unsigned char ov7431_init(void);

//#######################################################################
//#									#
//#	init lfsr default						#
//#									#
//#######################################################################
void init_lfsr_default(void);

//#######################################################################
//#									#
//#	init lfsr custom						#
//#									#
//#######################################################################
void init_lfsr_custom(unsigned long val32, unsigned long val31);

//#######################################################################
//#									#
//#	get random value						#
//#									#
//#######################################################################
unsigned short get_lfsr_random(void);

extern volatile unsigned char unilib_matrix_c57_data[16];

//#######################################################################
//#									#
//#	init 7x5 red/green matrix (landscape)				#
//#									#
//#	row_port	= pport with ROW signals			#
//#	red_port	= pport with RED COLUMN signals			#
//#	green_port	= pport with GREEN COLUMN signals		#
//#	polarity	= bit0=ROW, bit1=COL				#
//#									#
//#######################################################################
void matrix_c57_init(PORT_TYPE row_port,PORT_TYPE red_port,PORT_TYPE green_port,char polarity);

//#######################################################################
//#									#
//#	brightness control						#
//#									#
//#	bright	= 1..4							#
//#									#
//#######################################################################
void matrix_c57_bright(char bright);

//#######################################################################
//#									#
//#	plot into 7x5 red/green matrix 					#
//#									#
//#	x		= X coordinate (0..7)				#
//#	y		= Y coordinate (0..7)				#
//#	color:								#
//#	0		= black						#
//#	1		= red						#
//#	2		= green						#
//#	3		= yellow					#
//#									#
//#######################################################################
void matrix_c57_plot(char x,char y,char color);

//#######################################################################
//#									#
//#	draw 7x5 red/green matrix from flash (landscape)		#
//#									#
//#######################################################################
void matrix_c57_romimage(FDATA_TYPE unsigned char *data);

//#######################################################################
//#									#
//#	draw 7x5 red/green matrix from ram (landscape)			#
//#									#
//#######################################################################
void matrix_c57_ramimage(unsigned char data[35]);

//#######################################################################
//#									#
//#	scroll				 				#
//#	dir:								#
//#	0		= up						#
//#	1		= right						#
//#	2		= down						#
//#	3		= left						#
//#	shift in color:							#
//#	0		= off						#
//#	1		= one						#
//#									#
//#######################################################################
void matrix_c57_scroll(char dir,char color);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, msb first			#
//#									#
//#######################################################################
void serve_matrix_c57_hh(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, lsb first			#
//#									#
//#######################################################################
void serve_matrix_c57_hh_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, msb first			#
//#									#
//#######################################################################
void serve_matrix_c57_hl(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, lsb first			#
//#									#
//#######################################################################
void serve_matrix_c57_hl_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, msb first			#
//#									#
//#######################################################################
void serve_matrix_c57_lh(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, lsb first			#
//#									#
//#######################################################################
void serve_matrix_c57_lh_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, msb first			#
//#									#
//#######################################################################
void serve_matrix_c57_ll(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, lsb first			#
//#									#
//#######################################################################
void serve_matrix_c57_ll_mirrored(void);

extern volatile unsigned char unilib_matrix_c75_data[16];

//#######################################################################
//#									#
//#	init 7x5 red/green matrix 					#
//#									#
//#	row_port	= pport with ROW signals			#
//#	red_port	= pport with RED COLUMN signals			#
//#	green_port	= pport with GREEN COLUMN signals		#
//#	polarity	= bit0=ROW, bit1=COL				#
//#									#
//#######################################################################
void matrix_c75_init(PORT_TYPE row_port,PORT_TYPE red_port,PORT_TYPE green_port,char polarity);

//#######################################################################
//#									#
//#	brightness control 						#
//#									#
//#	bright	= 1..4							#
//#									#
//#######################################################################
void matrix_c75_bright(char bright);

//#######################################################################
//#									#
//#	plot into 7x5 red/green matrix 					#
//#									#
//#	x		= X coordinate (0..7)				#
//#	y		= Y coordinate (0..7)				#
//#	color:								#
//#	0		= black						#
//#	1		= red						#
//#	2		= green						#
//#	3		= yellow					#
//#									#
//#######################################################################
void matrix_c75_plot(char x,char y,char color);

//#######################################################################
//#									#
//#	draw 7x5 red/green matrix from flash				#
//#									#
//#######################################################################
void matrix_c75_romimage(FDATA_TYPE unsigned char *data);

//#######################################################################
//#									#
//#	draw 7x5 red/green matrix from ram				#
//#									#
//#######################################################################
void matrix_c75_ramimage(unsigned char data[35]);

//#######################################################################
//#									#
//#	scroll				 				#
//#	dir:								#
//#	0		= up						#
//#	1		= right						#
//#	2		= down						#
//#	3		= left						#
//#	shift in color:							#
//#	0		= off						#
//#	1		= one						#
//#									#
//#######################################################################
void matrix_c75_scroll(char dir,char color);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, msb first			#
//#									#
//#######################################################################
void serve_matrix_c75_hh(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, lsb first			#
//#									#
//#######################################################################
void serve_matrix_c75_hh_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, msb first			#
//#									#
//#######################################################################
void serve_matrix_c75_hl(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, lsb first			#
//#									#
//#######################################################################
void serve_matrix_c75_hl_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, msb first			#
//#									#
//#######################################################################
void serve_matrix_c75_lh(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, lsb first			#
//#									#
//#######################################################################
void serve_matrix_c75_lh_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, msb first			#
//#									#
//#######################################################################
void serve_matrix_c75_ll(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, lsb first			#
//#									#
//#######################################################################
void serve_matrix_c75_ll_mirrored(void);

extern volatile unsigned char unilib_matrix_g16_data[48];

//#######################################################################
//#									#
//#	init 8x16 greyscale matrix 					#
//#									#
//#	row_port	= pport with ROW signals			#
//#	col1_port	= pport with COLUMN signals (left)		#
//#	col2_port	= pport with COLUMN signals (right)		#
//#	num_rows	= rows to serve (1..8)				#
//#	polarity	= bit0=ROW, bit1=COL				#
//#									#
//#######################################################################
void matrix_g16_init(PORT_TYPE row_port,PORT_TYPE col1_port,PORT_TYPE col2_port,char num_rows,char polarity);

//#######################################################################
//#									#
//#	plot into 8x8 greyscale matrix 					#
//#									#
//#	x		= X coordinate (0..7)				#
//#	y		= Y coordinate (0..7)				#
//#	color:								#
//#	0..3		= brightness					#
//#									#
//#######################################################################
void matrix_g16_plot(char x,char y,char color);

//#######################################################################
//#									#
//#	write two chars			 				#
//#									#
//#	x		= X coordinate (0..7)				#
//#	y		= Y coordinate (0..7)				#
//#	0..3		= brightness					#
//#									#
//#######################################################################
void matrix_g16_text(char c1,char c2,char color,char side);

//#######################################################################
//#									#
//#	scroll				 				#
//#	dir:								#
//#	0		= up						#
//#	1		= right						#
//#	2		= down						#
//#	3		= left						#
//#	shift in color:							#
//#	0..3		= brightness					#
//#									#
//#######################################################################
void matrix_g16_scroll(char dir,char color);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, columns high active, msb left			#
//#									#
//#######################################################################
void serve_matrix_g16_hh(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, columns high active, lsb left			#
//#									#
//#######################################################################
void serve_matrix_g16_hh_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, columns low active, msb left			#
//#									#
//#######################################################################
void serve_matrix_g16_hl(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, columns low active, lsb left			#
//#									#
//#######################################################################
void serve_matrix_g16_hl_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows low active, columns high active, msb left			#
//#									#
//#######################################################################
void serve_matrix_g16_lh(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows low active, columns high active, lsb left			#
//#									#
//#######################################################################
void serve_matrix_g16_lh_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows low active, columns low active, msb left			#
//#									#
//#######################################################################
void serve_matrix_g16_ll(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows low active, columns low active, lsb left			#
//#									#
//#######################################################################
void serve_matrix_g16_ll_mirrored(void);

extern volatile unsigned char unilib_matrix_g8_data[24];

//#######################################################################
//#									#
//#	init 8x8 greyscale matrix 					#
//#									#
//#	row_port	= pport with ROW signals			#
//#	col1_port	= pport with COLUMN signals			#
//#	num_rows	= rows to serve (1..8)				#
//#	polarity	= bit0=COLS, bit1=ROWS, 0=low_act		#
//#									#
//#######################################################################
void matrix_g8_init(PORT_TYPE row_port,PORT_TYPE col1_port,char num_rows,char polarity);

//#######################################################################
//#									#
//#	plot into 8x8 greyscale matrix 					#
//#									#
//#	x		= X coordinate (0..7)				#
//#	y		= Y coordinate (0..7)				#
//#	color:								#
//#	0..3		= brightness					#
//#									#
//#######################################################################
void matrix_g8_plot(char x,char y,char color);

//#######################################################################
//#									#
//#	write two chars			 				#
//#									#
//#	x		= X coordinate (0..7)				#
//#	y		= Y coordinate (0..7)				#
//#	0..3		= brightness					#
//#									#
//#######################################################################
void matrix_g8_text(char c1,char c2,char color);

//#######################################################################
//#									#
//#	scroll				 				#
//#	dir:								#
//#	0		= up						#
//#	1		= right						#
//#	2		= down						#
//#	3		= left						#
//#	shift in color:							#
//#	0		= off						#
//#	1		= one						#
//#									#
//#######################################################################
void matrix_g8_scroll(char dir,char color);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, columns high active, msb left			#
//#									#
//#######################################################################
void serve_matrix_g8_hh(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, columns high active, lsb left			#
//#									#
//#######################################################################
void serve_matrix_g8_hh_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, columns low active, msb left			#
//#									#
//#######################################################################
void serve_matrix_g8_hl(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, columns low active, lsb left			#
//#									#
//#######################################################################
void serve_matrix_g8_hl_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows low active, columns high active, msb left			#
//#									#
//#######################################################################
void serve_matrix_g8_lh(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows low active, columns high active, lsb left			#
//#									#
//#######################################################################
void serve_matrix_g8_lh_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows low active, columns low active, msb left			#
//#									#
//#######################################################################
void serve_matrix_g8_ll(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows low active, columns low active, lsb left			#
//#									#
//#######################################################################
void serve_matrix_g8_ll_mirrored(void);

extern volatile unsigned char unilib_matrix_m16_data[16];

//#######################################################################
//#									#
//#	init 8x8 monochrome matrix 					#
//#									#
//#	row_port	= pport with ROW signals			#
//#	col1_port	= pport with COLUMN signals (left)		#
//#	col2_port	= pport with COLUMN signals (right)		#
//#	num_rows	= rows to serve (1..8)				#
//#	polarity	= bit0=ROW, bit1=COL				#
//#									#
//#######################################################################
void matrix_m16_init(PORT_TYPE row_port,PORT_TYPE col1_port,PORT_TYPE col2_port,char num_rows,char polarity);

//#######################################################################
//#									#
//#	brightness control						#
//#									#
//#	bright	= 1..4							#
//#									#
//#######################################################################
void matrix_m16_bright(char bright);

//#######################################################################
//#									#
//#	plot into 8x8 monochrome matrix 				#
//#									#
//#	x		= X coordinate (0..7)				#
//#	y		= Y coordinate (0..7)				#
//#	color:								#
//#	0		= set dark					#
//#	1		= set bright					#
//#	2...		= toggle					#
//#									#
//#######################################################################
void matrix_m16_plot(char x,char y,char color);

//#######################################################################
//#									#
//#	write two chars			 				#
//#									#
//#	c1		= left char					#
//#	c2		= right char					#
//#	side		= side (0=left, 1=right)			#
//#	color:								#
//#	0		= inverse					#
//#	1		= normal					#
//#	2...		= toggle					#
//#									#
//#######################################################################
void matrix_m16_text(char c1,char c2,char side,char color);

//#######################################################################
//#									#
//#	scroll				 				#
//#	dir:								#
//#	0		= up						#
//#	1		= right						#
//#	2		= down						#
//#	3		= left						#
//#	shift in color:							#
//#	0		= off						#
//#	1		= one						#
//#									#
//#######################################################################
void matrix_m16_scroll(char dir,char color);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, msb first			#
//#									#
//#######################################################################
void serve_matrix_m16_hh(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, lsb first			#
//#									#
//#######################################################################
void serve_matrix_m16_hh_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols low_active, msb first			#
//#									#
//#######################################################################
void serve_matrix_m16_hl(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols low_active, lsb first			#
//#									#
//#######################################################################
void serve_matrix_m16_hl_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows low active, cols high_active, msb first			#
//#									#
//#######################################################################
void serve_matrix_m16_lh(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows low active, cols high_active, lsb first			#
//#									#
//#######################################################################
void serve_matrix_m16_lh_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 500Hz)			#
//#	rows low active, cols low_active, msb left			#
//#									#
//#######################################################################
void serve_matrix_m16_ll(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 500Hz)			#
//#	rows low active, cols low_active, lsb left			#
//#									#
//#######################################################################
void serve_matrix_m16_ll_mirrored(void);

extern volatile unsigned char unilib_matrix_m8_data[8];

//#######################################################################
//#									#
//#	init 8x8 monochrome matrix 					#
//#									#
//#	row_port	= pport with ROW signals			#
//#	col1_port	= pport with COLUMN signals			#
//#	num_rows	= rows to serve (1..8)				#
//#	polarity	= bit0=ROW, bit1=COL 0=low active		#
//#									#
//#######################################################################
void matrix_m8_init(PORT_TYPE row_port,PORT_TYPE col1_port,char num_rows, char polarity);

//#######################################################################
//#									#
//#	brightness control			 			#
//#									#
//#	bright	= 1..4							#
//#									#
//#######################################################################
void matrix_m8_bright(char bright);

//#######################################################################
//#									#
//#	plot into 8x8 monochrome matrix 				#
//#									#
//#	x		= X coordinate (0..7)				#
//#	y		= Y coordinate (0..7)				#
//#	color:								#
//#	0		= set dark					#
//#	1		= set bright					#
//#	2...		= toggle					#
//#									#
//#######################################################################
void matrix_m8_plot(char x,char y,char color);

//#######################################################################
//#									#
//#	write two chars			 				#
//#									#
//#	x		= X coordinate (0..7)				#
//#	y		= Y coordinate (0..7)				#
//#	color:								#
//#	0		= inverse					#
//#	1		= normal					#
//#	2...		= toggle					#
//#									#
//#######################################################################
void matrix_m8_text(char c1,char c2,char color);

//#######################################################################
//#									#
//#	scroll				 				#
//#	dir:								#
//#	0		= up						#
//#	1		= right						#
//#	2		= down						#
//#	3		= left						#
//#	shift in color:							#
//#	0		= off						#
//#	1		= one						#
//#									#
//#######################################################################
void matrix_m8_scroll(char dir,char color);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, msb first			#
//#									#
//#######################################################################
void serve_matrix_m8_hh(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows high active, cols high_active, lsb first			#
//#									#
//#######################################################################
void serve_matrix_m8_hh_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 500Hz)			#
//#	rows high active, cols low_active, msb first			#
//#									#
//#######################################################################
void serve_matrix_m8_hl(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 500Hz)			#
//#	rows high active, cols low_active, lsb first			#
//#									#
//#######################################################################
void serve_matrix_m8_hl_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows low active, cols high_active, msb first			#
//#									#
//#######################################################################
void serve_matrix_m8_lh(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows low active, cols high_active, lsb first			#
//#									#
//#######################################################################
void serve_matrix_m8_lh_mirrored(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows low active, cols low_active, msb left			#
//#									#
//#######################################################################
void serve_matrix_m8_ll(void);

//#######################################################################
//#									#
//#	service	routine (per row, should be > 16KHz)			#
//#	rows low active, cols low_active, lsb left			#
//#									#
//#######################################################################
void serve_matrix_m8_ll_mirrored(void);

//#######################################################################
//# init the mlx90316							#
//#######################################################################
char MLX90316_init(PORT_TYPE cs_port, unsigned char cs_pin);

//#######################################################################
//# get angle								#
//#######################################################################
int mlx90316_read(void);

void mlx90363_comm(void);

//#######################################################################
//# reboot cmd								#
//#######################################################################
void mlx90363_reboot(void);

//#######################################################################
//# reboot cmd								#
//#######################################################################
void mlx90363_sleep(void);

//#######################################################################
//# init the mlx90363							#
//#######################################################################
char MLX90363_init(PORT_TYPE cs_port, unsigned char cs_pin);

//#######################################################################
//# read data (non-blocked)						#
//# result: 0=data / 1=no data						#
//#######################################################################
char MLX90363_get_xyz(short *xval,short *yval,short *zval);

//#######################################################################
//# wait for data available and read 					#
//# result: 0=data / 1=no data (timeout)				#
//#######################################################################
char MLX90363_wait_xyz(short *xval,short *yval,short *zval);

//#######################################################################
//# set channel								#
//#######################################################################
void rfm12_set_channel(int channel);

//#######################################################################
//# send a byte								#
//#######################################################################
void rfm12_byte_send(unsigned char data);

//#######################################################################
//# send a hamming coded byte						#
//#######################################################################
void rfm12_hamm_send(unsigned char data);

//#######################################################################
//# reveive a byte							#
//#######################################################################
unsigned char rfm12_byte_recv(void);

//#######################################################################
//# get hamming coded byte						#
//#######################################################################
unsigned char rfm12_hamm_recv(void);

//#######################################################################
//# init the RFM12 (default 869,4MHz)					#
//#######################################################################
void rfm12_init_433(unsigned char cs_port, unsigned char cs_pin);

//#######################################################################
//# init the RFM12 (default 869,4MHz)					#
//#######################################################################
void rfm12_init_868(unsigned char cs_port, unsigned char cs_pin);

//#######################################################################
//# send a packet							#
//#######################################################################
void rfm12_send(unsigned char *txBuffer,unsigned char len,unsigned char addr);

//#######################################################################
//# receive a packet							#
//#######################################################################
unsigned char rfm12_recv(unsigned char *rxBuffer,unsigned char len,unsigned char addr);

//#######################################################################
//# sleep								#
//#######################################################################
void rfm12_sleep(void);

//#######################################################################
//# wakeup								#
//#######################################################################
void rfm12_wakeup(void);

//######################################################################
//# cos_table table
//######################################################################
extern TAB_TYPE unsigned char cos_table[256];

//######################################################################
//# csin_table table
//######################################################################
extern TAB_TYPE unsigned char csin_table[256];

//######################################################################
//# charset 4x8
//######################################################################
extern TAB_TYPE unsigned char ctable_04_08[512];

//######################################################################
//# charset 6x10
//######################################################################
extern TAB_TYPE unsigned char ctable_06_10[1280];

//######################################################################
//# charset 8x12
//######################################################################
extern TAB_TYPE unsigned char ctable_08_12[1536];

extern TAB_TYPE unsigned char unilib_mask8l[8];

extern TAB_TYPE unsigned char unilib_imask8l[8];

extern TAB_TYPE unsigned short unilib_mask16l[16];

extern TAB_TYPE unsigned short unilib_imask16l[16];

extern TAB_TYPE unsigned char unilib_mask8m[8];

extern TAB_TYPE unsigned char unilib_imask8m[8];

extern TAB_TYPE unsigned short unilib_mask16m[16];

extern TAB_TYPE unsigned short unilib_imask16m[16];

extern TAB_TYPE unsigned char mirror_byte_table[256];

extern TAB_TYPE unsigned char ctable_segm32_60[5040];

//#######################################################################
//# convert date/time structure to unix stamp				#
//#######################################################################
unsigned char get_max_day(unsigned short year, unsigned char month);

//#######################################################################
//# convert unix timestamp to date/time structure			#
//#######################################################################
void unix2date(unsigned long stamp, unilib_date_time *tr);

//#######################################################################
//# convert date/time structure to unix stamp				#
//#######################################################################
unsigned long date2unix(unilib_date_time *tr);

//#######################################################################
//# ckeck presence							#
//#######################################################################
unsigned char pcf8583_check(unsigned char devaddr);

//#######################################################################
//# read bytes								#
//#######################################################################
unsigned char pcf8583_read(unsigned char *buffer,unsigned char addr, int len, unsigned char devaddr);

//#######################################################################
//# write bytes								#
//#######################################################################
unsigned char pcf8583_write(unsigned char *buffer,unsigned char addr, int len, unsigned char devaddr);

//#######################################################################
//# read seconds							#
//#######################################################################
int pcf8583_get_sec(unsigned char devaddr);

//#######################################################################
//# read minutes							#
//#######################################################################
int pcf8583_get_min(unsigned char devaddr);

//#######################################################################
//# read hour								#
//#######################################################################
int pcf8583_get_hour(unsigned char devaddr);

//#######################################################################
//# read day								#
//#######################################################################
int pcf8583_get_day(unsigned char devaddr);

//#######################################################################
//# read weekday							#
//#######################################################################
int pcf8583_get_wday(unsigned char devaddr);

//#######################################################################
//# read month								#
//#######################################################################
int pcf8583_get_month(unsigned char devaddr);

//#######################################################################
//# read month								#
//#######################################################################
int pcf8583_get_year(unsigned char devaddr);

//#######################################################################
//# set time								#
//#######################################################################
int pcf8583_set_sec(unsigned char devaddr, unsigned char val);

//#######################################################################
//# set minutes								#
//#######################################################################
int pcf8583_set_min(unsigned char devaddr, unsigned char val);

//#######################################################################
//# set hours								#
//#######################################################################
int pcf8583_set_hour(unsigned char devaddr, unsigned char val);

//#######################################################################
//# set day								#
//#######################################################################
int pcf8583_set_day(unsigned char devaddr, unsigned char val);

//#######################################################################
//# set weekday								#
//#######################################################################
int pcf8583_set_wday(unsigned char devaddr, unsigned char val);

//#######################################################################
//# set month								#
//#######################################################################
int pcf8583_set_month(unsigned char devaddr, unsigned char val);

//#######################################################################
//# set year								#
//#######################################################################
int pcf8583_set_year(unsigned char devaddr, unsigned char val);

//#######################################################################
//# init the RTC interface						#
//#######################################################################
void xrtc_init(PORT_TYPE cport,unsigned char cpin,PORT_TYPE dport,unsigned char dpin);

//#######################################################################
//# get current date/time						#
//# param: regnum (0-15)						#
//# result: 32 bit data							#
//#######################################################################
unsigned long xrtc_read_register(unsigned char reg);

//#######################################################################
//# write register value						#
//# param: regnum (0-15), 32 bit data					#
//#######################################################################
void xrtc_write_register(unsigned char reg,unsigned long value);

//#######################################################################
//# get current date/time						#
//# result: 32 bit timestamp						#
//#######################################################################
unsigned long xrtc_get_date(void);

//#######################################################################
//# set current date/time						#
//# param: 32 bit timestamp						#
//#######################################################################
void xrtc_set_date(unsigned long date);

//#######################################################################
//# send a command							#
//#######################################################################
unsigned char sdcard_cmd(unsigned char cmd, unsigned long addr, unsigned char crc);

//#######################################################################
//# init the sdcard							#
//# this must be done with low speed					#
//# 0 = no card / failure						#
//# 
//#######################################################################
unsigned char sdcard_init(PORT_TYPE cs_port, unsigned char cs_pin);

//#######################################################################
//# read status								#
//# 0=busy, 1=ready							#
//#######################################################################
unsigned char sdcard_read_status(void);

//#######################################################################
//# read data block							#
//#######################################################################
unsigned char sdcard_read_sector(unsigned char *buffer,unsigned long sector);

//#######################################################################
//# read data block word wise						#
//#######################################################################
unsigned char sdcard_read_sector_ww(unsigned short *buffer,unsigned long sector);

//#######################################################################
//# read data block longword wise					#
//#######################################################################
unsigned char sdcard_read_sector_lw(unsigned long *buffer,unsigned long sector);

//#######################################################################
//# write data block							#
//#######################################################################
unsigned char sdcard_write_sector(unsigned char *buffer,unsigned long sector);

//#######################################################################
//# write data block word wise						#
//#######################################################################
unsigned char sdcard_write_sector_ww(unsigned short *buffer,unsigned long sector);

//#######################################################################
//# write data block word wise						#
//#######################################################################
unsigned char sdcard_write_sector_lw(unsigned long *buffer,unsigned long sector);

//#######################################################################
//# check device ID							#
//#######################################################################
unsigned char m25p16_checkid(void);

//#######################################################################
//# init the m25p16							#
//#######################################################################
unsigned char m25p16_init(PORT_TYPE cs_port, unsigned char cs_pin);

//#######################################################################
//# read ststus byte							#
//#######################################################################
unsigned char m25p16_read_status(void);

//#######################################################################
//# wait for device is ready						#
//#######################################################################
unsigned char m25p16_wait_ready(unsigned long timeout);

//#######################################################################
//# checx for ready (0=ready  1=busy)					#
//#######################################################################
unsigned char m25p16_check_ready(void);

//#######################################################################
//# read data block							#
//#######################################################################
unsigned char m25p16_read_block(unsigned char *buffer,unsigned long addr, unsigned long len);

//#######################################################################
//# write data block							#
//#######################################################################
unsigned char m25p16_write_page(unsigned char *buffer,unsigned long addr,unsigned char blocked);

//#######################################################################
//# erase 64K sector							#
//#######################################################################
unsigned char m25p16_erase_sector(unsigned long addr, unsigned char blocked);

//#######################################################################
//# bulk erase								#
//#######################################################################
unsigned char m25p16_erase_bulk(unsigned char blocked);

//#######################################################################
//# sleep								#
//#######################################################################
unsigned char m25p16_sleep(void);

//#######################################################################
//# wakeup								#
//#######################################################################
unsigned char m25p16_wakeup(void);

//#######################################################################
//# check device ID							#
//#######################################################################
unsigned char m25p32_checkid(void);

//#######################################################################
//# init the m25p32							#
//#######################################################################
unsigned char m25p32_init(PORT_TYPE cs_port, unsigned char cs_pin);

//#######################################################################
//# read ststus byte							#
//#######################################################################
unsigned char m25p32_read_status(void);

//#######################################################################
//# wait for device is ready						#
//#######################################################################
unsigned char m25p32_wait_ready(unsigned long timeout);

//#######################################################################
//# checx for ready (0=ready  1=busy)					#
//#######################################################################
unsigned char m25p32_check_ready(void);

//#######################################################################
//# read data block							#
//#######################################################################
unsigned char m25p32_read_block(unsigned char *buffer,unsigned long addr, unsigned long len);

//#######################################################################
//# write data block							#
//#######################################################################
unsigned char m25p32_write_page(unsigned char *buffer,unsigned long addr,unsigned char blocked);

//#######################################################################
//# erase 64K sector							#
//#######################################################################
unsigned char m25p32_erase_sector(unsigned long addr, unsigned char blocked);

//#######################################################################
//# bulk erase								#
//#######################################################################
unsigned char m25p32_erase_bulk(unsigned char blocked);

//#######################################################################
//# sleep								#
//#######################################################################
unsigned char m25p32_sleep(void);

//#######################################################################
//# wakeup								#
//#######################################################################
unsigned char m25p32_wakeup(void);

//#######################################################################
//# check device ID							#
//#######################################################################
unsigned char m25p64_checkid(void);

//#######################################################################
//# init the m25p64							#
//#######################################################################
unsigned char m25p64_init(PORT_TYPE cs_port, unsigned char cs_pin);

//#######################################################################
//# read ststus byte							#
//#######################################################################
unsigned char m25p64_read_status(void);

//#######################################################################
//# wait for device is ready						#
//#######################################################################
unsigned char m25p64_wait_ready(unsigned long timeout);

//#######################################################################
//# checx for ready (0=ready  1=busy)					#
//#######################################################################
unsigned char m25p64_check_ready(void);

//#######################################################################
//# read data block							#
//#######################################################################
unsigned char m25p64_read_block(unsigned char *buffer,unsigned long addr, unsigned long len);

//#######################################################################
//# write data block							#
//#######################################################################
unsigned char m25p64_write_page(unsigned char *buffer,unsigned long addr,unsigned char blocked);

//#######################################################################
//# erase 64K sector							#
//#######################################################################
unsigned char m25p64_erase_sector(unsigned long addr, unsigned char blocked);

//#######################################################################
//# bulk erase								#
//#######################################################################
unsigned char m25p64_erase_bulk(unsigned char blocked);

//#######################################################################
//# sleep								#
//#######################################################################
unsigned char m25p64_sleep(void);

//#######################################################################
//# wakeup								#
//#######################################################################
unsigned char m25p64_wakeup(void);

//#######################################################################
//# check device ID							#
//#######################################################################
unsigned char m25p80_checkid(void);

//#######################################################################
//# init the m25p80							#
//#######################################################################
unsigned char m25p80_init(PORT_TYPE cs_port, unsigned char cs_pin);

//#######################################################################
//# read ststus byte							#
//#######################################################################
unsigned char m25p80_read_status(void);

//#######################################################################
//# wait for device is ready						#
//#######################################################################
unsigned char m25p80_wait_ready(unsigned long timeout);

//#######################################################################
//# checx for ready (0=ready  1=busy)					#
//#######################################################################
unsigned char m25p80_check_ready(void);

//#######################################################################
//# read data block							#
//#######################################################################
unsigned char m25p80_read_block(unsigned char *buffer,unsigned long addr, unsigned long len);

//#######################################################################
//# write data block							#
//#######################################################################
unsigned char m25p80_write_page(unsigned char *buffer,unsigned long addr,unsigned char blocked);

//#######################################################################
//# erase 64K sector							#
//#######################################################################
unsigned char m25p80_erase_sector(unsigned long addr, unsigned char blocked);

//#######################################################################
//# bulk erase								#
//#######################################################################
unsigned char m25p80_erase_bulk(unsigned char blocked);

//#######################################################################
//# sleep								#
//#######################################################################
unsigned char m25p80_sleep(void);

//#######################################################################
//# wakeup								#
//#######################################################################
unsigned char m25p80_wakeup(void);

//#######################################################################
//# check device ID							#
//#######################################################################
unsigned char w25q128_checkid(void);

//#######################################################################
//# init the w25q128							#
//#######################################################################
unsigned char w25q128_init(PORT_TYPE cs_port, unsigned char cs_pin);

//#######################################################################
//# read ststus byte							#
//#######################################################################
unsigned char w25q128_read_status(void);

//#######################################################################
//# wait for device is ready						#
//#######################################################################
unsigned char w25q128_wait_ready(unsigned long timeout);

//#######################################################################
//# checx for ready (0=ready  1=busy)					#
//#######################################################################
unsigned char w25q128_check_ready(void);

//#######################################################################
//# read data block							#
//#######################################################################
unsigned char w25q128_read_block(unsigned char *buffer,unsigned long addr, unsigned long len);

//#######################################################################
//# write data block							#
//#######################################################################
unsigned char w25q128_write_page(unsigned char *buffer,unsigned long addr,unsigned char blocked);

//#######################################################################
//# erase 64K sector							#
//#######################################################################
unsigned char w25q128_erase_sector(unsigned long addr, unsigned char blocked);

//#######################################################################
//# erase 4K sector							#
//#######################################################################
unsigned char w25q128_erase_sector4k(unsigned long addr, unsigned char blocked);

//#######################################################################
//# bulk erase								#
//#######################################################################
unsigned char w25q128_erase_bulk(unsigned char blocked);

//#######################################################################
//# sleep								#
//#######################################################################
unsigned char w25q128_sleep(void);

//#######################################################################
//# wakeup								#
//#######################################################################
unsigned char w25q128_wakeup(void);

//#######################################################################
//# init the xsram							#
//#######################################################################
void xsram_init(PORT_TYPE cs_port, unsigned char cs_pin);

//#######################################################################
//# read data block							#
//#######################################################################
void xsram_read_block(unsigned char *buffer,unsigned long addr, unsigned long len);

//#######################################################################
//# write data block							#
//#######################################################################
void xsram_write_block(unsigned char *buffer,unsigned long addr, unsigned long len);

//#######################################################################
//#									#
//# integer absolute value (magnitude)					#
//#									#
//#######################################################################
int abs (int i);

//#######################################################################
//#									#
//# ascii to long							#
//#									#
//#######################################################################
float atof(const char *s);

//#######################################################################
//#									#
//# ascii to integer							#
//#									#
//#######################################################################
int atoi(const char *s);

//#######################################################################
//#									#
//# ascii to long							#
//#									#
//#######################################################################
long atol(const char *s);

//#######################################################################
//#									#
//# divide two integers							#
//#									#
//#######################################################################
//#######################################################################
//#									#
//# search for character in string					#
//#									#
//#######################################################################
char * index(char *s1, int i);

//#######################################################################
//#									#
//# check if char is a alphabetic letter				#
//#									#
//#######################################################################
int isalpha(char c);

//#######################################################################
//#									#
//# check if char is a digit						#
//#									#
//#######################################################################
int isdigit(char c);

//#######################################################################
//#									#
//# check if char is lowercase						#
//#									#
//#######################################################################
int islower(char c);

//#######################################################################
//#									#
//# check if char is a whitespace					#
//#									#
//#######################################################################
int isspace(char c);

//#######################################################################
//#									#
//# check if char is uppercase						#
//#									#
//#######################################################################
int isupper(char c);

//#######################################################################
//#									#
//# integer to ascii							#
//#									#
//#######################################################################
char * itoa(int value, char *ptr, int base);

//#######################################################################
//#									#
//# divide two long integers						#
//#									#
//#######################################################################
//#######################################################################
//#									#
//# copy memory regions with end-token check				#
//#									#
//#######################################################################
void * memccpy(void * dst0, const void * src0, int endchar0, size_t num);

//#######################################################################
//#									#
//# find character in memory						#
//#									#
//#######################################################################
void * memchr(const void *ptr,int value,size_t num);

//#######################################################################
//#									#
//# compare two memory areas						#
//#									#
//#######################################################################
int memcmp(void *ptr1,void *ptr2,size_t num);

//#######################################################################
//#									#
//# copy memory regions							#
//#									#
//#######################################################################
void * memcpy(void * dst0,const void * src0,size_t num);

//#######################################################################
//#									#
//# compare two strings							#
//#									#
//#######################################################################
void * memmove(void * dst0,const void * src0, size_t num);

//#######################################################################
//#									#
//# set an area of memory						#
//#									#
//#######################################################################
void * memset(void * ptr,int value,size_t num);

//#######################################################################
//#									#
//# reverse search for character in string				#
//#									#
//#######################################################################
char * rindex(char *s1, int i);

//#######################################################################
//#									#
//# copy string	and return a pointer to its end				#
//#									#
//#######################################################################
char* stpcpy(char *dst0, char *src0);

//#######################################################################
//#									#
//# copy string	and return a pointer to its end				#
//#									#
//#######################################################################
char* stpncpy(char *dst0, char *src0,size_t num);

//#######################################################################
//#									#
//# compare two strings and ignore case					#
//#									#
//#######################################################################
int strcasecmp(char *s1, char *s2);

//#######################################################################
//#									#
//# find string segment and ignore case					#
//#									#
//#######################################################################
char * strcasestr(char *haystack, char *needle);

//#######################################################################
//#									#
//# concatenate two strings						#
//#									#
//#######################################################################
char * strcat(char *dst, const char *src);

//#######################################################################
//#									#
//# search for character in string					#
//#									#
//#######################################################################
char * strchr(const char *s1, int i);

//#######################################################################
//#									#
//# compare two strings							#
//#									#
//#######################################################################
int strcmp(const char *s1, const char *s2);

//#######################################################################
//#									#
//# copy string								#
//#									#
//#######################################################################
char* strcpy(char *dst0, const char *src0);

//#######################################################################
//#									#
//# count characters not in string					#
//#									#
//#######################################################################
size_t strcspn(const char *s1,const char *s2);

//#######################################################################
//#									#
//# determine lenght of a string					#
//#									#
//#######################################################################
size_t strlen (const char *str);

//#######################################################################
//#									#
//# force string to lowercase						#
//#									#
//#######################################################################
char * strlwr (char * s);

//#######################################################################
//#									#
//# compare two strings and ignore case					#
//#									#
//#######################################################################
int strncasecmp(char *s1, char *s2, size_t num);

//#######################################################################
//#									#
//# concatenate two strings						#
//#									#
//#######################################################################
char * strncat(char *s1, const char *s2, size_t num);

//#######################################################################
//#									#
//# compare two strings							#
//#									#
//#######################################################################
int strncmp(const char *s1, const char *s2, size_t num);

//#######################################################################
//#									#
//# copy string								#
//#									#
//#######################################################################
char* strncpy(char *dst0, const char *src0, size_t num);

//#######################################################################
//#									#
//# determine lenght of a string					#
//#									#
//#######################################################################
size_t strnlen (const char *str, size_t num);

//#######################################################################
//#									#
//# find characters in string						#
//#									#
//#######################################################################
char * strpbrk(const char *s1, const char *s2);

//#######################################################################
//#									#
//# reverse search for character in string				#
//#									#
//#######################################################################
char * strrchr(const char *s1, int i);

//#######################################################################
//#									#
//# find initial match							#
//#									#
//#######################################################################
size_t strspn(const char *s1, const char *s2);

//#######################################################################
//#									#
//# find string segment							#
//#									#
//#######################################################################
char * strstr(const char *haystack, const char *needle);

//#######################################################################
//#									#
//# Convert a string to a float						#
//#									#
//#######################################################################
float strtof (const char *ptr, char** endptr);

//#######################################################################
//#									#
//# split string into tokens						#
//#									#
//#######################################################################
char * strtok(char *str, const char *delimiters);

//#######################################################################
//#									#
//# Convert a string to a long integer.					#
//#									#
//#######################################################################
long strtol (const char *ptr, char** endptr);

//#######################################################################
//#									#
//# force string to uppercase						#
//#									#
//#######################################################################
char * strupr (char * s);

//#######################################################################
//#									#
//# swap adjacent bytes							#
//#									#
//#######################################################################
void swab(void *src0, void *dst0, size_t num);

//#######################################################################
//#									#
//# convert char to lowercase						#
//#									#
//#######################################################################
char tolower(char c);

//#######################################################################
//#									#
//# convert char to uppercase						#
//#									#
//#######################################################################
char toupper(char c);

//#######################################################################
//#									#
//# unsigned integer to string						#
//#									#
//#######################################################################
char * utoa(unsigned int value, char * str, int base);

//#######################################################################
//#									#
//#	enable stepper (unipolar mode)					#
//#									#
//#######################################################################
void enable_stepper(int stepper,PORT_TYPE port,int pina,int pinb,int pinc,int pind);

//#######################################################################
//#									#
//#	reset stepper							#
//#									#
//#######################################################################
void reset_stepper(int stepper);

//#######################################################################
//#									#
//#	move stepper							#
//#									#
//#######################################################################
void move_stepper(int stepper,int stepper_steps,int stepper_scale);

//#######################################################################
//#									#
//#	wait for move end						#
//#									#
//#######################################################################
void wait_stepper(void);

//#######################################################################
//#									#
//#	check stepper (1=move0, 2=move1)				#
//#									#
//#######################################################################
int check_stepper(void);

//#######################################################################
//#									#
//#	this must be called periodicly (in interrupt)			#
//#									#
//#######################################################################
void serve_stepper(void);

//#######################################################################
//#									#
//#	wait for screeen sync						#
//#									#
//#######################################################################
void tlcd_waitsync(void);

//#######################################################################
//#									#
//#	clear screen							#
//#									#
//#######################################################################
void tlcd_clrscr(void);

//#######################################################################
//#									#
//#	set screen position						#
//#									#
//#######################################################################
void tlcd_gotoxy(unsigned char x,unsigned char y);

//#######################################################################
//#									#
//#	new line							#
//#									#
//#######################################################################
void tlcd_newline(void);

//#######################################################################
//#									#
//#	set wrapmode							#
//#									#
//#######################################################################
void tlcd_wrapmode(void);

//#######################################################################
//#									#
//#	set scrollmode							#
//#									#
//#######################################################################
void tlcd_scrollmode(void);

//#######################################################################
//#									#
//#	get maximal x coordinate					#
//#									#
//#######################################################################
unsigned int tlcd_get_textsize_x(void);

//#######################################################################
//#									#
//#	get maximal y coordinate					#
//#									#
//#######################################################################
unsigned int tlcd_get_textsize_y(void);

//#######################################################################
//#									#
//#	get current x coordinate					#
//#									#
//#######################################################################
unsigned int tlcd_get_textpos_x(void);

//#######################################################################
//#									#
//#	get current y coordinate					#
//#									#
//#######################################################################
unsigned int tlcd_get_textpos_y(void);

//#######################################################################
//#									#
//#	write char							#
//#									#
//#######################################################################
void tlcd_char(unsigned char data);

//#######################################################################
//#									#
//#	write char (raw mode)						#
//#									#
//#######################################################################
void tlcd_user_char(unsigned char num,unsigned char* pixline);

//#######################################################################
//#									#
//#	init LCD							#
//#									#
//#######################################################################
void init_tlcd161_n0(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char rspin,unsigned char epin)	;

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_tlcd161_n0(void);

//#######################################################################
//#									#
//#	init LCD							#
//#									#
//#######################################################################
void init_tlcd161_n1(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char rspin,unsigned char epin)	;

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_tlcd161_n1(void);

//#######################################################################
//#									#
//#	init LCD							#
//#									#
//#######################################################################
void init_tlcd161_n2(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char rspin,unsigned char epin)	;

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_tlcd161_n2(void);

//#######################################################################
//#									#
//#	init LCD							#
//#									#
//#######################################################################
void init_tlcd161_n3(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char rspin,unsigned char epin)	;

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_tlcd161_n3(void);

//#######################################################################
//#									#
//#	init LCD							#
//#									#
//#######################################################################
void init_tlcd161b_n0(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char rspin,unsigned char epin)	;

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_tlcd161b_n0(void);

//#######################################################################
//#									#
//#	init LCD							#
//#									#
//#######################################################################
void init_tlcd161b_n1(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char rspin,unsigned char epin)	;

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_tlcd161b_n1(void);

//#######################################################################
//#									#
//#	init LCD							#
//#									#
//#######################################################################
void init_tlcd161b_n2(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char rspin,unsigned char epin)	;

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_tlcd161b_n2(void);

//#######################################################################
//#									#
//#	init LCD							#
//#									#
//#######################################################################
void init_tlcd161b_n3(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char rspin,unsigned char epin)	;

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_tlcd161b_n3(void);

//#######################################################################
//#									#
//#	init LCD							#
//#									#
//#######################################################################
void init_tlcd162_n0(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char rspin,unsigned char epin)	;

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_tlcd162_n0(void);

//#######################################################################
//#									#
//#	init LCD							#
//#									#
//#######################################################################
void init_tlcd162_n1(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char rspin,unsigned char epin)	;

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_tlcd162_n1(void);

//#######################################################################
//#									#
//#	init LCD							#
//#									#
//#######################################################################
void init_tlcd162_n2(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char rspin,unsigned char epin)	;

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_tlcd162_n2(void);

//#######################################################################
//#									#
//#	init LCD							#
//#									#
//#######################################################################
void init_tlcd162_n3(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char rspin,unsigned char epin)	;

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_tlcd162_n3(void);

//#######################################################################
//#									#
//#	init LCD							#
//#									#
//#######################################################################
void init_tlcd202_n0(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char rspin,unsigned char epin)	;

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_tlcd202_n0(void);

//#######################################################################
//#									#
//#	init LCD							#
//#									#
//#######################################################################
void init_tlcd202_n1(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char rspin,unsigned char epin)	;

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_tlcd202_n1(void);

//#######################################################################
//#									#
//#	init LCD							#
//#									#
//#######################################################################
void init_tlcd202_n2(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char rspin,unsigned char epin)	;

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_tlcd202_n2(void);

//#######################################################################
//#									#
//#	init LCD							#
//#									#
//#######################################################################
void init_tlcd202_n3(PORT_TYPE dataport,PORT_TYPE ctrlport,unsigned char rspin,unsigned char epin)	;

//#######################################################################
//#									#
//#	service								#
//#									#
//#######################################################################
void serve_tlcd202_n3(void);

extern unsigned char *memptr;

//#######################################################################
//#									#
//#	set printf channel						#
//#									#
//#######################################################################
void set_outchar_routine(void (*outfunc)(unsigned char));

//#######################################################################
//#									#
//#	dummy output							#
//#									#
//#######################################################################
void OUTPUT_NONE(unsigned char c);

//#######################################################################
//#									#
//#	memory output							#
//#									#
//#######################################################################
void OUTPUT_MEMORY(unsigned char c);

//#######################################################################
//#									#
//#	memory output							#
//#									#
//#######################################################################
void set_outchar_memory(unsigned char *c);

//#######################################################################
//#									#
//#	get printf channel						#
//#									#
//#######################################################################
void uprintf (const char *format, ...);

//#######################################################################
//#									#
//#	get printf channel						#
//#									#
//#######################################################################
void uxprintf (const char *format, ...);

//#######################################################################
//#									#
//# rotate 8x8 buffer for glcd						#
//#									#
//#######################################################################
void unilib_rotate_buffer8(unsigned char * src,unsigned char *dst);

//#######################################################################
//#									#
//# rotate 8x8 buffer for glcd						#
//#									#
//#######################################################################
void unilib_rotate_buffer8a(unsigned char * src,unsigned char *dst);

extern int mylib_uart_stat;

extern volatile unsigned char unilib_cpu_freq;

//#######################################################################
//#									#
//#	setup and enable caches						#
//#									#
//#######################################################################
void unilib_init(void);

//#######################################################################
//#									#
//#	set CLOCK / PLL							#
//#									#
//#######################################################################
void set_clock(unsigned long mode);

//#######################################################################
//#									#
//#	trim int osc							#
//#									#
//#######################################################################
void trim_clock(int val);

//#######################################################################
//#									#
//#	wait n cycles							#
//#									#
//#######################################################################
void unilib_pause(long cycles);

//#######################################################################
//#									#
//#	wait n ms							#
//#									#
//#######################################################################
void unilib_pause_ms(int ms);

//#######################################################################
//#									#
//#									#
//#									#
//#######################################################################
void __attribute__((naked)) enable_interrupts(void);

//#######################################################################
//#									#
//#									#
//#									#
//#######################################################################
void __attribute__((naked)) disable_interrupts(void);

//#######################################################################
//#									#
//#	get flash byte							#
//#									#
//#######################################################################
unsigned char unilib_getrombyte(unsigned char *address);

//#######################################################################
//#									#
//#	get flash word							#
//#									#
//#######################################################################
unsigned short unilib_getromword(unsigned short *address);

extern volatile unsigned char unilib_adc_mask;

extern volatile unsigned char unilib_adc_mode;

extern volatile unsigned char unilib_adc_chan;

//#######################################################################
//#									#
//#	init ADC (max 8 channels)					#
//#									#
//#######################################################################
void init_adc(unsigned char mask);

//#######################################################################
//#									#
//#	init ADC (max 8 channels)					#
//#									#
//#######################################################################
void init_adc_cont(unsigned char mask);

//#######################################################################
//#									#
//#	disable ADC							#
//#									#
//#######################################################################
void disable_adc(void);

//#######################################################################
//#									#
//#	get adc value							#
//#									#
//#######################################################################
unsigned int get_adc(unsigned char channel);

//#######################################################################
//#									#
//#	start adc conversion						#
//#									#
//#######################################################################
void start_adc_channel(unsigned char channel);

//#######################################################################
//#									#
//#	check conversion status (0=ready  1=busy)			#
//#									#
//#######################################################################
unsigned char check_adc(void);

//#######################################################################
//#									#
//#	get conversion result 						#
//#									#
//#######################################################################
unsigned int get_adc_value(void);

//#######################################################################
//#									#
//#	init SPI as master						#
//#									#
//#######################################################################
void enable_ext_int0(unsigned char mode,unsigned char pin);

//#######################################################################
//#									#
//#	init SPI as slave						#
//#									#
//#######################################################################
void disable_ext_int0(void);

//#######################################################################
//#									#
//#	init SPI as master						#
//#									#
//#######################################################################
void enable_ext_int0(unsigned char mode,unsigned char pin);

//#######################################################################
//#									#
//#	init SPI as slave						#
//#									#
//#######################################################################
void disable_ext_int0(void);

extern volatile unsigned char i2c1_state;

//#######################################################################
//#									#
//#	init I2C (0=slow, 1=fast)					#
//#									#
//#######################################################################
void enable_i2c_master(unsigned char mode);

//#######################################################################
//#									#
//#	disable I2C							#
//#									#
//#######################################################################
void disable_i2c(void);

//#######################################################################
//#									#
//#	send stop condition						#
//#									#
//#######################################################################
void i2c_stop(void);

//#######################################################################
//#									#
//#	send start condition						#
//#									#
//#######################################################################
unsigned char i2c_start(void);

//#######################################################################
//#									#
//#	send repeated start condition					#
//#									#
//#######################################################################
unsigned char i2c_rstart(void);

//#######################################################################
//#									#
//#	send address for read						#
//#									#
//#######################################################################
unsigned char i2c_master_addr(unsigned char address);

//#######################################################################
//#									#
//#	send data							#
//#									#
//#######################################################################
unsigned char i2c_master_send(unsigned char data);

//#######################################################################
//#									#
//#	read data							#
//#									#
//#######################################################################
unsigned char i2c_master_recv(unsigned char ack);

//#######################################################################
//#									#
//#	set port pin to alternate function				#
//#									#
//# mode (16-19)	AFR	(alt function 0..7)			#
//# mode (12,13)	PUPDR	(0=nopull  1=pullup  2=pulldown)	#
//# mode (8)		OTYPE	(0=pushpull  1=opendrain)		#
//# mode (4,5)		OSPEED	(0=low  1=medium  3=high)		#
//# mode (0,1)		MODER	(0=in  1=out  2=alt  3=analog)		#
//#									#
//#######################################################################
void set_portpin_alt(unsigned long port,int pin, unsigned long mode);

//#######################################################################
//#									#
//#	set port pin to output						#
//#									#
//#######################################################################
void set_portpin_output(unsigned long port,int pin);

//#######################################################################
//#									#
//#	set port pin to input without pullup				#
//#									#
//#######################################################################
void set_portpin_input_nopull(unsigned long port,int pin);

//#######################################################################
//#									#
//#	set port pin to input with pullup				#
//#									#
//#######################################################################
void set_portpin_input_pullup(unsigned long port,int pin);

//#######################################################################
//#									#
//#	set port pin to ADC input					#
//#									#
//#######################################################################
void set_portpin_analog(unsigned long port,int pin);

//#######################################################################
//#									#
//#	set port to output						#
//#									#
//#######################################################################
void set_port_output(unsigned long port);

//#######################################################################
//#									#
//#	set port to input without pullup				#
//#									#
//#######################################################################
void set_port_input_nopull(unsigned long port);

//#######################################################################
//#									#
//#	set port to input with pullup					#
//#									#
//#######################################################################
void set_port_input_pullup(unsigned long port);

//#######################################################################
//#									#
//#	set port pin to one						#
//#									#
//#######################################################################
void set_portpin_one(unsigned long port,int pin);

//#######################################################################
//#									#
//#	set port pin to zero						#
//#									#
//#######################################################################
void set_portpin_zero(unsigned long port,int pin);

//#######################################################################
//#									#
//#	set port pin status						#
//#									#
//#######################################################################
void set_portpin_level(unsigned long port,int pin,int val);

//#######################################################################
//#									#
//#	get port pin value						#
//#									#
//#######################################################################
int get_portpin_level(unsigned long port,int pin);

//#######################################################################
//#									#
//#	get port value							#
//#									#
//#######################################################################
int get_port_level(unsigned long port);

//#######################################################################
//#									#
//#	set port value							#
//#									#
//#######################################################################
void set_port_level(unsigned long port,unsigned short val);

//#######################################################################
//#									#
//#	set port nibble value						#
//#									#
//#######################################################################
void set_port_n0_level(unsigned long port,unsigned short val);

//#######################################################################
//#									#
//#	set port nibble value						#
//#									#
//#######################################################################
void set_port_n1_level(unsigned long port,unsigned short val);

//#######################################################################
//#									#
//#	set port nibble value						#
//#									#
//#######################################################################
void set_port_n2_level(unsigned long port,unsigned short val);

//#######################################################################
//#									#
//#	set port nibble value						#
//#									#
//#######################################################################
void set_port_n3_level(unsigned long port,unsigned short val);

//#######################################################################
//#									#
//#	set port pins							#
//#									#
//#######################################################################
void set_port_one(unsigned long port,unsigned short mask);

//#######################################################################
//#									#
//#	clear port pins							#
//#									#
//#######################################################################
void set_port_zero(unsigned long port,unsigned short mask);

//#######################################################################
//#									#
//#	scale PWM0-PWM3							#
//#									#
//#######################################################################
void scale_pwm03(unsigned int scale);

//#######################################################################
//#									#
//#	enable PWM0							#
//#									#
//#######################################################################
void enable_pwm0(unsigned int scale, unsigned int value);

//#######################################################################
//#									#
//#	enable PWM1							#
//#									#
//#######################################################################
void enable_pwm1(unsigned int scale, unsigned int value);

//#######################################################################
//#									#
//#	enable PWM2							#
//#									#
//#######################################################################
void enable_pwm2(unsigned int scale, unsigned int value);

//#######################################################################
//#									#
//#	enable PWM2							#
//#									#
//#######################################################################
void enable_pwm3(unsigned int scale, unsigned int value);

//#######################################################################
//#									#
//#	set PWM0 value							#
//#									#
//#######################################################################
void set_pwm0(unsigned int value);

//#######################################################################
//#									#
//#	set PWM1 value							#
//#									#
//#######################################################################
void set_pwm1(unsigned int value);

//#######################################################################
//#									#
//#	set PWM2 value							#
//#									#
//#######################################################################
void set_pwm2(unsigned int value);

//#######################################################################
//#									#
//#	set PWM2 value							#
//#									#
//#######################################################################
void set_pwm3(unsigned int value);

//#######################################################################
//#									#
//#	disable PWM0							#
//#									#
//#######################################################################
void disable_pwm0(void);

//#######################################################################
//#									#
//#	disable PWM1							#
//#									#
//#######################################################################
void disable_pwm1(void);

//#######################################################################
//#									#
//#	disable PWM2							#
//#									#
//#######################################################################
void disable_pwm2(void);

//#######################################################################
//#									#
//#	disable PWM3							#
//#									#
//#######################################################################
void disable_pwm3(void);

//#######################################################################
//#									#
//#	scale PWM4-PWM7							#
//#									#
//#######################################################################
void scale_pwm47(unsigned int scale);

//#######################################################################
//#									#
//#	enable PWM4							#
//#									#
//#######################################################################
void enable_pwm4(unsigned int scale, unsigned int value);

//#######################################################################
//#									#
//#	enable PWM5							#
//#									#
//#######################################################################
void enable_pwm5(unsigned int scale, unsigned int value);

//#######################################################################
//#									#
//#	enable PWM6							#
//#									#
//#######################################################################
void enable_pwm6(unsigned int scale, unsigned int value);

//#######################################################################
//#									#
//#	enable PWM7							#
//#									#
//#######################################################################
void enable_pwm7(unsigned int scale, unsigned int value);

//#######################################################################
//#									#
//#	set PWM4 value							#
//#									#
//#######################################################################
void set_pwm4(unsigned int value);

//#######################################################################
//#									#
//#	set PWM5 value							#
//#									#
//#######################################################################
void set_pwm5(unsigned int value);

//#######################################################################
//#									#
//#	set PWM6 value							#
//#									#
//#######################################################################
void set_pwm6(unsigned int value);

//#######################################################################
//#									#
//#	set PWM7 value							#
//#									#
//#######################################################################
void set_pwm7(unsigned int value);

//#######################################################################
//#									#
//#	disable PWM4							#
//#									#
//#######################################################################
void disable_pwm4(void);

//#######################################################################
//#									#
//#	disable PWM5							#
//#									#
//#######################################################################
void disable_pwm5(void);

//#######################################################################
//#									#
//#	disable PWM6							#
//#									#
//#######################################################################
void disable_pwm6(void);

//#######################################################################
//#									#
//#	disable PWM7							#
//#									#
//#######################################################################
void disable_pwm7(void);

extern volatile unsigned long spi1_sso_port;

extern volatile unsigned char spi1_sso_pin;

//#######################################################################
//#									#
//#	init SPI as master						#
//#									#
//#######################################################################
void enable_spi1_master(unsigned char divider,unsigned char mode,unsigned long sso_port,unsigned char sso_pin);

//#######################################################################
//#									#
//#	init SPI as slave						#
//#									#
//#######################################################################
void enable_spi1_slave(unsigned char mode);

//#######################################################################
//#									#
//#	disable SPI master						#
//#									#
//#######################################################################
void disable_spi1_master(void);

//#######################################################################
//#									#
//#	disable SPI slave						#
//#									#
//#######################################################################
void disable_spi1_slave(void);

//#######################################################################
//#									#
//#	send/recv byte							#
//#									#
//#######################################################################
unsigned char comm_spi1(unsigned char data);

//#######################################################################
//#									#
//#	send byte							#
//#									#
//#######################################################################
void send_spi1(unsigned char data);

//#######################################################################
//#									#
//#	send byte (fast mode)						#
//#									#
//#######################################################################
void sendfast_spi1(unsigned char data);

//#######################################################################
//#									#
//#	select (SSO-pin=low)						#
//#									#
//#######################################################################
void select_spi1(void);

//#######################################################################
//#									#
//#	de-select (SSO-pin=high)					#
//#									#
//#######################################################################
void unselect_spi1(void);

extern volatile unsigned long spi2_sso_port;

extern volatile unsigned char spi2_sso_pin;

//#######################################################################
//#									#
//#	init SPI as master						#
//#									#
//#######################################################################
void enable_spi2_master(unsigned char divider,unsigned char mode,unsigned long sso_port,unsigned char sso_pin);

//#######################################################################
//#									#
//#	init SPI as slave						#
//#									#
//#######################################################################
void enable_spi2_slave(unsigned char mode);

//#######################################################################
//#									#
//#	disable SPI master						#
//#									#
//#######################################################################
void disable_spi2_master(void);

//#######################################################################
//#									#
//#	disable SPI slave						#
//#									#
//#######################################################################
void disable_spi2_slave(void);

//#######################################################################
//#									#
//#	send/recv byte							#
//#									#
//#######################################################################
unsigned char comm_spi2(unsigned char data);

//#######################################################################
//#									#
//#	send byte							#
//#									#
//#######################################################################
void send_spi2(unsigned char data);

//#######################################################################
//#									#
//#	send byte (fast mode)						#
//#									#
//#######################################################################
void sendfast_spi2(unsigned char data);

//#######################################################################
//#									#
//#	select (SSO-pin=low)						#
//#									#
//#######################################################################
void select_spi2(void);

//#######################################################################
//#									#
//#	de-select (SSO-pin=high)					#
//#									#
//#######################################################################
void unselect_spi2(void);

extern volatile unsigned int unilib_tick_counter;

//#######################################################################
//#									#
//#	wait a number of ticks						#
//#									#
//#######################################################################
void wait_ticks(unsigned int ticks);

//#######################################################################
//#									#
//#	set tick counter						#
//#									#
//#######################################################################
void set_ticks(unsigned int ticks);

//#######################################################################
//#									#
//#	get tick counter						#
//#									#
//#######################################################################
unsigned int get_ticks(void);

//#######################################################################
//#									#
//#	init tick timer							#
//#									#
//#######################################################################
void enable_tick(unsigned long scale);

//#######################################################################
//#									#
//#	disable tick interrupt						#
//#									#
//#######################################################################
void disable_tick(void);

//#######################################################################
//#									#
//#	init uart at PA9/PA10						#
//#									#
//#######################################################################
void enable_uart1(unsigned int divider);

//#######################################################################
//#									#
//#	init uart at PB6/PB7						#
//#									#
//#######################################################################
void enable_uart1_alt(unsigned int divider);

//#######################################################################
//#									#
//#	disable UART							#
//#									#
//#######################################################################
void disable_uart1(void);

//#######################################################################
//#									#
//#	send serial byte						#
//#									#
//#######################################################################
void send_uart1(unsigned char data);

//#######################################################################
//#									#
//#	send serial byte						#
//#									#
//#######################################################################
void send_uart1_nonblock(unsigned char data);

//#######################################################################
//#									#
//#	read serial byte						#
//#									#
//#######################################################################
unsigned char get_uart1(void);

//#######################################################################
//#									#
//#	check uart receiver status (1=char)				#
//#									#
//#######################################################################
unsigned char check_uart1_rx(void);

//#######################################################################
//#									#
//#	check uart transmit status (1=empty)				#
//#									#
//#######################################################################
unsigned char check_uart1_tx(void);

//#######################################################################
//#									#
//#	enable tx interrupt						#
//#									#
//#######################################################################
void enable_uart1_transmit_interrupt(void);

//#######################################################################
//#									#
//#	disable tx interrupt						#
//#									#
//#######################################################################
void disable_uart1_transmit_interrupt(void);

//#######################################################################
//#									#
//#	enable rx interrupt						#
//#									#
//#######################################################################
void enable_uart1_receive_interrupt(void);

//#######################################################################
//#									#
//#	disable rx interrupt						#
//#									#
//#######################################################################
void disable_uart1_receive_interrupt(void);

//#######################################################################
//#									#
//#	init uart							#
//#									#
//#######################################################################
void enable_uart2(unsigned int divider);

//#######################################################################
//#									#
//#	disable UART							#
//#									#
//#######################################################################
void disable_uart2(void);

//#######################################################################
//#									#
//#	send serial byte						#
//#									#
//#######################################################################
void send_uart2(unsigned char data);

//#######################################################################
//#									#
//#	send serial byte						#
//#									#
//#######################################################################
void send_uart2_nonblock(unsigned char data);

//#######################################################################
//#									#
//#	read serial byte						#
//#									#
//#######################################################################
unsigned char get_uart2(void);

//#######################################################################
//#									#
//#	check uart receiver status (1=char)				#
//#									#
//#######################################################################
unsigned char check_uart2_rx(void);

//#######################################################################
//#									#
//#	check uart transmitter status (1=empty)				#
//#									#
//#######################################################################
unsigned char check_uart2_tx(void);

//#######################################################################
//#									#
//#	enable tx interrupt						#
//#									#
//#######################################################################
void enable_uart2_transmit_interrupt(void);

//#######################################################################
//#									#
//#	disable tx interrupt						#
//#									#
//#######################################################################
void disable_uart2_transmit_interrupt(void);

//#######################################################################
//#									#
//#	enable rx interrupt						#
//#									#
//#######################################################################
void enable_uart2_receive_interrupt(void);

//#######################################################################
//#									#
//#	disable rx interrupt						#
//#									#
//#######################################################################
void disable_uart2_receive_interrupt(void);

//#######################################################################
//#									#
//#	init uart							#
//#									#
//#######################################################################
void enable_uart3(unsigned int divider);

//#######################################################################
//#									#
//#	disable UART							#
//#									#
//#######################################################################
void disable_uart3(void);

//#######################################################################
//#									#
//#	send serial byte						#
//#									#
//#######################################################################
void send_uart3(unsigned char data);

//#######################################################################
//#									#
//#	send serial byte						#
//#									#
//#######################################################################
void send_uart3_nonblock(unsigned char data);

//#######################################################################
//#									#
//#	read serial byte						#
//#									#
//#######################################################################
unsigned char get_uart3(void);

//#######################################################################
//#									#
//#	check uart receiver status (1=char)				#
//#									#
//#######################################################################
unsigned char check_uart3_rx(void);

//#######################################################################
//#									#
//#	check uart transmitter status (1=empty)				#
//#									#
//#######################################################################
unsigned char check_uart3_tx(void);

//#######################################################################
//#									#
//#	enable tx interrupt						#
//#									#
//#######################################################################
void enable_uart3_transmit_interrupt(void);

//#######################################################################
//#									#
//#	disable tx interrupt						#
//#									#
//#######################################################################
void disable_uart3_transmit_interrupt(void);

//#######################################################################
//#									#
//#	enable rx interrupt						#
//#									#
//#######################################################################
void enable_uart3_receive_interrupt(void);

//#######################################################################
//#									#
//#	disable rx interrupt						#
//#									#
//#######################################################################
void disable_uart3_receive_interrupt(void);

